/* -*- Mode: C++ -*- */

/* The class(es) which represents one node of a plan */

#ifndef _PLAN_NODE_H_	
#define _PLAN_NODE_H_ 

#include <stream.h>
#include "BinaryString.h"
#include "Geometry.h"
#include "soccer_utils.h"

/* note that both component of all vectors need to be in the range [-127, 128] */
const int bits_for_vector_comp = 8;
const int min_val_for_vector_comp = -127;

enum plan_node_type_t {
  //PN_NoValue = -1,
  PN_InitialPos = 0,
  PN_StartPass,
  PN_EndPass,
  PN_StartGoto,
  PN_EndGoto,
  PN_SendBall, //no start/end pair because we don't rely on a sending going anywhere
  PN_StartDribble,
  PN_EndDribble,
  //Choose,Anyof are not currently supported
  PN_Choose,
  PN_AnyOf
};
/*if you get more than 16 types (not counting NoValue), 
  you'll have to change this */
const int bits_for_plan_node_type = 4;
const int num_plan_node_types = 10;

#define PLAN_NODE_STRINGS \
{"InitialPos", \
"StartPass", "EndPass", "StartGoto", "EndGoto", \
"SendBall", "StartDribble", "EndDribble",\
"Choose", "AnyOf" }
const int plan_node_string_len = 30;

enum player_involved_t {
  PI_Yes,
  PI_No,
  PI_CheckPrevious //For end types, we have to check previous
} ;

// This is basically a big hack to get the number of player info here
const int NUM_PLAYERS = 11;
const int BITS_FOR_PLAYER_NUM = 4;

class Plan; //defined in Plan.h
class PlanExecution; //defined in PlanExecution.h

/****************************** PlanNode ***************************/
class PlanNode
{
  friend class Plan;
  friend class PlanExecution;  

public:
  PlanNode();  
  PlanNode(const PlanNode & pn);  
  //~PlanNode();
  
  virtual plan_node_type_t GetType() const = 0;  

  void Print(ostream& out) const;
  void PrintCompact(BinaryString* pBS) const;

#ifdef NO_ACTION_LOG
  void Log(int level, int node_num = -1) {}
#else
  void Log(int level, int node_num = -1);
#endif //NO_ACTION_LOG

  virtual void PrintDataStr(char* str) const {str[0] = 0;}
  void PrintData(ostream& out) const ;  
  virtual void PrintDataCompact(BinaryString* pBS) const {}

  virtual bool ReadData(istream& in)
  { return true; }
  virtual bool ReadDataCompact(BinaryString* pBS) 
  { return true; }

  virtual PlanNode* makeCopy() const = 0; //makes a copy of the node  

  virtual void FlipCoords(bool flipX, bool flipY) {} //flip all coords across (0,0)
  virtual void AddToAllLocations(VecPosition v) {} //add a vector to all locations

  //replace old player nums with new ones
  virtual void ReplacePlayerNums(Unum * arrRep) {}   

protected:
  virtual player_involved_t IsPlayerInvolved(Unum num) const {return PI_No;} 

};

/****************************** Utility Funcs ***************************/
PlanNode* GetNewPlanNodeOfType(plan_node_type_t t);
PlanNode* GetNewPlanNodeOfType(char* type_name);
const char* GetPlanNodeName(plan_node_type_t t);

/******************* specific PlanNode classes **************************/
class InitialPosPN : public PlanNode
{
public:
  InitialPosPN();
  InitialPosPN(const InitialPosPN& pn);
  
  plan_node_type_t GetType() const {return PN_InitialPos;}

  void PrintDataStr(char* str) const;
  void PrintDataCompact(BinaryString* pBS) const;

  bool ReadData(istream& in);
  bool ReadDataCompact(BinaryString* pBS);
  
  PlanNode* makeCopy() const { return new InitialPosPN(*this); }

  void FlipCoords(bool flipX, bool flipY);  
  void AddToAllLocations(VecPosition v);

  void ReplacePlayerNums(Unum * arrRep);   

  //These are 1 based!
  VecPosition GetPlayerPos(Unum num) const
  { return player_pos[num-1]; }
  bool GetPlayerPosSet(Unum num) const
  { return pos_set[num-1]; }

  void SetPlayerPos(Unum num, VecPosition v);
  void SetPlayerPosSet(Unum num, bool val = true);

  void AdjustPosToWithin(const Rectangle* r);

protected:
  //This is 1 based!
  player_involved_t IsPlayerInvolved(Unum num) const ;

private:
  bool pos_set[NUM_PLAYERS]; //whether player pos is specified
  VecPosition player_pos[NUM_PLAYERS];
} ;
  
  
class ChoosePN : public PlanNode
{
public:
  ChoosePN() {}
  ChoosePN(const ChoosePN& pn) {}  
  plan_node_type_t GetType() const {return PN_Choose;}
  PlanNode* makeCopy() const { return new ChoosePN(*this); }
} ;
  
class AnyOfPN : public PlanNode
{
public:
  AnyOfPN() {}
  AnyOfPN(const AnyOfPN& pn) {}  
  plan_node_type_t GetType() const {return PN_AnyOf;}
  PlanNode* makeCopy() const { return new AnyOfPN(*this); }
} ;
  
class StartPassPN : public PlanNode
{
public:
  StartPassPN();  
  StartPassPN(const StartPassPN& pn);

  plan_node_type_t GetType() const {return PN_StartPass;}

  void PrintDataStr(char* str) const ;
  void PrintDataCompact(BinaryString* pBS) const ;

  bool ReadData(istream& in);
  bool ReadDataCompact(BinaryString* pBS);

  PlanNode* makeCopy() const { return new StartPassPN(*this); }

  void FlipCoords(bool flipX, bool flipY) 
  { if (flipX) toPos.setX(-toPos.getX()); if (flipY) toPos.setY(-toPos.getY()); } 
  void AddToAllLocations(VecPosition v) {toPos += v;} 

  void ReplacePlayerNums(Unum * arrRep) { fromPlayer = arrRep[fromPlayer]; }   

  //specific to this type
  Unum GetPlayer() const { return fromPlayer; }
  VecPosition GetToPos() const { return toPos; } 
  void SetPlayer(int num) { fromPlayer = num; }
  void SetToPos(VecPosition v) { toPos = v; } 

protected:
  player_involved_t IsPlayerInvolved(Unum num) const 
  {return (fromPlayer == num) ? PI_Yes : PI_No;}  

private:
  Unum fromPlayer;
  VecPosition toPos;
  //SMURF: may want more information about relative position of pt and player
} ;
  
class EndPassPN : public PlanNode
{
public:
  EndPassPN() {}
  EndPassPN(const EndPassPN& pn) {}  
  plan_node_type_t GetType() const {return PN_EndPass;}
  PlanNode* makeCopy() const { return new EndPassPN(*this); }
protected:
  player_involved_t IsPlayerInvolved(Unum num) const {return PI_CheckPrevious;}  
} ;
  
class StartGotoPN : public PlanNode
{
public:
  StartGotoPN();
  StartGotoPN(const StartGotoPN& pn);
  
  plan_node_type_t GetType() const {return PN_StartGoto;}

  void PrintDataStr(char* str) const ;
  void PrintDataCompact(BinaryString* pBS) const ;

  bool ReadData(istream& in);
  bool ReadDataCompact(BinaryString* pBS);

  PlanNode* makeCopy() const { return new StartGotoPN(*this); }

  void FlipCoords(bool flipX, bool flipY)
  { if (flipX) toPos.setX(-toPos.getX()); if (flipY) toPos.setY(-toPos.getY()); } 
  void AddToAllLocations(VecPosition v) {toPos += v;} 

  void ReplacePlayerNums(Unum * arrRep) { player = arrRep[player]; }   

  //specific to this type
  Unum GetPlayer() const { return player; }
  VecPosition GetToPos() const { return toPos; } 
  void SetPlayer(int num) { player = num; }
  void SetToPos(VecPosition v) { toPos = v; } 

protected:
  player_involved_t IsPlayerInvolved(Unum num) const 
  {return (player == num) ? PI_Yes : PI_No;}  

private:
  Unum player;
  VecPosition toPos;
} ;
  
class EndGotoPN : public PlanNode
{
public:
  EndGotoPN() {}
  EndGotoPN(const EndGotoPN& pn) {}  
  plan_node_type_t GetType() const {return PN_EndGoto;}
  PlanNode* makeCopy() const { return new EndGotoPN(*this); }
protected:
  player_involved_t IsPlayerInvolved(Unum num) const {return PI_CheckPrevious;}  
} ;
  
class SendBallPN : public PlanNode
{
public:
  SendBallPN();
  SendBallPN(const SendBallPN& pn);
  
  plan_node_type_t GetType() const {return PN_SendBall;}

  void PrintDataStr(char* str) const ;
  void PrintDataCompact(BinaryString* pBS) const ;

  bool ReadData(istream& in);
  bool ReadDataCompact(BinaryString* pBS);

  PlanNode* makeCopy() const { return new SendBallPN(*this); }

  void FlipCoords(bool flipX, bool flipY)
  { if (flipX) toPos.setX(-toPos.getX()); if (flipY) toPos.setY(-toPos.getY()); } 
  void AddToAllLocations(VecPosition v) {toPos += v;} 

  void ReplacePlayerNums(Unum * arrRep) { player = arrRep[player]; }   

  //specific to this type
  Unum GetPlayer() const { return player; }
  VecPosition GetToPos() const { return toPos; } 
  void SetPlayer(int num) { player = num; }
  void SetToPos(VecPosition v) { toPos = v; } 

protected:
  player_involved_t IsPlayerInvolved(Unum num) const 
  {return (player == num) ? PI_Yes : PI_No;}  

private:
  Unum player;
  VecPosition toPos;
} ;
  
class StartDribblePN : public PlanNode
{
public:
  StartDribblePN();
  StartDribblePN(const StartDribblePN& pn);
  
  plan_node_type_t GetType() const {return PN_StartDribble;}

  void PrintDataStr(char* str) const ;
  void PrintDataCompact(BinaryString* pBS) const ;

  bool ReadData(istream& in);
  bool ReadDataCompact(BinaryString* pBS);

  PlanNode* makeCopy() const { return new StartDribblePN(*this); }

  void FlipCoords(bool flipX, bool flipY)
  { if (flipX) toPos.setX(-toPos.getX()); if (flipY) toPos.setY(-toPos.getY()); } 
  void AddToAllLocations(VecPosition v) {toPos += v;} 

  void ReplacePlayerNums(Unum * arrRep) { player = arrRep[player]; }   

  //specific to this type
  Unum GetPlayer() const { return player; }
  VecPosition GetToPos() const { return toPos; } 
  void SetPlayer(int num) { player = num; }
  void SetToPos(VecPosition v) { toPos = v; } 

protected:
  player_involved_t IsPlayerInvolved(Unum num) const 
  {return (player == num) ? PI_Yes : PI_No;}  

private:
  Unum player;
  VecPosition toPos;
} ;
  
class EndDribblePN : public PlanNode
{
public:
  EndDribblePN() {}
  EndDribblePN(const EndDribblePN& pn) {}  
  plan_node_type_t GetType() const {return PN_EndDribble;}
  PlanNode* makeCopy() const { return new EndDribblePN(*this); }
protected:
  player_involved_t IsPlayerInvolved(Unum num) const {return PI_CheckPrevious;}  
} ;
  




#endif 
