/* The class(es) which represents one node of a plan */

#include <stdio.h>
#include <string.h>
#include "PlanNode.h"
#include "misc.h"
#include "utility.h"
#include "Logger.h"
using namespace spades;


const char* plan_node_strings[num_plan_node_types] = PLAN_NODE_STRINGS;

/************************************************************************/
/***************************** Utility Funcs ***************************/
/************************************************************************/
PlanNode* GetNewPlanNodeOfType(plan_node_type_t t)
{
  PlanNode* pPN = NULL;

  switch (t) {
  case PN_InitialPos: pPN = new InitialPosPN; break;
  case PN_Choose: pPN = new ChoosePN; break;
  case PN_AnyOf: pPN = new AnyOfPN; break;
  case PN_StartPass: pPN = new StartPassPN; break;
  case PN_EndPass: pPN = new EndPassPN; break;
  case PN_StartGoto: pPN = new StartGotoPN; break;
  case PN_SendBall: pPN = new SendBallPN; break;
  case PN_EndGoto: pPN = new EndGotoPN; break;
  case PN_StartDribble: pPN = new StartDribblePN; break;
  case PN_EndDribble: pPN = new EndDribblePN; break;
  default:
    errorlog << "Unknown PlanNode type: " << t << ende;
    return NULL;    
  }
  
  if (pPN == NULL) {
    errorlog << "PlanNode allocation failed" << ende;
  }

  return pPN;  
}

PlanNode* GetNewPlanNodeOfType(char* type_name) 
{
  for (int i=0; i<num_plan_node_types; i++) {
    if (strcasecmp(type_name, plan_node_strings[i]) == 0)
      return GetNewPlanNodeOfType((plan_node_type_t)i);
  }
  return NULL;  
}

const char* GetPlanNodeName(plan_node_type_t t)
{
  return plan_node_strings[(int)t];  
}



/************************************************************************/
/*************************** PlanNode **********************************/
/************************************************************************/
PlanNode::PlanNode()
{
}
PlanNode::PlanNode(const PlanNode & pn)
{
}
void PlanNode::Print(ostream& out) const 
{
  out << plan_node_strings[GetType()] << ":";
  PrintData(out);  
}

void PlanNode::PrintCompact(BinaryString* pBS) const 
{
  pBS->PutInt(GetType(), bits_for_plan_node_type);
  PrintDataCompact(pBS);  
}

void PlanNode::PrintData(ostream& out) const 
{
  char outstring[200];
  PrintDataStr(outstring);
  out << outstring;  
}

#ifndef NO_ACTION_LOG
void PlanNode::Log(int level, int node_num)
{
  char datastr[250];
  PrintDataStr(datastr);
  actionlog(level) << "Node " << node_num << " (" << plan_node_strings[GetType()] << "): "
		   << datastr << ende;
}
#endif


/************************************************************************/
/******************* specific PlanNode classes **************************/
/************************************************************************/

/*************************** InitialPosPN *******************************/
InitialPosPN::InitialPosPN()
{
  for (int i=0; i<NUM_PLAYERS; i++) {
    pos_set[i] = false;
    player_pos[i] = VecPosition(0,0);    
  }
}
InitialPosPN::InitialPosPN(const InitialPosPN& pn)
{
  for (int i=0; i<NUM_PLAYERS; i++) {
    pos_set[i] = pn.pos_set[i];
    player_pos[i] = pn.player_pos[i];    
  }
}
void InitialPosPN::PrintDataStr(char* str) const
{
  char* pc = str;
  
  *(pc++) = ' ';
  for (int i=0; i<NUM_PLAYERS; i++) {
    if (pos_set[i])
      *(pc++) = char_for_num(i+1);
    else
      *(pc++) =  '_';
  }

  for (int i=0; i<NUM_PLAYERS; i++) {
    if (pos_set[i]) {
      *(pc++) = ' ';
      std::ostrstream o(pc,100);
      o << player_pos[i] << ends;
      pc += strlen(pc);      
    }
    
  }

  *pc = 0;
  
}
void InitialPosPN::PrintDataCompact(BinaryString* pBS) const
{
  for (int i=0; i<NUM_PLAYERS; i++) {
    pBS->PutBit(pos_set[i]);
    if (pos_set[i]) {
      errorlog << "I can't print to a BinaryString!" << ende;
      //player_pos[i].PrintToBinStr(pBS, bits_for_vector_comp, -min_val_for_vector_comp);
    }    
  }  
}
bool InitialPosPN::ReadData(istream& in)
{
  char ch;
  
  skip_white_space(in);
  for (int i=0; i<NUM_PLAYERS; i++) {
    if (!in.get(ch)) return false;
    if (ch == '_') {
      pos_set[i] = false;
      continue;      
    }
    pos_set[i] = true;    
    if (ch != char_for_num(i+1))
      errorlog << "InitialPosPN::ReadData: pos_set char mismatch: saw:"
	       << ch << " expected:" << char_for_num(i+1) << ende;
  }
  for (int i=0; i<NUM_PLAYERS; i++) {
    if (pos_set[i])
      in >> player_pos[i];    
  }
  return in;  
}
bool InitialPosPN::ReadDataCompact(BinaryString* pBS)
{
  for (int i=0; i<NUM_PLAYERS; i++) {
    pos_set[i] = pBS->GetBit();
    if (pos_set[i]) {
      errorlog << "I can not deal with BinaryString" << ende;
      //if (!player_pos[i].ReadFromBinStr(pBS, bits_for_vector_comp, 
      //-min_val_for_vector_comp))
      //return false;      
    }    
  }  
  return true;  
}
void InitialPosPN::FlipCoords(bool flipX, bool flipY)
{
  for (int i=0; i<NUM_PLAYERS; i++) {
    if (flipX)
      player_pos[i].setX(-player_pos[i].getX());    
    if (flipY)
      player_pos[i].setY(-player_pos[i].getY());    
  }  
}
void InitialPosPN::AddToAllLocations(VecPosition v)
{
  for (int i=0; i<NUM_PLAYERS; i++) {
    player_pos[i] += v;    
  }  
}

void InitialPosPN::ReplacePlayerNums(Unum * arrRep)
{
  bool new_pos_set[NUM_PLAYERS];
  VecPosition new_player_pos[NUM_PLAYERS];
  
  actionlog(230) << "InitialPosPN: clearing new arrays" << ende;
  for (int i=0; i<NUM_PLAYERS; i++) {
    new_pos_set[i] = false;
    new_player_pos[i] = 0;    
  }
  
  actionlog(230) << "InitialPosPN: transfering values" << ende;
  for (int i = 0; i<NUM_PLAYERS; i++) {
    if (pos_set[i]) {
      new_pos_set[arrRep[i+1]-1] = true;
      new_player_pos[arrRep[i+1]-1] = player_pos[i];
    }
  }

  actionlog(230) << "InitialPosPN: moving back to original" << ende;
  for (int i=0; i<NUM_PLAYERS; i++) {
    pos_set[i] = new_pos_set[i];
    player_pos[i] = new_player_pos[i];    
  }

  actionlog(230) << "InitialPosPN: done replacing" << ende;  
}



player_involved_t InitialPosPN::IsPlayerInvolved(Unum num) const
{
  if (num == Unum_Unknown)
    errorlog << "InitialPosPN::IsPlayerInvolved: don't ask about Unum_Unknown" << ende;
  return (pos_set[num-1]) ? PI_Yes : PI_No;  
}

void InitialPosPN::SetPlayerPos(Unum num, VecPosition v)
{ 
  if (num < 0 || num >= NUM_PLAYERS) {
    errorlog << "InitialPosPN: trying to set out of range player: " << num << ende;
    return;
  }
  player_pos[num-1] = v; 
  pos_set[num-1] = true;  
}
void InitialPosPN::SetPlayerPosSet(Unum num, bool val)
{ 
  if (num < 0 || num >= NUM_PLAYERS) {
    errorlog << "InitialPosPN: trying to set out of range player: " << num << ende;
    return;
  }
  pos_set[num-1] = val; 
}

void InitialPosPN::AdjustPosToWithin(const Rectangle* r)
{
  for (int i = 0; i<NUM_PLAYERS; i++) {
    if (pos_set[i]) {
      player_pos[i] = r->adjustToWithin(player_pos[i], 0);
    }
  }
}



/*************************** StartPassPN *******************************/
StartPassPN::StartPassPN()
{
  fromPlayer = Unum_Unknown;
  toPos = VecPosition(0,0);
}
StartPassPN::StartPassPN(const StartPassPN& pn)
{
  fromPlayer = pn.fromPlayer;
  toPos = pn.toPos;  
}
void StartPassPN::PrintDataStr(char* str) const
{
  char* pc;
  sprintf(str, " %d ", fromPlayer);
  pc = &str[strlen(str)];
  std::ostrstream o(pc,100);
  o << toPos << ends;
}
void StartPassPN::PrintDataCompact(BinaryString* pBS) const
{
  errorlog << "I can not deal with BinaryString" << ende;
  //pBS->PutInt(fromPlayer, BITS_FOR_PLAYER_NUM);
  //toPos.PrintToBinStr(pBS, bits_for_vector_comp, -min_val_for_vector_comp);
}
bool StartPassPN::ReadData(istream& in)
{
  in >> fromPlayer >> toPos;
  return in;  
}
bool StartPassPN::ReadDataCompact(BinaryString* pBS)
{
  fromPlayer = pBS->GetInt(BITS_FOR_PLAYER_NUM);
  if (fromPlayer == -1)
    return false;
  errorlog << "I can not deal with BinaryString" << ende;
  return false;
  //return  toPos.ReadFromBinStr(pBS, bits_for_vector_comp, -min_val_for_vector_comp);
}


/*************************** StartGotoPN *******************************/
StartGotoPN::StartGotoPN()
{
  player = Unum_Unknown;
  toPos = VecPosition(0,0);
}
StartGotoPN::StartGotoPN(const StartGotoPN& pn)
{
  player = pn.player;  
  toPos = pn.toPos;  
}
void StartGotoPN::PrintDataStr(char* str) const
{
  char* pc;
  sprintf(str, " %d ", player);
  pc = &str[strlen(str)];
  std::ostrstream o(pc,100);
  o << toPos << ends;
}
void StartGotoPN::PrintDataCompact(BinaryString* pBS) const
{
  pBS->PutInt(player, BITS_FOR_PLAYER_NUM);
  errorlog << "I can not deal with BinaryString" << ende;
  //toPos.PrintToBinStr(pBS, bits_for_vector_comp, -min_val_for_vector_comp);
}
bool StartGotoPN::ReadData(istream& in)
{
  in >> player >> toPos;
  return in;  
}
bool StartGotoPN::ReadDataCompact(BinaryString* pBS)
{
  player = pBS->GetInt(BITS_FOR_PLAYER_NUM);
  if (player == -1)
    return false;  
  errorlog << "I can not deal with BinaryString" << ende;
  return false;
  //return  toPos.ReadFromBinStr(pBS, bits_for_vector_comp, -min_val_for_vector_comp);
}


/*************************** SendBallPN *******************************/
SendBallPN::SendBallPN()
{
  player = Unum_Unknown;
  toPos = VecPosition(0,0);
}
SendBallPN::SendBallPN(const SendBallPN& pn)
{
  player = pn.player;  
  toPos = pn.toPos;  
}
void SendBallPN::PrintDataStr(char* str) const
{
  char* pc;
  sprintf(str, " %d ", player);
  pc = &str[strlen(str)];
  std::ostrstream o(pc,100);
  o << toPos << ends;
}
void SendBallPN::PrintDataCompact(BinaryString* pBS) const
{
  pBS->PutInt(player, BITS_FOR_PLAYER_NUM);
  errorlog << "I can not deal with BinaryString" << ende;
  //toPos.PrintToBinStr(pBS, bits_for_vector_comp, -min_val_for_vector_comp);
}
bool SendBallPN::ReadData(istream& in)
{
  in >> player >> toPos;
  return in;  
}
bool SendBallPN::ReadDataCompact(BinaryString* pBS)
{
  player = pBS->GetInt(BITS_FOR_PLAYER_NUM);
  if (player == -1)
    return false;  
  errorlog << "I can not deal with BinaryString" << ende;
  return false;
  //return toPos.ReadFromBinStr(pBS, bits_for_vector_comp, -min_val_for_vector_comp);
}


/*************************** StartDribblePN *******************************/
StartDribblePN::StartDribblePN()
{
  player = Unum_Unknown;
  toPos = VecPosition(0,0);
}
StartDribblePN::StartDribblePN(const StartDribblePN& pn)
{
  player = pn.player;  
  toPos = pn.toPos;  
}
void StartDribblePN::PrintDataStr(char* str) const
{
  char* pc;
  sprintf(str, " %d ", player);
  pc = &str[strlen(str)];
  std::ostrstream o(pc,100);
  o << toPos << ends;
}
void StartDribblePN::PrintDataCompact(BinaryString* pBS) const
{
  pBS->PutInt(player, BITS_FOR_PLAYER_NUM);
  errorlog << "I can not deal with BinaryString" << ende;
  //toPos.PrintToBinStr(pBS, bits_for_vector_comp, -min_val_for_vector_comp);
}
bool StartDribblePN::ReadData(istream& in)
{
  in >> player >> toPos;
  return in;  
}
bool StartDribblePN::ReadDataCompact(BinaryString* pBS)
{
  player = pBS->GetInt(BITS_FOR_PLAYER_NUM);
  if (player == -1)
    return false;  
  errorlog << "I can not deal with BinaryString" << ende;
  return false;
  //return toPos.ReadFromBinStr(pBS, bits_for_vector_comp, -min_val_for_vector_comp);
}


