/* -*- Mode: C++ -*- */

/* The class which represents the current execution state of a plan */

#include <stdio.h>
#include <iostream.h>
#include <list>

#include "PlanExecution.h"
#include "Plan.h"
#include "ServerParam.h"
#include "CoachParam.h"
#include "Logger.h"
using namespace spades;
    
PlanExecution::PlanExecution()
{
  thePlan = NULL;  
  ClearNodeInfo();  
  fresh_plan = false;  
  callResetOnLoad = true;
  curr_time = -1;
}

/*
 You should explictly use Load
PlanExecution::PlanExecution(Plan* p)
{
  Load(p);  
  ClearNodeInfo();  
}
*/

PlanExecution::~PlanExecution()
{
}

   
  
   
//reset the execution	
void PlanExecution::Reset()
{
  if (thePlan == NULL) 
    return;

  thePlan->ComputeAllPairsDistances(); //needed to set up upper and lower bounds  

  for (int i=0; i<thePlan->num_nodes; i++) {
    node_status[i] = NS_Pending;
    node_exec_time[i] = -1;    
    lower_exec_bound[i] = -thePlan->edge_len[i][0];
    upper_exec_bound[i] =  thePlan->edge_len[0][i];    
  }

  node_status[0] = NS_Ready; //should be a PN_InitialPos    

  //SMURF: I think the filtering is broken!
  //thePlan->FilterEdges(); //removes unneccesary edges  

  ClearExtraNodeInfo();  

}

//removes the current plan
void PlanExecution::Clear()
{
  thePlan = NULL;
  ClearNodeInfo();
  fresh_plan = false;  
}

//loads a new plan for the plan execution
void PlanExecution::Load(Plan* p, Problem* prob)
{
  if (p != NULL && p->num_nodes == 0) {
    errorlog << "PlanExecution::Load: Plan is not null, but it's empty!" << ende;
    return;
  }

  if (p != NULL && 
      p->nodes[0]->GetType() != PN_InitialPos) {
    errorlog << "PlanExecution::Load: Plan must start with InitialPos!" << ende;
    return;
  }
  
  theProblem = prob;
  thePlan = p;

  //note that Reset makes the Plan dispatchable
  if (callResetOnLoad) {
    Reset();  
    LogBounds(100);  
  }

  /*
  if (thePlan)
    thePlan->Print(cout);  
  */

  fresh_plan = true;  
}

  
void PlanExecution::ClearNodeInfo()
{
  for (int i=0; i<Plan_max_num_nodes; i++) {
    node_status[i] = NS_NoValue;
    node_exec_time[i] = -1;    
    lower_exec_bound[i] = -1;
    upper_exec_bound[i] = -1;    
  }  
  ClearExtraNodeInfo();  
}


int PlanExecution::GetPlayerActiveNode(Unum num, int start_node)
{
  if (thePlan == NULL)
    return -1;
  
  for (int i=start_node; i < thePlan->num_nodes; i++) {
    if (node_status[i] == NS_Complete)
      continue;    
    if (num == Unum_Unknown || thePlan->IsPlayerInvolved(num,i))
      return i;    
  }
  return -1;  
}

int PlanExecution::GetNodeHolding(int node_num)
{
  if (node_status[node_num] != NS_Pending) {
    errorlog << "GetNodeHolding not meaningful if not NS_Pending" << ende;
    return -1;    
  }
  
  //a node X is holding us from becoming Ready if there is a <= 0 edge to X from us
  for (int i=0; i<thePlan->num_nodes; i++) {
    if (i==node_num)
      continue; //nodes don't hold themselves    
    if (node_status[i] != NS_Ready && 
	node_status[i] != NS_Pending ) {
      actionlog(240) << "Hold: " << i << " can't hold " << node_num
		     << " since it's not ready or pending" << ende;      
      continue; //only worry about ready and pending nodes constraining us
    }
    
    if (thePlan->edge_len[node_num][i] <= 0) {
      actionlog(240) << "Hold: " << i << " holds " << node_num << " since edge_len is "
		     << thePlan->edge_len[node_num][i] << ende;      
      return i; //returning the first one is fine
    }    
    actionlog(240) << "Hold: " << i << " does not hold " << node_num
		   << " since edge_len is " << thePlan->edge_len[node_num][i] << ende;      
  }

  actionlog(240) << "Nothing holds " << node_num << ende;
  return -1;
}

//NOTE: I'm not entirely sure that forcing all holding nodes to also be executed is
// the right thing to do
void PlanExecution::SetNodeExecTime(int idx, int t)
{
  if (node_status[idx] == NS_Complete) {
    errorlog << "Node " << idx << " was already set complete at "
	     << node_exec_time[idx] << " (tried " << t << ")" << ende;
    return;
  }

  while (node_status[idx] != NS_Ready) {
    int holding_node = GetNodeHolding(idx);    
    if (holding_node != -1) {
      actionlog(50) << "In order to set " << idx << " complete, I'm recursing on "
		    << holding_node << ende;
      SetNodeExecTime(holding_node, t);
    } else {
      actionlog(70) << "Node " << idx << " is held by time only "
		    << "(" << lower_exec_bound[idx] << ", " << upper_exec_bound[idx]
		    << "but I'll set exec time anyway"  << ende;
      node_status[idx] = NS_Ready;
    }
  }
    
  actionlog(40) << "Plan Node " << idx << " complete" << ende;

  if (idx != 0 &&
      (t < lower_exec_bound[idx] || t > upper_exec_bound[idx])) {
    errorlog << "Setting execution time for " << idx << " out of bounds "
	     << lower_exec_bound[idx] << " "
	     << t << " " 
	     << upper_exec_bound[idx] << " "
	     << ende;
  }
  
  node_status[idx] = NS_Complete;
  node_exec_time[idx] = t;
  lower_exec_bound[idx] = upper_exec_bound[idx] = t;
  
  if (idx == 0) {
    //0 is the start node, so we know shift all the upper and lower bounds to be in
    //global time
    for (int prop_idx=1; prop_idx<thePlan->num_nodes; prop_idx++) {
      lower_exec_bound[prop_idx] += t;
      upper_exec_bound[prop_idx] += t;      
    }    
  }  

  //propogate exec time info to upper/lower bounds!
  for (int prop_idx=0; prop_idx<thePlan->num_nodes; prop_idx++) {
    if (thePlan->edge_len[idx][prop_idx] == edge_infty)
      continue; //infty edges are ignored
    /* Muscettola, Morris, and Tsamardinos show that you only have to propogate along
       these kinds of edges */
    if (thePlan->edge_len[idx][prop_idx] < 0) {
      //propogate lower bound
      lower_exec_bound[prop_idx] = 
	Min(lower_exec_bound[prop_idx], t - thePlan->edge_len[idx][prop_idx]);      
    } else {
      //propogate upper bound
      upper_exec_bound[prop_idx] =
	Max(upper_exec_bound[prop_idx], t + thePlan->edge_len[idx][prop_idx]);      
    }
  } 

  LogBounds(100);  

  //NOTE: I'm not entirely sure this is the right time to pass on
  MakeNodesReady(t);  

}

void PlanExecution::MakeNodesReady(int time) 
{
  if (thePlan == NULL)
    return;
  
  for (int idx = 0; idx < thePlan->GetNumNodes(); idx++) {
    if (GetNodeStatus(idx) != NS_Pending) {      
      actionlog(220) << "Node " << idx << "(" << GetPlanNodeName(GetPlanNode(idx)->GetType())
		     << ") status: " << GetNodeStatus(idx) << ", so can't make ready"
		     << ende;	
      continue; //only pending nodes become ready
    }
    
    int holding_node = GetNodeHolding(idx);    
    if (holding_node == -1) {
      //no node holds it, but only make ready if the lower time bound has been achieved
      if (idx == 0 || 
	  (idx != 0 && GetLowerExecBound(idx) <= time)) {
	actionlog(40) << "Node " << idx << "(" << GetPlanNodeName(GetPlanNode(idx)->GetType())
		      << ") made ready"  << ende;	
	SetNodeStatus(idx, NS_Ready);      
      } else {
	actionlog(220) << "Node " << idx << "(" << GetPlanNodeName(GetPlanNode(idx)->GetType())
		       << ") not held, but haven't hit lower bound "
		       << GetLowerExecBound(idx) << ende;	
      }
      
    } else {
      actionlog(220) << "Node " << idx << "(" << GetPlanNodeName(GetPlanNode(idx)->GetType())
		     << ") being held by " << holding_node << ", so can't make ready"
		     << ende;	
    }
    
  }
}



bool PlanExecution::CheckConsistency()
{
  for (int i=0; i<thePlan->num_nodes; i++) {
    if (lower_exec_bound[i] > upper_exec_bound[i])
      return false;    
  }
  return true;  
}

#ifndef NO_ACTION_LOG

void PlanExecution::LogBounds(int level)
{
  if (thePlan == NULL)
    return;

  if (level > Logger::instance()->getMaxActionLogLevel())
    return; //don't log anything!

  char outstring[300];
  char* pc;
  
  //pc should always point to the null
  strcpy(outstring, "Lower Bounds");
  pc = outstring + strlen(outstring);  
  for (int i=0; i<thePlan->num_nodes; i++) {
    sprintf(pc, "%5d", lower_exec_bound[i]);
    pc += 5;    
  }
  actionlog(level) << outstring << ende;  

  strcpy(outstring, "Upper Bounds");
  pc = outstring + strlen(outstring);  
  for (int i=0; i<thePlan->num_nodes; i++) {
    sprintf(pc, "%5d", upper_exec_bound[i]);
    pc += 5;    
  }
  actionlog(level) << outstring << ende;  
}

#endif //NO_ACTION_LOG


void PlanExecution::ClearExtraNodeInfo()
{
  for (int i= ((thePlan) ? thePlan->num_nodes : Plan_max_num_nodes)-1;
       i>=0; i--) {
    extra_node_info[i] = VecPosition(1000,1000); //off the field    
  }
}

/****************************************************************************/
const std::string PlanExecution::NAME_PREFIX = "SP_";
const std::string PlanExecution::NAME_INIT_POS_COND_PREFIX = NAME_PREFIX + "IP_cond_";
const std::string PlanExecution::NAME_STOP_RULE = NAME_PREFIX + "stop";
const std::string PlanExecution::NAME_GOALIE_NO_GK_RULE = NAME_PREFIX + "goalienogk";

bool PlanExecution::ConvertToStandardLanguage(CoachMessageQueue* pQueue)
{
  //return ConvertToStandardLanguage_Simple(pQueue);
  return ConvertToStandardLanguage_Start(pQueue);
}

bool PlanExecution::StandardLanguageSetup(CoachMessageQueue* pQueue)
{
  // the stop rule
  {
    rcss::clang::Cond* pCond =
      new rcss::clang::CondNot(std::auto_ptr<rcss::clang::Cond>(new rcss::clang::CondPlayMode(rcss::clang::PlayOn)));
  
    rcss::clang::SimpleRule* rule =
      new rcss::clang::SimpleRule(std::auto_ptr<rcss::clang::Cond>(pCond));

    // For each player, tell it to position where it is
    for (int player = 1; player <= NUM_PLAYERS; player++)
      {
	rcss::clang::UNumSet uset;
	uset.clear();
	uset.add(rcss::clang::UNum(player));

	rcss::clang::ActPos* act = 
	  new rcss::clang::ActPos(std::auto_ptr<rcss::clang::Region>(new rcss::clang::RegPoint(std::auto_ptr<rcss::clang::Point>(new rcss::clang::PointPlayer(true, rcss::clang::UNum(player))))));

	std::list<rcss::clang::Action*> listact(1, act);
	rcss::clang::DirComm* dir =
	  new rcss::clang::DirComm(true, true, uset, listact);
  
	rule->push_back(std::auto_ptr<rcss::clang::Dir>(dir));
      }

    //Also put in a don't intercept
    rcss::clang::UNumSet uset;

    uset.add(rcss::clang::UNum::uAll);

    std::list<rcss::clang::Action*> listact(1, new rcss::clang::ActIntercept);
    rcss::clang::DirComm* dir =
      new rcss::clang::DirComm(false, true, uset, listact);
  
    rule->push_back(std::auto_ptr<rcss::clang::Dir>(dir));

    // Now finally define this thing and we're done!
    rcss::clang::DefRule* deftok =
      new rcss::clang::DefRule(NAME_STOP_RULE,
			       std::auto_ptr<rcss::clang::Rule>(rule),
			       false);

    pQueue->getDefineContainer().push(deftok);
  }

#ifdef USE_GOALIE_NO_GK	
  //rule to tell the goalie not to take a goal kick
  {
    rcss::clang::Cond* pCond =
      new rcss::clang::CondPlayMode(rcss::clang::GoalKick_Our);
  
    rcss::clang::SimpleRule* rule =
      new rcss::clang::SimpleRule(std::auto_ptr<rcss::clang::Cond>(pCond));

    //Also put in a don't intercept
    rcss::clang::UNumSet uset;

    uset.add(rcss::clang::UNum::u1);

    std::list<rcss::clang::Action*> listact(1, new rcss::clang::ActIntercept);
    rcss::clang::DirComm* dir =
      new rcss::clang::DirComm(false, true, uset, listact);
  
    rule->push_back(std::auto_ptr<rcss::clang::Dir>(dir));

    // define it
    rcss::clang::DefRule* deftok =
      new rcss::clang::DefRule(NAME_GOALIE_NO_GK_RULE,
			       std::auto_ptr<rcss::clang::Rule>(rule),
			       false);

    pQueue->getDefineContainer().push(deftok);

    // and turn it on
    rcss::clang::RuleIDList idlist;
    idlist.clear();
    idlist.push_front(NAME_GOALIE_NO_GK_RULE);
    pQueue->getRuleContainer().push(new rcss::clang::ActivateRules(true, idlist));

  }
#endif
  
  return true;
}

bool PlanExecution::StandardLanguageStartPlan(CoachMessageQueue* pQueue)
{
  StandardLanguageDeleteAll(pQueue);
  rcss::clang::RuleIDList idlist;
  idlist.clear();
  idlist.push_front(NAME_STOP_RULE);
  pQueue->getRuleContainer().push(new rcss::clang::ActivateRules(true, idlist));
  return true;
}

void
PlanExecution::addStandardLanguageRule(const std::string& prefix,
				       rcss::clang::Rule* rule,
				       CoachMessageQueue* pQueue)
{
  std::string rule_name = prefix + toString(curr_rule_id++);
      
  rcss::clang::DefRule* deftok =
    new rcss::clang::DefRule(rule_name,
			     std::auto_ptr<rcss::clang::Rule>(rule),
			     false);
  
  pQueue->getDefineContainer().push(deftok);
  setplay_rule_id_list.push_back(rule_name);
}




/* This function takes advantage of some oddities in the way I happen to produce STNs
   via the RoleMaker mechanism, and does NOT work for all STNs represenatable (even with
   the restrictions noted in Plan.h */
/* this is like the _Simple variety, except that it does some different processing
   for the starter so that it doesn't go right to the ball immediately */
bool PlanExecution::ConvertToStandardLanguage_Start(CoachMessageQueue* pQueue)
{
  const float FIRST_PASSER_DISP = 4.0;

  bool error = false;
  std::string this_set_play_prefix = NAME_PREFIX + toString(curr_time) + "_";
  std::string this_set_play_ip_cond = NAME_INIT_POS_COND_PREFIX + toString(curr_time);
  
  curr_rule_id = 0;
  setplay_rule_id_list.clear();

  if (!thePlan) {
    errorlog << "Plan::ConvertToStandardLanguage: I have no plan!" << ende;
    return false;
  }

  //this insures that all the execution bounds are set up
  Reset();

  int max_exec_time = -1;
  for (int i=0; i<thePlan->num_nodes; i++)
    if (upper_exec_bound[i] > max_exec_time)
      max_exec_time = upper_exec_bound[i];

  int first_passer = -1;
  VecPosition first_passer_target;
  bool off_set_play = thePlan->num_nodes > 1;
  if (off_set_play) {
    StartPassPN* p = dynamic_cast<StartPassPN*>(thePlan->nodes[1]);
    if (p) {
      first_passer = p->GetPlayer();
      first_passer_target = p->GetToPos();
      actionlog(160) << "SPP Conversion: first passer=" << first_passer << ende;
    } else {
      errorlog << "Why isn't the first plan node a start pass?" << ende;
    }
  }

  //Create the condition for all but the passer being in position
  if (off_set_play) {
    InitialPosPN* pnIP = thePlan->GetInitPos();
    rcss::clang::UNumSet uset;
    rcss::clang::CondAnd* pCond = new rcss::clang::CondAnd;
    for (int num = 1; num <= ServerParam::instance()->getSPTeamSize(); num++) {
      if (!pnIP->GetPlayerPosSet(num))
	continue;
      if (num == first_passer)
	continue; // we do not include the passer in these conditions
      uset.clear();
      uset.add(rcss::clang::UNum(num));
      pCond->push_back(std::auto_ptr<rcss::clang::Cond>(new rcss::clang::CondPlayerPos(true/*ourside*/,
										       uset,
										       1, 1, 
										       std::auto_ptr<rcss::clang::Region>(new rcss::clang::RegArc(Circle(pnIP->GetPlayerPos(num),
																			 CoachParam::instance()->getSppPositionBuffer()))))));
    }
    pQueue->getDefineContainer().push(new rcss::clang::DefCond(this_set_play_ip_cond,
							       std::auto_ptr<rcss::clang::Cond>(pCond)));
  }

  //This condition will be used below
  int this_set_play_setup_time =
    (theProblem->mode == SPM_My_Goalie_Catch) ?
    CoachParam::instance()->getSppMaxPlayerSetupForGC() :
    CoachParam::instance()->getSppMaxPlayerSetup();

  rcss::clang::CondOr* pFirstPasserIPCond = new rcss::clang::CondOr;
  pFirstPasserIPCond->push_back(std::auto_ptr<rcss::clang::Cond>(new rcss::clang::CondNamed(this_set_play_ip_cond)));
  
  pFirstPasserIPCond->push_back(std::auto_ptr<rcss::clang::Cond>(new rcss::clang::CondTime(curr_time + this_set_play_setup_time,										   rcss::util::CompOp::greaterEqual())));
  

  // Now tell everyone who is not the kicker not to intercept
  // I put this in originally for UTAustin
  {
    rcss::clang::UNumSet uset;
    uset = rcss::clang::UNumSet(rcss::clang::UNum::uAll) -
      rcss::clang::UNumSet(rcss::clang::UNum(first_passer));
 
    std::list<rcss::clang::Action*> listact(1, new rcss::clang::ActIntercept);
    rcss::clang::DirComm* pNoIntDir =
      new rcss::clang::DirComm(false, true, uset, listact);
    rcss::clang::Cond* pCond = GetCMCondForSetPlayMode(theProblem->mode, curr_time);
    rcss::clang::SimpleRule* pNoIntRule =
      new rcss::clang::SimpleRule ( std::auto_ptr<rcss::clang::Cond>(pCond) );
    pNoIntRule->getDirs().push_back( pNoIntDir );

    addStandardLanguageRule(this_set_play_prefix, pNoIntRule, pQueue);
  }
   
  // Now go through all the nodes and create the advice
  int next_idx;
  Unum last_ball_owner = -1;
  VecPosition player_loc[ServerParam::instance()->getSPTeamSize()+1]; //1-based index
  
  for (int idx=0; 0 <= idx && idx < thePlan->num_nodes; idx = next_idx) {
    next_idx = -1;
    if (!thePlan->nodes[idx])
      errorlog << "Plan::ConvertToStandardLanguage how is node NULL?" << ende;

    switch(thePlan->nodes[idx]->GetType()) {

    case PN_InitialPos: {
      actionlog(150) << "Converting an InitialPos node" << ende;
      InitialPosPN* pn = (InitialPosPN*)thePlan->nodes[idx];
      rcss::clang::SimpleRule* rule =
	new rcss::clang::SimpleRule(std::auto_ptr<rcss::clang::Cond>(GetCMCondForSetPlayMode(theProblem->mode, curr_time)));

      for (int num=1; num<=ServerParam::instance()->getSPTeamSize(); num++) {
	if (pn->GetPlayerPosSet(num)) {
	  player_loc[num] = pn->GetPlayerPos(num);
	  //the first passer will be handled below
	  if (num == first_passer)
	    continue;
	  rcss::clang::UNumSet uset;
	  uset.add(rcss::clang::UNum(num));

	  rcss::clang::ActPos* act = 
	    new rcss::clang::ActPos(std::auto_ptr<rcss::clang::Region>(new rcss::clang::RegPoint(std::auto_ptr<rcss::clang::Point>(new rcss::clang::PointSimple(pn->GetPlayerPos(num).convertToVector2D())))));

	  std::list<rcss::clang::Action*> listact(1, act);
	  rcss::clang::DirComm* dir =
	    new rcss::clang::DirComm(true, true, uset, listact);
  
	  rule->push_back(std::auto_ptr<rcss::clang::Dir>(dir));
	} else if (num == CoachParam::instance()->getSppUnumMoveForKickoff()) {
	  // This is sort of a hack because player 9 keeps getting in the way
	  rcss::clang::UNumSet uset;
	  uset.add(rcss::clang::UNum(num));

	  rcss::clang::ActPos* act = 
	    new rcss::clang::ActPos(std::auto_ptr<rcss::clang::Region>(new rcss::clang::RegPoint(std::auto_ptr<rcss::clang::Point>(new rcss::clang::PointSimple(rcss::geom::Vector2D(-10, 0))))));

	  std::list<rcss::clang::Action*> listact(1, act);
	  rcss::clang::DirComm* dir =
	    new rcss::clang::DirComm(true, true, uset, listact);
  
	  rule->push_back(std::auto_ptr<rcss::clang::Dir>(dir));
	}
      }
      addStandardLanguageRule(this_set_play_prefix, rule, pQueue);

      //first passer handling!
      if (first_passer > 0) {
	rcss::clang::Point* pPtBall;
	/* After a goal, the ball is not moved until kick off is called (this is after the
	   players can use the move command), so we don't want to use ball relative
	   regions
	   For goalie catches, the goalie will issue a move to move with the ball */
	if (theProblem->mode == SPM_My_Kick_Off) {
	  pPtBall = new rcss::clang::PointSimple(-.2,0);
	} else if (theProblem->mode == SPM_My_Goalie_Catch) {
	  pPtBall = new rcss::clang::PointSimple(player_loc[first_passer].convertToVector2D());
	} else {
	  pPtBall = new rcss::clang::PointBall();
	}
	
	rcss::clang::CondAnd* pCond;
	rcss::clang::UNumSet uset;
	uset.add(rcss::clang::UNum(first_passer));

	/* if it's a goalie catch, we only want our goalie to move once everyone else is ready */
	if (theProblem->mode != SPM_My_Goalie_Catch) {
	  pCond = new rcss::clang::CondAnd;
	  pCond->push_back(std::auto_ptr<rcss::clang::Cond>(GetCMCondForSetPlayMode(theProblem->mode, curr_time)));
	  pCond->push_back(std::auto_ptr<rcss::clang::Cond>(new rcss::clang::CondNot(std::auto_ptr<rcss::clang::Cond>(new rcss::clang::CondNamed(this_set_play_ip_cond)))));
	  pCond->push_back(std::auto_ptr<rcss::clang::Cond>(new rcss::clang::CondTime(curr_time + this_set_play_setup_time,
										      rcss::util::CompOp::less())));
	  rule = new rcss::clang::SimpleRule(std::auto_ptr<rcss::clang::Cond>(pCond));

	  VecPosition offset = (first_passer_target - theProblem->ball_pos).scaleTo(FIRST_PASSER_DISP);
	  rcss::clang::PointArith* ptTarg =
	    new rcss::clang::PointArith(std::auto_ptr<rcss::clang::Point>(new rcss::clang::PointSimple(offset.getX(), offset.getY())),
					pPtBall->deepCopy(),
					rcss::util::ArithOp::plus());
	  rcss::clang::Region* rTarg =
	    new rcss::clang::RegPoint(std::auto_ptr<rcss::clang::Point>(ptTarg));

	  std::list<rcss::clang::Action*> listact;
	  rcss::clang::DirComm* dir;
	  
	  // Tell the agent to go to this pos
	  listact.clear();
	  listact.push_back(new rcss::clang::ActPos(std::auto_ptr<rcss::clang::Region>(rTarg)));
	  dir = new rcss::clang::DirComm(true, true, uset, listact);
	  rule->push_back(std::auto_ptr<rcss::clang::Dir>(dir));

	  //Now tell it no to intercept.
	  listact.clear();
	  listact.push_back(new rcss::clang::ActIntercept);
	  dir = new rcss::clang::DirComm(false, true, uset, listact);
	  rule->push_back(std::auto_ptr<rcss::clang::Dir>(dir));

	  addStandardLanguageRule(this_set_play_prefix, rule, pQueue);
	}
	
	pCond = new rcss::clang::CondAnd;
	pCond->push_back(std::auto_ptr<rcss::clang::Cond>(GetCMCondForSetPlayMode(theProblem->mode, curr_time)));
	pCond->push_back(std::auto_ptr<rcss::clang::Cond>(pFirstPasserIPCond->deepCopy()));

#ifdef COND_KICKER_INTERCEPT
	// Condition this on not being close to the ball so that the player only has one
	// advice at a time to deal with
	if (theProblem->mode != SPM_My_Goalie_Catch)
	  {
	    rcss::clang::RegArc* r =
	      new rcss::clang::RegArc( std::auto_ptr<rcss::clang::Point>(new rcss::clang::PointBall),
				       0, ServerParam::instance()->getSPKickableArea(),
				       0, 360);
	    rcss::clang::Cond* p = 
	      new rcss::clang::CondPlayerPos(true,
					     rcss::clang::UNumSet(rcss::clang::UNum(first_passer)),
					     1, 1,
					     //std::auto_ptr<rcss::clang::Region>(NULL));
					     std::auto_ptr<rcss::clang::Region>(r));
	    pCond->push_back(std::auto_ptr<rcss::clang::Cond>( new rcss::clang::CondNot( std::auto_ptr<rcss::clang::Cond>(p))));
	  }
#endif
	rule = new rcss::clang::SimpleRule(std::auto_ptr<rcss::clang::Cond>(pCond));

#ifdef USE_ACT_TO_POSITION_KICKER	
	rcss::clang::ActPos* act = 
	  new rcss::clang::ActPos(std::auto_ptr<rcss::clang::Region>(new rcss::clang::RegPoint(std::auto_ptr<rcss::clang::Point>(pPtBall->deepCopy()))));
#else
	rcss::clang::ActIntercept* act = 
	  new rcss::clang::ActIntercept();
#endif	
	std::list<rcss::clang::Action*> listact(1, act);
	rcss::clang::DirComm* dir =
	  new rcss::clang::DirComm(true, true, uset, listact);
  
	rule->push_back(std::auto_ptr<rcss::clang::Dir>(dir));

	addStandardLanguageRule(this_set_play_prefix, rule, pQueue);
	  
	delete pPtBall;
      }
      
      next_idx = idx+1;
    } break;
    
    case PN_StartPass: {
      actionlog(150) << "Converting a StartPass node" << ende;
      StartPassPN* pnsp = (StartPassPN*)thePlan->nodes[idx];
      StartGotoPN* pnsg = (StartGotoPN*)thePlan->nodes[idx+1];

      rcss::clang::CondAnd* pCond = new rcss::clang::CondAnd;

      rcss::clang::UNumSet usetpasser, usetreceiver;
      usetpasser.add(rcss::clang::UNum(pnsp->GetPlayer()));
      usetreceiver.add(rcss::clang::UNum(pnsg->GetPlayer()));
      
      if (idx == 1) {
	if (first_passer <= 0)
	  errorlog << "Why don't I have a first passer? " << first_passer << ende;
	//wait for everyone to be ready
	pCond->push_back(pFirstPasserIPCond->deepCopy());
	// we don't want to condition on the play mode because then we
	// drop out of the condition matching after the first kick!
	//pCond->push_back(std::auto_ptr<rcss::clang::Cond>(GetCMCondForSetPlayMode(theProblem->mode, curr_time)));
      } else {
      /* condition on our position.
	 This will hopefully prevent this rule from randomly firing later on */
	pCond->push_back(std::auto_ptr<rcss::clang::Cond>(new rcss::clang::CondPlayerPos(true, usetpasser, 1, 1,
											 std::auto_ptr<rcss::clang::Region>(new rcss::clang::RegArc(Circle(player_loc[pnsp->GetPlayer()],
																			   CoachParam::instance()->getSppPositionBuffer()))))));
      }
      
      rcss::clang::SimpleRule* rule = new rcss::clang::SimpleRule(std::auto_ptr<rcss::clang::Cond>(pCond));

      rcss::clang::ActPassUNum* act = new rcss::clang::ActPassUNum(usetreceiver);

      std::list<rcss::clang::Action*> listact(1, act);
      rcss::clang::DirComm* dir =
	new rcss::clang::DirComm(true, true, usetpasser, listact);
      
      rule->push_back(std::auto_ptr<rcss::clang::Dir>(dir));

      addStandardLanguageRule(this_set_play_prefix, rule, pQueue);

      next_idx = idx + 1;
    } break;

    case PN_StartGoto: {
      if (last_ball_owner > 0) {
	/* we only want to do this if it's not the first pass */
	actionlog(150) << "Converting a StartGoto node" << ende;
	StartGotoPN* pnsg = (StartGotoPN*)thePlan->nodes[idx];

	rcss::clang::UNumSet uset;
	uset.add(rcss::clang::UNum(pnsg->GetPlayer()));
	rcss::clang::UNumSet usetpasser;
	usetpasser.add(rcss::clang::UNum(last_ball_owner));

	rcss::clang::CondAnd* pCond = new rcss::clang::CondAnd();

	pCond->push_back(std::auto_ptr<rcss::clang::Cond>(new rcss::clang::CondBallOwner(true, usetpasser)));
	pCond->push_back(std::auto_ptr<rcss::clang::Cond>(new rcss::clang::CondPlayerPos(true, usetpasser, 1, 1,
							std::auto_ptr<rcss::clang::Region>(new rcss::clang::RegArc(Circle(player_loc[last_ball_owner],
															  CoachParam::instance()->getSppPositionBuffer()))))));
	pCond->push_back(std::auto_ptr<rcss::clang::Cond>(new rcss::clang::CondPlayMode(rcss::clang::PlayOn)));
      
	rcss::clang::SimpleRule* rule = new rcss::clang::SimpleRule(std::auto_ptr<rcss::clang::Cond>(pCond));

	rcss::clang::ActPos* act = 
	  new rcss::clang::ActPos(std::auto_ptr<rcss::clang::Region>(new rcss::clang::RegPoint(std::auto_ptr<rcss::clang::Point>(new rcss::clang::PointSimple(pnsg->GetToPos().convertToVector2D())))));

	std::list<rcss::clang::Action*> listact(1, act);
	rcss::clang::DirComm* dir =
	  new rcss::clang::DirComm(true, true, uset, listact);
  
	rule->push_back(std::auto_ptr<rcss::clang::Dir>(dir));

	addStandardLanguageRule(this_set_play_prefix, rule, pQueue);
      } else {
	actionlog(150) << "Converting a StartGoto node, first one, ignoring" << ende;
      }
      
      next_idx = idx + 1;
    } break;

    case PN_SendBall: {
      actionlog(150) << "Converting a SendBall node" << ende;
      SendBallPN* pnsb = (SendBallPN*)thePlan->nodes[idx];
      rcss::clang::CondAnd* pCond = new rcss::clang::CondAnd;
      
      /* condition on our position.
	 This will hopefully prevent this rule from randomly firing later on */
      rcss::clang::UNumSet uset;
      uset.add(rcss::clang::UNum(pnsb->GetPlayer()));
      pCond->push_back(std::auto_ptr<rcss::clang::Cond>(new rcss::clang::CondPlayerPos(true, uset, 1, 1,
										       std::auto_ptr<rcss::clang::Region>(new rcss::clang::RegArc(Circle(player_loc[pnsb->GetPlayer()],
																			 CoachParam::instance()->getSppPositionBuffer()))))));
      rcss::clang::SimpleRule* rule = new rcss::clang::SimpleRule(std::auto_ptr<rcss::clang::Cond>(pCond));

      rcss::clang::ActClear* act = 
	new rcss::clang::ActClear(std::auto_ptr<rcss::clang::Region>(new rcss::clang::RegArc(Circle(player_loc[pnsb->GetPlayer()], 10))));

      std::list<rcss::clang::Action*> listact(1, act);
      rcss::clang::DirComm* dir =
	new rcss::clang::DirComm(true, true, uset, listact);
  
      rule->push_back(std::auto_ptr<rcss::clang::Dir>(dir));

      addStandardLanguageRule(this_set_play_prefix, rule, pQueue);

      next_idx = idx + 1;
    } break;

    case PN_EndPass: {
      actionlog(150) << "Converting an EndPass node" << ende;
      StartGotoPN* pnsg = (StartGotoPN*)thePlan->nodes[idx-1];
      last_ball_owner = pnsg->GetPlayer();
      player_loc[last_ball_owner] = pnsg->GetToPos();
      next_idx = idx + 1;
    } break;
    
    case PN_EndGoto: {
      actionlog(150) << "Converting an EndGoto node" << ende;
      StartGotoPN* pnsg = (StartGotoPN*)thePlan->nodes[idx-1];
      player_loc[pnsg->GetPlayer()] = pnsg->GetToPos();
      next_idx = idx + 1;
    } break;
      
    case PN_StartDribble:
    case PN_EndDribble:
    case PN_Choose:
    case PN_AnyOf:
      errorlog << "Plan::ConvertToStandardLanguage: unsupported node type["
	       << idx << "] = " << thePlan->nodes[idx]->GetType() << ende;
      error = true;
      break;
      
    default:
      errorlog << "Plan::ConvertToStandardLanguage: what is node type type["
	       << idx << "] = " << thePlan->nodes[idx]->GetType() << ende;
      
      error = true;
      break;
    }
  }

  delete pFirstPasserIPCond;
  
  rcss::clang::RuleIDList idlist;

  //#define USE_CASPIAN_HACK
  // It didn't help
#ifdef USE_CASPIAN_HACK
  // This is the old way, where we turned all the rules on indiv. Now we'll
   // do it with the nested rule above.
   //pQueue->getRuleContainer().push(new rcss::clang::ActivateRules(true, setplay_rule_id_list));

  // The caspian hack spits that further
  splitIndividualNamesTo( rcss::clang::ActivateRules(on, setplay_rule_id_list),
			  pQueue->getRuleContainer());

#else	
  //Now turn on all these rules
  rcss::clang::Cond* pTopCond =
    new rcss::clang::CondTime(curr_time +
 			      this_set_play_setup_time +
 			      CoachParam::instance()->getSppTimeLimit(),
 			      rcss::util::CompOp::less());
  rcss::clang::NestedRule* pTopRule =
    new rcss::clang::NestedRule(std::auto_ptr<rcss::clang::Cond>(pTopCond));
  pTopRule->getRules().push_back( new rcss::clang::IDListRule(setplay_rule_id_list));
  rcss::clang::DefRule* pTopDef =
    new rcss::clang::DefRule(this_set_play_prefix,
 			     std::auto_ptr<rcss::clang::Rule>(pTopRule),
 			     false);
  pQueue->getDefineContainer().push(pTopDef);
 
  idlist.clear();
  idlist.push_back(this_set_play_prefix);
  pQueue->getRuleContainer().push(new rcss::clang::ActivateRules(true, idlist));
  setplay_rule_id_list.push_back(this_set_play_prefix);
#endif
  

  //Now turn off that all players stop rule
  idlist.clear();
  idlist.push_front(NAME_STOP_RULE);
  pQueue->getRuleContainer().push(new rcss::clang::ActivateRules(false, idlist));

  return !error;
}

bool
PlanExecution::StandardLanguageDeleteAll(CoachMessageQueue* pQueue)
{
  //oddly enough, an "empty" RuleIDList means a list of ALL rules.
  // Therefore, we check for a;; here.
  if (setplay_rule_id_list.all())
    return false;

  rcss::clang::RuleIDList* p = new rcss::clang::RuleIDList(setplay_rule_id_list);
  pQueue->getDelContainer().push(p);
  setplay_rule_id_list.clear();
  return true;
}

