/* -*- Mode: C++ -*- */

/* The class which represents a plan */

#ifndef _PLAN_H_	
#define _PLAN_H_ 

#include <stream.h>
#include "PlanNode.h"
#include "soccer_utils.h"
#include "misc.h"

enum coordinate_frame_t {
  CF_Absolute = 0,
  CF_BallRelative = 1
};
const int num_coordinate_frame = 2;
const int bits_coordinate_frame = 1;
#define COORDINATE_FRAME_STRINGS {"Absolute", "BallRelative"}


const int Plan_max_compact_len = 1024;
const int Plan_max_num_nodes = 40;
const int Plan_bits_for_num_nodes = 6;

/* The Plan here is a modified Simple Temporal Network */
/* Choose/Anyof is *NOT* implemented!	
   We have added "Choose" edges which allow a choice at execution time
   Corresponding with that, we have added "AnyOf" edges to gather up
   the branched caused by choose */
/* the graph is represented with two data strucutre:
   an array of Node* specifying the type and extra data of each node
   an adjancency matrix which specifies the underlying distance graph of 
   the STN */
/* The InitialPosPN node *must* be the first node */
/* The Kicker *must* start in node 1 */
/* For each player, the order in which their nodes appear in the array should be
   the order in which the nodes should be executed */
/* The corresponding node to an end must immediately precede it in the array */
/* A StartGoto can be ended by an EndPass. All passes should look like
   StartPass StartGoto EndPass
    fromP      toP     
     loc       loc
*/

const int bits_for_edge_length = 9; 
const int min_val_for_edge_length = -256;
const int edge_infty = 255;

class PlanExecution; //defined in PlanExecution.h

class Plan 
{
  friend class PlanExecution;
  
public:
  Plan();
  ~Plan();

  void Clear();  

  void Print(ostream& out);
  void PrintOnlyEdgeLens(ostream& out);
  bool Read(istream& in);

#ifdef NO_ACTION_LOG
  void Log(int level) {}
#else
  void Log(int level);
#endif //NO_ACTION_LOG

  //NOTE: we leave the char one before the return as an allocated char
  char* PrintStrCompact()
  {PrintStrCompact(compact_str+1); return compact_str+1; }
  void PrintStrCompact(char*);

  bool ReadStrCompact(char*);
  bool PartialReadInit()
  { bs.NewString(); Clear(); return true; }
  bool PartialRead(char* instr)
  { return bs.LoadPartial(instr); }
  bool PartialReadComplete();

  void ToAbsoluteFrame(VecPosition ball_pos);  
  void ToRelativeFrame(VecPosition ball_pos);  
  void FlipCoords(bool flipX = true, bool flipY = true); 
  void FlipCoordsX() { FlipCoords(true, false); } 
  void FlipCoordsY() { FlipCoords(false, true); } 

  FuzzyBool IsPlayerInvolved(Unum num, int node_num = 0);  
  void ReplacePlayerNums(Unum* arrRep);  

  void ComputeAllPairsDistances();  
  bool CheckConsistency();  
  //returns false if inconsistent
  bool MakeMinimalDispatchable();
  void FilterEdges();  

  void copyFrom(Plan* p);
  void copyTo(Plan* p) { p->copyFrom(this); }

  coordinate_frame_t GetCoordFrame() { return coord_frame; }
  int GetNumNodes() { return num_nodes; }
  PlanNode* GetPlanNode(int idx) { return nodes[idx]; }
  int GetEdgeLen(int i, int j) { return edge_len[i][j]; }
  InitialPosPN* GetInitPos()
  { if (!this) return NULL; else return (InitialPosPN*)nodes[0]; }

  /* These are used in creating the plan */
  void SetPlanNode(int idx, PlanNode* pn);  
  void SetEdgeLen(int i, int j, int len);
  void SetEdgeLowerBound(int i, int j, int len)
  { SetEdgeLen(j,i,-len); }
  void SetEdgeUpperBound(int i, int j, int len)
  { SetEdgeLen(i,j,len); }

protected:

  bool read_compact_helper();  

  bool DoesEdgeDominate(int e1s, int e1d, int e2s, int e2d);  

  char compact_str[Plan_max_compact_len+1];  
  
  //SMURF: do I need a mutex protecting bs?
  static BinaryString bs;
  static bool bsInit;  

  coordinate_frame_t coord_frame;  
  PlanNode* nodes[Plan_max_num_nodes];
  int num_nodes;  
  short int edge_len[Plan_max_num_nodes][Plan_max_num_nodes];


    
};

#endif 
