/* -*- Mode: C++ -*- */

/* The class which represents a plan */

/* the form for the plan is currently undecided */

#include <stdio.h>
#include <string.h>
#include <iomanip.h>
#include "Plan.h"
#include "ServerParam.h"
#include "CoachParam.h"
#include "soccer_utils.h"
#include "misc.h"
#include "utility.h"
#include "Logger.h"
using namespace spades;


const char* coordinate_frame_strings[num_coordinate_frame] = COORDINATE_FRAME_STRINGS;
const int linesize = 150;

bool Plan::bsInit = false;
BinaryString Plan::bs;


/****************************************************************************/
bool set_coord_frame(char* name, coordinate_frame_t* pCF)
{
  for (int i=0; i<num_coordinate_frame; i++) {
    if (strcasecmp(coordinate_frame_strings[i], name) == 0) {
      *pCF = (coordinate_frame_t)i;
      return true;
    }
  }
  return false;  
}

int add_with_infty(int x, int y)
{
  if (x == edge_infty || y == edge_infty)
    return edge_infty;
  else return Min(edge_infty, x+y);  
}


/****************************************************************************/

Plan::Plan()
{
  if (!bsInit) {
    bs.SetValidChars("");
    bsInit = true;
  }

  for (int i=0; i<Plan_max_num_nodes; i++)
    nodes[i] = NULL;  
  num_nodes = 0;  
  Clear();
}

Plan::~Plan()
{
  Clear();  
}


void Plan::Clear()
{
  coord_frame = CF_Absolute;  
  for (int i=0; i<num_nodes; i++) {
    delete nodes[i];  
    nodes[i] = NULL;    
  }  
  num_nodes = 0;  
  //init edge lens to infty
  for (int i=0; i<Plan_max_num_nodes; i++) {
    for (int j=0; j<Plan_max_num_nodes; j++) {
      edge_len[i][j] = edge_infty;      
    }
  }
  compact_str[0] = 0;
}


void Plan::Print(ostream& out)
{
  out << "coordinate frame: " 
      << coordinate_frame_strings[coord_frame] << endl;  
  out << "num_nodes: " << num_nodes << endl;  
  for (int i=0; i<num_nodes; i++) {
    if (nodes[i] == NULL) {
      errorlog << "Plan::Print: why is " << i << " null?" << ende;
      continue;      
    }
    nodes[i]->Print(out);    
    out << "\t\t# Node: " << i << endl;    
  }  
  PrintOnlyEdgeLens(out);  
}

void Plan::PrintOnlyEdgeLens(ostream& out)
{
  //edge lens
  for (int i=0; i<num_nodes; i++) {
    out << "#" << setw(4) << i;      
  }  
  out << endl;  
  for (int i=0; i<num_nodes; i++) {
    for (int j=0; j<num_nodes; j++) {
      out << setw(5) << edge_len[i][j];      
    }
    out << endl;    
  }
}


bool Plan::Read(istream& in)
{
  char name[plan_node_string_len];
  //read coordinate frame
  if (!skip_to_non_comment(in)) return false;
  if (!advance_to_on_line(in, ':')) return false;
  in.get(); //skip colon
  if (!(in >> name)) return false;
  if (!set_coord_frame(name, &coord_frame)) return false;  

  //read num_nodes
  if (!skip_to_non_comment(in)) return false;
  if (!advance_to_on_line(in, ':')) return false;
  in.get(); //skip colon
  if (!(in >> num_nodes)) return false;

  for (int i=0; i<num_nodes; i++) {
    bool seen_colon = false;    

    if (!skip_to_non_comment(in)) return false;
    if (!(in >> name)) return false;

    //if the colon is attached to the name, we'll get it now
    if (name[strlen(name)-1] == ':') {
      name[strlen(name)-1] = 0;    
      seen_colon = true;      
    }
    
    nodes[i] = GetNewPlanNodeOfType(name);
    if (nodes[i] == NULL) {
      errorlog << "Plan::Read: Could not read node " << i << ende;
      Clear();      
      return false;      
    }
    if (i==0 && nodes[0]->GetType() != PN_InitialPos)
      errorlog << "Node 0 of plan must be initial pos!" << ende;    
    if (seen_colon || advance_to_on_line(in, ':')) {
      //try to read data
      in.get(); //skip colon    
      if (!nodes[i]->ReadData(in)) {
	errorlog << "Plan Read: ReadData of node type " << name << "(" << i << ") failed" << ende;	
	return false;    
      }      
    }    
  }
  
  //edge lens
  for (int i=0; i<num_nodes; i++) {
    skip_to_non_comment(in);    
    for (int j=0; j<num_nodes; j++) {
      in >> edge_len[i][j];      
    }
  }
  
  return true;  
}

#ifndef NO_ACTION_LOG
void Plan::Log(int level)
{
  actionlog(level) << "Logging a Plan:" << ende;

  actionlog(level) << "coordinate frame: " << coordinate_frame_strings[coord_frame] << ende;
  actionlog(level) << "num_nodes: " << num_nodes << ende;
  
  for (int i=0; i<num_nodes; i++) {
    if (nodes[i] == NULL) {
      errorlog << "Plan::Log: why is " << i << " null?" << ende;
      continue;      
    }
    nodes[i]->Log(level, i);    
  }  

  //edge lens
  for (int i=0; i<num_nodes; i++) {
    actionlog(level) << std::setw(5) << i;
  }  
  actionlog(level) << ende;

  for (int i=0; i<num_nodes; i++) {
    for (int j=0; j<num_nodes; j++) {
      actionlog(level) << std::setw(5) << edge_len[i][j];
    }
    actionlog(level) << ende;
  }
}
#endif


void Plan::PrintStrCompact(char* str)
{
  errorlog << "I don't support printing to BinaryString right now!" << ende;
  if (!bsInit)
    errorlog << "Plan::PrintStrCompact: bs not initialized!" << ende;  
  bs.NewString();
  //coord frame
  if (!bs.PutInt(coord_frame, bits_coordinate_frame)) 
    errorlog << "What couldn't I put coordinate frame bits?" << ende;  
  //num_nodes
  if (!bs.PutInt(num_nodes, Plan_bits_for_num_nodes)) 
    errorlog << "What couldn't I put num_nodes bits?" << ende;  
  //the nodes
  for (int i=0; i<num_nodes; i++) {
    nodes[i]->PrintCompact(&bs);    
  }

  //edge lens
  for (int i=0; i<num_nodes; i++) {
    for (int j=0; j<num_nodes; j++) {
      if (i == j) 
	continue; //don't put diagnonal bits       
      if (edge_len[i][j] == edge_infty) {
	if (!bs.PutBit(0))
	  errorlog << "Couldn't put " << i << " " << j << " infty bit" << ende;	
      } else {
	if (!bs.PutBit(1))
	  errorlog << "Couldn't put " << i << " " << j << " not infty bit" << ende;	
	if (!bs.PutInt(edge_len[i][j]-min_val_for_edge_length, bits_for_edge_length))
	  errorlog << "Couldn't put " << i << " " << j << " edge len bits ("
		   << edge_len[i][j]-min_val_for_edge_length << ")" << ende;	
      }
    }
  }
  
  if (strlen(bs.CompleteString()) > (unsigned)Plan_max_compact_len) {
    errorlog << "Plan (in compact form) is too long!" << ende;
    str[0] = 0;    
  } else {
    strcpy(str, bs.CompleteString());  
  }
  
}

bool Plan::ReadStrCompact(char* instr)
{
  if (!bsInit)
    errorlog << "Plan::ReadStrCompact: bs not initialized!" << ende;  

  Clear(); //remove any current plan
  
  if (!bs.LoadString(instr)) {
    errorlog << "Plan::ReadStrCompact: BinaryString could not load message" << ende;    
    return false;  
  }  
  
  return read_compact_helper();  
}

bool Plan::PartialReadComplete() 
{ 
  if (!bsInit)
    errorlog << "Plan::PrintStrCompact: bs not initialized!" << ende;  

  if (!bs.LoadPartialComplete()) {
    errorlog << "Plan::PartialReadComplete: BinaryString could not load the string" << ende;
    return false;
  }
  
  return read_compact_helper();
}

/* binary string should already be loaded */
bool Plan::read_compact_helper()
{
  //coord frame
  coord_frame = (coordinate_frame_t)bs.GetInt(bits_coordinate_frame); 
  //num_nodes
  num_nodes = bs.GetInt(Plan_bits_for_num_nodes);
  //the nodes
  for (int i=0; i<num_nodes; i++) {
    plan_node_type_t pntype;
    pntype = (plan_node_type_t)bs.GetInt(bits_for_plan_node_type);
    if (pntype == -1) {
      errorlog << "Plan::read_compact_helper: ran out of string" << ende;
      Clear();      
      return false;      
    }
    nodes[i] = GetNewPlanNodeOfType(pntype);
    if (nodes[i] == NULL) {
      errorlog << "Plan::read_compact_helper: bad plan node type " << pntype << ende;
      Clear();      
      return false;      
    }
    if (i==0 && nodes[0]->GetType() != PN_InitialPos)
      errorlog << "Node 0 of plan must be initial pos!" << ende;    
    if (!nodes[i]->ReadDataCompact(&bs)) {
      errorlog << "Plan::read_compact_helper: failed reading data node " << i
	       << " of type " << pntype << ende;
      Clear();      
      return false;      
    }
  }

  //edge lens
  for (int i=0; i<num_nodes; i++) {
    for (int j=0; j<num_nodes; j++) {
      if (i == j) { // diagonal is not actually in the message	
	edge_len[i][j] = 0;
	continue; 
      }
      if (bs.GetBit()) {
	edge_len[i][j] = bs.GetInt(bits_for_edge_length)+min_val_for_edge_length;
      } else {
	edge_len[i][j] = edge_infty;	
      }
      if (edge_len[i][j] == -1) {
	errorlog << "Plan::read_compact_helper: ran out of string on edge len" << ende;
	return false;	
      }      
    }
  }
  
  return true;  
}

void Plan::ToAbsoluteFrame(VecPosition ball_pos)
{
  if (coord_frame != CF_BallRelative) {
    errorlog << "Cannot change to Absolute Frame if not in BallRelative" << ende;
    return;    
  }

  for (int i=0; i<Plan_max_num_nodes; i++) 
    if (nodes[i]) 
      nodes[i]->AddToAllLocations(ball_pos);  

  coord_frame = CF_Absolute;
}

void Plan::ToRelativeFrame(VecPosition ball_pos)
{
  if (coord_frame != CF_Absolute) {
    errorlog << "Cannot change to BallRelative if not Absolute Frame" << ende;
    return;    
  }

  for (int i=0; i<Plan_max_num_nodes; i++) 
    if (nodes[i]) 
      nodes[i]->AddToAllLocations(-ball_pos);  

  coord_frame = CF_BallRelative;  
}


void Plan::FlipCoords(bool flipX, bool flipY)
{
  for (int i=0; i<Plan_max_num_nodes; i++) 
    if (nodes[i]) 
      nodes[i]->FlipCoords(flipX, flipY);  
}

FuzzyBool Plan::IsPlayerInvolved(Unum num, int node_num) 
{
  if (this == NULL || nodes[0] == NULL)
    return FB_Unknown;  

  if (node_num < 0 || node_num >= num_nodes) {
    errorlog << "Plan::IsPlayerInvolved: node num out of range! " << node_num << ende;
    return FB_Unknown;    
  }
  
  /*  if (nodes[0]->GetType() != PN_InitialPos)
      errorlog << "Node 0 of plan must be initial pos!" << ende;     */
  player_involved_t inv_ret;
  do {
    if (node_num < 0) {
      errorlog << "CheckPrevious went below 0!" << ende;
      return FB_Unknown;      
    }
    inv_ret = nodes[node_num]->IsPlayerInvolved(num);
    node_num--;    
  } while (inv_ret == PI_CheckPrevious);
  
  return (inv_ret == PI_Yes) ? FB_True : FB_False;  
}

void Plan::ReplacePlayerNums(Unum* arrRep)
{
  for (int i=0; i<num_nodes; i++) {
    if (nodes[i])
      nodes[i]->ReplacePlayerNums(arrRep);
  }
}


/****************************************************************************/
/* this is the Floyd-Warshall algorithm */
void Plan::ComputeAllPairsDistances()
{
  int i,j,k;  
  for (i=0; i<num_nodes; i++) 
    edge_len[i][i] = 0;
  for (k=0; k<num_nodes; k++) {
    for (i=0; i<num_nodes; i++) {
      for (j=0; j<num_nodes; j++) {
	if (edge_len[i][k] == edge_infty || edge_len[k][j] == edge_infty)
	  continue; //edge i->j won't be changed
	edge_len[i][j] = Min(edge_len[i][j], edge_len[i][k]+edge_len[k][j]);    
      }
    }    
  }
}

//we rely on the fact that this actually sets the edge lens
bool Plan::CheckConsistency()
{
  ComputeAllPairsDistances();
  for (int i=0; i<num_nodes; i++) 
    if (edge_len[i][i] < 0)
      return false;
  return true;  
}

bool Plan::MakeMinimalDispatchable()
{
  //this also creates the all pairs graph
  if (!CheckConsistency())
    return false;

  FilterEdges();  
  
  //now we'll reset the diagnonals to infty
  for (int i=0; i<num_nodes; i++)
    edge_len[i][i] = edge_infty;  
  
  return true;  
}

/* this is the algorithm given by Muscettola, Morris, and Tsamardinos in
  Reformulating Temporal Plans for Efficient Execution (KR98)
  They later gave another algorithm which has faster asymptotic time, but our
  networks are so small that I don't think it matters */
void Plan::FilterEdges()
{
  int tri[3]; //the vertics of the triangle we are current considering
  bool edge_marked[Plan_max_num_nodes][Plan_max_num_nodes];  

  //clear marked edges
  for (int i=0; i<num_nodes; i++)
    for (int j=0; j<num_nodes; j++)
      edge_marked[i][j] = false;  

  //loop through triangles and mark edges
  for (tri[0]=0; tri[0]<num_nodes; tri[0]++) {
    for (tri[1]=tri[0]+1; tri[1]<num_nodes; tri[1]++) {
      for (tri[2]=tri[1]+1; tri[2]<num_nodes; tri[2]++) {
	//now we need to consider all intersecting pairs of edges in this tri
	for (int i=0; i<6; i++) {
	  int e1s = tri[i / 2], e1d = tri[(6-i-1) % 3];
	  int e2s = tri[(e1d + 2) % 3], e2d = tri[(e1s + 2) % 3];
	  if (DoesEdgeDominate(e1s, e1d, e2s, e2d)) {
	    if (DoesEdgeDominate(e2s, e2d, e1s, e1d)) {
	      //if neither is marked, pick one and mark it
	      if (edge_marked[e1s][e1d] || edge_marked[e2s][e2d])
		continue; //something already marked

	      //we can mark either, so we'll mark the first so it's deterministic
	      edge_marked[e1s][e1d] = true;	    
	      /*
	      if (int_random(2) == 0) 
		edge_marked[e1s][e1d] = true;	    
	      else
		edge_marked[e2s][e2d] = true;
	      */
	    } else {
	      edge_marked[e2s][e2d] = true;
	    }	    
	  } else if (DoesEdgeDominate(	e2s, e2d, e1s, e1d)) {
	    edge_marked[e1s][e1d] = true;	    
	  }
	} //pairs in the triangle	
      }
    }
  }  

  //remove the marked edges
  for (int i=0; i<num_nodes; i++)
    for (int j=0; j<num_nodes; j++)
      if (edge_marked[i][j])
	edge_len[i][j] = edge_infty;  
}

//this is domination as defined by Muscettola, Morris, and Tsamardinos
//does e1 dominate e2
bool Plan::DoesEdgeDominate(int e1s, int e1d, int e2s, int e2d)
{
  if (edge_len[e1s][e1d] == edge_infty || edge_len[e2s][e2d] == edge_infty)
    return false; //both edges must exist for domination
  
  if (edge_len[e1s][e1d] >= 0) {
    if (edge_len[e2s][e2d] < 0)
      return false; //must both be positive or both negative
    if (e1d != e2d)
      return false; //non-neg edges must have same dest
    if (add_with_infty(edge_len[e2s][e1s],edge_len[e1s][e1d]) == edge_len[e2s][e2d])
      return true;
    else
      return false;    
  } else {
    if (edge_len[e2s][e2d] >= 0)
      return false; //must both be positive or both negative
    if (e1s != e2s)
      return false; //neg edges must have same source
    if (add_with_infty(edge_len[e1s][e1d],edge_len[e1d][e2d]) == edge_len[e2s][e2d])
      return true;
    else
      return false;    
  }    
}




/****************************************************************************/

void Plan::copyFrom(Plan* p)
{
  if (this == NULL || p == NULL)
    return;
  
  Clear(); //free our mem
  
  coord_frame = p->coord_frame;  
  num_nodes = p->num_nodes;
  for (int i=0; i<Plan_max_num_nodes; i++) 
    if (p->nodes[i]) nodes[i] = p->nodes[i]->makeCopy();
  
  strcpy(compact_str, p->compact_str);
  
  for (int i=0; i<num_nodes; i++) {
    for (int j=0; j<num_nodes; j++) {
      edge_len[i][j] = p->edge_len[i][j];      
    }
  }
}


/****************************************************************************/

void Plan::SetEdgeLen(int i, int j, int len)
{
  if (nodes[i] == NULL || nodes[j] == NULL) {
    errorlog << "Setting edge len for a node that does not exist " << i << " " << j << ende;
  }

  edge_len[i][j] = Min(len, edge_infty);  
}

void Plan::SetPlanNode(int idx, PlanNode* pn)
{
  if (idx < 0 || idx >= Plan_max_num_nodes) {
    errorlog << "SetPlanNode: idx " << idx << " out of range" << ende;
    return;    
  }
  
  if (nodes[idx])
    delete nodes[idx];
  
  nodes[idx] = pn;  

  if (idx >= num_nodes)
    num_nodes = idx+1;  
}

