/* -*- Mode: C++ -*- */

/* This file contains base classes and a threaded ControllerModule (see Planner.h)
   for use with the PlannerController */

#ifndef _FOUR_LEVEL_PLANNER_H_	
#define _FOUR_LEVEL_PLANNER_H_ 

#include "Planner.h"
#include "Plan.h"
#include "Geometry.h"
#include "PlayerDistribution.h"

/* Level 1: Waypoint planner
   INPUTS: the problem, the current opponent locations
   OUTPUT: List of waypoints */
/* Level 2: Role maker
   INPUT: List of waypoints, current opponent locations
   OUTPUT: Agent and ball movements, considering offsides */
/* Level 3: STN compilation
   INPUT: Agent and ball movements
   OUTPUT: STN plan (see Plan.h) */
/* Level 4: Agent Instantiation (done on the agents, not the coach)
   INPUT: STN Plan
   OUTPUT STN Plan with agents assigned the the temporary numbers in the previous plan */

/* what follows are virtual base classes for the levels */


/* this is for the time_left variables which are passed around 
   (actually to which pointers are passed around) */
//more time left shoudl always be a higher value
enum time_left_t {
  TL_Normal = 10,
  TL_MidWay1 = 1,
  TL_Hurry = 0,
  TL_Abort = -1
} ;


/************************* Supporting Data Structures **************/
const int MAX_WAYPOINTS = 10;
class WaypointList 
{
public:
  WaypointList();

  bool SetWaypoint(int idx, VecPosition v, PlayerDistribution* pd);
  //Don't forget to set the PlayerDistribution if you use this next version!
  bool SetWaypoint(int idx, VecPosition v);
  
  VecPosition GetPt(int idx) const ;
  PlayerDistribution* GetPlayDist(int idx);
  
  void Clear();

  int GetNumPts() const { return num_pts; }

  void draw(FieldImage* pimg);
  void draw(const std::string& fn, int scale);
  
#ifdef NO_ACTION_LOG
  void Log(int level) {}
#else
  void Log(int level);
#endif //NO_ACTION_LOG
  
private:
  int num_pts;
  VecPosition pts[MAX_WAYPOINTS];
  PlayerDistribution play_dist[MAX_WAYPOINTS];

public:
  static FieldImage::Color COLOR_PASS;
  static FieldImage::Color COLOR_CLEAR;
  
} ;

enum ABM_order_t {
  ABMO_NoValue,
  ABMO_NoConstraint,
  ABMO_AfterLastLastEnd,
  ABMO_AfterLastStart,
  ABMO_AfterLastEnd,
  ABMO_AfterThisStart //only valid for a receiver of a pass!
} ;

//This is a list of player and ball locations
const int MAX_LOCATIONS = 100;
class AgentBallMovements
{
public:
  static const int ball_num;
  static const int no_num;

  AgentBallMovements();

  void Clear();
  void SetPlayerLoc(int idx, int num, VecPosition v, ABM_order_t relorder);
  void SetBallLoc(int idx, VecPosition v);

  void AddPlayerLoc(int num, VecPosition v, ABM_order_t relorder)
    { SetPlayerLoc(num_loc, num, v, relorder); }
  void AddBallLoc(VecPosition v)
    { SetBallLoc(num_loc, v); }
  
  int GetObjectNum(int idx) const;
  VecPosition GetObjectLoc(int idx) const;
  ABM_order_t GetObjectOrder(int idx) const;

  int GetNumLoc() const { return num_loc; }

#ifdef NO_ACTION_LOG
  void Log(int level) {}
#else
  void Log(int level);
#endif //NO_ACTION_LOG
  
private:
  struct agent_ball_loc_t {
    int num; //0 is the ball
    VecPosition loc;    
    //relation of this position to previous ball movement
    ABM_order_t relorder; 
  } ;  

  int num_loc;
  agent_ball_loc_t obj[MAX_LOCATIONS];
  
} ;


/************************* Level -: Getting Initial Position **************/
/* this isn't really a level, but in order to make this usable for
   both coach and player we need to have a function which compilies the
   current position into a PlayerDistribution */
class GetPlayerDistribution
{
public:
  virtual void SetPositions(PlayerDistribution* pd) = 0;  
} ;


/************************* Level 1: Waypoint ****************************/


class Waypoint 
{
public:
  Waypoint() { pTimeLeft = NULL; }
  virtual ~Waypoint() {}

  virtual bool run(Problem* theProb, PlayerDistribution* pDist, 
		   GetPlayerDistribution* getPD, WaypointList* output) = 0;

  void SetTimeLeftVar(const int* p) { pTimeLeft = p; }  

protected:
  const int* pTimeLeft;
  
} ;

/************************* Level 2: RoleMaker ****************************/

class RoleMaker
{
public:
  RoleMaker() { pTimeLeft = NULL; }
  virtual ~RoleMaker() {}
  
  virtual bool run(WaypointList* pWaypoints, Problem* theProb, 
		   AgentBallMovements* output) = 0;  

  void SetTimeLeftVar(const int* p) { pTimeLeft = p; }  

protected:
  const int* pTimeLeft;
  
} ;

/************************* Level 3: STNCompilation ****************************/
class STNCompilation
{
public:
  STNCompilation() { pTimeLeft = NULL; }
  virtual ~STNCompilation() {}

  virtual bool run (AgentBallMovements* pMoves, Problem* theProb, Plan* output) = 0;

  void SetTimeLeftVar(const int* p) { pTimeLeft = p; }  

protected:
  const int* pTimeLeft;
  
} ;

/************************* Level 4: AgentInstantiation ****************************/
class AgentInstantiation
{
public:
  AgentInstantiation() { pTimeLeft = NULL; }
  virtual ~AgentInstantiation() {}

  //Modifies the plan given to it!
  //Returns NULL on error
  virtual bool run (Plan* thePlan, Problem* theProb) = 0;

  void SetTimeLeftVar(const int* p) { pTimeLeft = p; }  

protected:
  const int* pTimeLeft;
  
} ;


/***************************************************************************/
/****************  The controller function to use the levels ***************/
/***************************************************************************/
class FourLevelPlanner : public Planner
{
public:
  FourLevelPlanner();
  ~FourLevelPlanner();  
  
  planner_return_t run(Problem* pTheProblem, Plan* pThePlan);

  //max number of cycles had for computation, and the buffer for how long beforehand
  //timeLeft should be adjusted
  void setTimeLimit(int lim, int buffer)
    { timeLimit = lim; timeLimitBuffer = buffer; }

  void setGetPD(GetPlayerDistribution* module);

  void setWaypoint(Waypoint* module);
  void setRoleMaker(RoleMaker* module);
  void setSTNCompilation(STNCompilation* module);
  void setAgentInstantiation(AgentInstantiation* module);

  void* asynch_worker();

private:
  enum thrd_status_t {
    TS_Initializing,
    TS_WaitingForData,
    TS_SettingUp,
    TS_RunningWaypoint,
    TS_RunningRoleMaker,
    TS_RunningSTNCompilation,
    TS_RunningAgentInstantiation,
    TS_Complete,
    TS_Error
  } ;

  int timeLimit;
  int timeLimitBuffer;
  int cyclesUsed;  

  GetPlayerDistribution* pGetPD;

  Waypoint* pWaypoint;
  RoleMaker* pRoleMaker;
  STNCompilation* pSTNCompilation;
  AgentInstantiation* pAgentInstantiation;  
  

  /* we don't need a mutex around time_left because the main thread is the only
     one who can change it */
  int timeLeft;
  
  pthread_mutex_t mutex_input;
  Problem* pCurrProb;
  PlayerDistribution* pCurrPlayDist;  

  //the worker thread writes to this. The main thread should then copy it out
  pthread_mutex_t mutex_outputPlan;
  Plan* pOutputPlan;
  
  pthread_mutex_t mutex_thrdStatus;
  int thrdStatus; //indicates where in processing it is
   

};



#endif //_FOUR_LEVEL_PLANNER_H
