/* -*- Mode: C++ -*- */

/* This file contains the implementation for the FourLevelPlanner.
   See the .h for more description */

#include <iomanip>
#include "FourLevelPlanner.h"
#include "soccer_utils.h"
#include "CoachParam.h"
#include "utility.h"
#include "Logger.h"
using namespace spades;

/************************* WaypointList ****************************/
FieldImage::Color WaypointList::COLOR_PASS = FieldImage::COLOR_RED;
FieldImage::Color WaypointList::COLOR_CLEAR = FieldImage::COLOR_GREEN;

WaypointList::WaypointList()
{
  Clear();  
}


VecPosition WaypointList::GetPt(int idx) const
{
  if (idx >= num_pts || idx < 0) {
    errorlog << "GetPt: idx out of range: " << idx << ende;
    return 0;    
  }
  
  return pts[idx];  
}

bool WaypointList::SetWaypoint(int idx, VecPosition v)
{
  if (idx >= MAX_WAYPOINTS || idx < 0) {
    errorlog << "GetPt: idx out of range: " << idx << ende;
    return false;    
  }
  
  pts[idx] = v;

  if (idx >= num_pts) 
    num_pts = idx+1;

  return true;
}


bool WaypointList::SetWaypoint(int idx, VecPosition v, PlayerDistribution* pd)
{
  if (idx >= MAX_WAYPOINTS || idx < 0) {
    errorlog << "GetPt: idx out of range: " << idx << ende;
    return false;    
  }
  if (pd == NULL) {
    errorlog << "Don't set waypoint " << idx << " with a null PlayerDistribution!" << ende;
    return false;
  }

  pts[idx] = v;
  play_dist[idx].copyFrom(pd);

  if (idx >= num_pts) 
    num_pts = idx+1;

  return true;  
}

PlayerDistribution* WaypointList::GetPlayDist(int idx) 
{
  if (idx >= MAX_WAYPOINTS || idx < 0) {
    errorlog << "GetPlayDist: idx out of range: " << idx << ende;
    return NULL;    
  }

  return &play_dist[idx];
}

  
void WaypointList::Clear()
{
  for (int i = 0; i< MAX_WAYPOINTS; i++) {
    pts[i] = VecPosition(0,0);
    play_dist[i].clear();
  }
  num_pts = 0;
}

void
WaypointList::draw(FieldImage* pimg)
{
  VecPosition curr_pt = pts[0];
  for (int i = 0; i < num_pts; i++)
    {
      FieldImage::Color c;
      if (curr_pt.getDistanceTo(pts[i]) > CoachParam::instance()->getSppMaxPassDist())
	c = COLOR_CLEAR;
      else
	c = COLOR_PASS;
      pimg->addArrow(curr_pt, pts[i], c, 2);
      curr_pt = pts[i];
    }
}

void
WaypointList::draw(const std::string& fn, int scale)
{
  FieldImage img(0, scale);
  img.addFieldLines();
  draw(&img);
  if (!img.writeTo(fn.c_str()))
    errorlog << "WaypointList::draw: Could not write file '" << fn << "'" << ende;
}


#ifndef NO_ACTION_LOG
void WaypointList::Log(int level)
{
  if (level > Logger::instance()->getMaxActionLogLevel())
    return;
  
  char outstring[200];
  char* pc = outstring;
  
  for (int i=0; i < num_pts; i++) {
    std::ostrstream o(pc,100);
    o << pts[i] << ends;
    pc = &pc[strlen(pc)];
    *(pc+1)= 0;
    *pc = ' ';
    pc++;
  }
  
  actionlog(level) << "Waypoints: " << outstring << ende;  
}
#endif

/************************* AgentBallMovements ****************************/
/* some static consts */
const int AgentBallMovements::ball_num = -1;
const int AgentBallMovements::no_num = 0;

AgentBallMovements::AgentBallMovements()
{
  Clear();
}

void AgentBallMovements::Clear()
{
  for (int i=0; i<MAX_LOCATIONS; i++) {
    obj[i].num = no_num;
    obj[i].loc = 0;    
  }
  num_loc = 0;  
}

void AgentBallMovements::SetPlayerLoc(int idx, int num, VecPosition v, 
				      ABM_order_t relorder)
{
  obj[idx].num = num;
  obj[idx].loc = v;
  if (relorder == ABMO_NoValue) 
    errorlog << "Player should not get a ABMO_NoValue order" << ende;
  obj[idx].relorder = relorder;

  if (idx >= num_loc) 
    num_loc = idx+1;
}

void AgentBallMovements::SetBallLoc(int idx, VecPosition v)
{
  obj[idx].num = ball_num;
  obj[idx].loc = v;
  obj[idx].relorder = ABMO_NoValue;

  if (idx >= num_loc) 
    num_loc = idx+1;
}

  
int AgentBallMovements::GetObjectNum(int idx) const
{
  if (idx < 0 || idx >= num_loc) {
    errorlog << "GetObjectNum: idx out of range " << idx << ende;
    return no_num;    
  }
  return obj[idx].num;  
}

VecPosition AgentBallMovements::GetObjectLoc(int idx) const
{
  if (idx < 0 || idx >= num_loc) {
    errorlog << "GetObjectNum: idx out of range " << idx << ende;
    return 0;    
  }
  return obj[idx].loc;  
}
ABM_order_t AgentBallMovements::GetObjectOrder(int idx) const
{
  if (idx < 0 || idx >= num_loc) {
    errorlog << "GetObjectAfterSP: idx out of range " << idx << ende;
    return ABMO_NoValue;    
  }
  if (obj[idx].num == ball_num) {
    errorlog << "GetObjectAfterSP: don't ask about ball " << idx << ende;
    return ABMO_NoValue;    
  }

  return obj[idx].relorder;
}



#ifndef NO_ACTION_LOG
void AgentBallMovements::Log(int level)
{
  if (level > Logger::instance()->getMaxActionLogLevel())
    return;
  
  for (int i=0; i < num_loc; i++) {
    //actionlog250) << "ABM::Log, pc %p ", (void*)pc << ende;    
    actionlog(level) << std::setw(4) << obj[i].num << obj[i].loc;
  }

  actionlog(level) << ende;  
}
#endif


/************************* STNCompilation ****************************/

/************************* FourLevelPlanner ****************************/
FourLevelPlanner::FourLevelPlanner()
  : Planner()
{
  int errcode;

  pGetPD = NULL;
  pWaypoint = NULL;
  pRoleMaker = NULL;
  pSTNCompilation = NULL;
  pAgentInstantiation = NULL;  

  timeLimit = 50;
  timeLimitBuffer = 5;
  cyclesUsed = 0;  
  
  if ((errcode=pthread_mutex_init(&mutex_input, NULL))) {
    errorlog << "4LP: mutex_init 1: " << strerror(errcode) << ende; 
    return;
  }
  if ((errcode=pthread_mutex_init(&mutex_outputPlan, NULL))) {
    errorlog << "4LP: mutex_init 2: " << strerror(errcode) << ende; 
    return;
  }
  if ((errcode=pthread_mutex_init(&mutex_thrdStatus, NULL))) {
    errorlog << "4LP: mutex_init 3: " << strerror(errcode) << ende; 
    return;
  }

  pCurrProb = new Problem;
  pOutputPlan = new Plan;  
  pCurrPlayDist = new PlayerDistribution;  

  thrdStatus = TS_Initializing;
  timeLeft = TL_Normal;  

  if ( (errcode=StartAsynchRun()) != 0 ) {
    errorlog << "FourLevelPlanner: Starting thread failed!" << ende;
  }
}

FourLevelPlanner::~FourLevelPlanner()
{
  pthread_mutex_destroy(&mutex_input);
  pthread_mutex_destroy(&mutex_outputPlan);
  pthread_mutex_destroy(&mutex_thrdStatus);  

  delete pWaypoint;
  delete pRoleMaker;
  delete pSTNCompilation;
  delete pAgentInstantiation;  
}

void FourLevelPlanner::setGetPD(GetPlayerDistribution* module)
{
  if (pGetPD)
    delete pGetPD;
  pGetPD = module;
}

void FourLevelPlanner::setWaypoint(Waypoint* module)
{
  if (pWaypoint)
    delete pWaypoint;
  module->SetTimeLeftVar(&timeLeft);  
  pWaypoint = module;
}

void FourLevelPlanner::setRoleMaker(RoleMaker* module)
{
  if (pRoleMaker)
    delete pRoleMaker;
  module->SetTimeLeftVar(&timeLeft);  
  pRoleMaker = module;
}

void FourLevelPlanner::setSTNCompilation(STNCompilation* module)
{
  if (pSTNCompilation)
    delete pSTNCompilation;
  module->SetTimeLeftVar(&timeLeft);  
  pSTNCompilation = module;
}

void FourLevelPlanner::setAgentInstantiation(AgentInstantiation* module)
{
  if (pAgentInstantiation)
    delete pAgentInstantiation;
  module->SetTimeLeftVar(&timeLeft);  
  pAgentInstantiation = module;
}


  
planner_return_t FourLevelPlanner::run(Problem* pTheProblem, Plan* pThePlan)
{
  planner_return_t retval;

  if (pGetPD == NULL ||
      pWaypoint == NULL ||
      pRoleMaker == NULL ||
      pSTNCompilation == NULL ||
      pAgentInstantiation == NULL) {
    errorlog << "4LP: one of my modules is NULL!" << ende;
    return PlnRet_Failure;    
  }  

  //this is just a way to short circuit the current computation 
  // useful (for example), if the ball gets dropped while we are planning
  if (pTheProblem == NULL) {
    if (thrdStatus != TS_WaitingForData) {
      actionlog(60) << "4LP: NULL problem, aborting computation" << ende;
      timeLeft = TL_Abort;
      cyclesUsed = 0; //for the next play
    }
    return PlnRet_Failure;
  }
  

  /* This is only for offensive setplays, so unless we're one of those, 
     return failure */
  if (!IsSetPlayOffensive(pTheProblem->mode)) {
    actionlog(60) << "4LP: I only handle offensive set plays " << pTheProblem->mode << ende;
    //See if there is something running which we need to abort
    if (thrdStatus != TS_WaitingForData) {
      actionlog(60) << "4LP: got called for defensive play, aborting offensive computation" << ende;
      timeLeft = TL_Abort;
      cyclesUsed = 0; //for the next offensive play
    }
    return PlnRet_Failure;
  }

  if (pTheProblem->fresh && 
      (thrdStatus == TS_Complete || thrdStatus == TS_Error)) {
    actionlog(60) << "4LP, fresh problem, old computation complete/error, so setting as ready" << ende;
    thrdStatus = TS_WaitingForData;
  }
  
  if (pTheProblem->fresh && thrdStatus != TS_WaitingForData) {
    /* we need to load the data, but the thread is running, abort and 
       hopefully next time we can send it the data */
    actionlog(60) << "4LP, Got a new problem to work on, aborting current computation" << ende;
    timeLeft = TL_Abort;
    if (cyclesUsed >= timeLimit) {
      //Abort the computation, and return failure
      actionlog(30) << "4LP: never started!" << ende;
      errorlog << "FourLevelPlanner did not start!!!" << ende;
      timeLeft = TL_Abort; 
      cyclesUsed = 0; //this abort needs to reset this for next planning sessions
      retval = PlnRet_Failure;
    } else if (cyclesUsed < 0) {
      actionlog(30) << "4LP: resetting cyclesUsed for new problem" << ende;
      cyclesUsed = 0;
      retval = PlnRet_NotReadyYet;
    } else {
      actionlog(30) << "4LP: still waiting for thread to abort" << ende;
      retval = PlnRet_NotReadyYet;
    }
    cyclesUsed++;
  } else {
  
    pthread_mutex_lock(&mutex_thrdStatus);
    switch (thrdStatus){
    case TS_Initializing:
      actionlog(60) << "4LP: Worker thread still initializing" << ende;    
      retval = PlnRet_NotReadyYet; //wait for worker to initialize
      break;
    
    case TS_WaitingForData:
      if (data_ready) {
	actionlog(60) << "4LP: Worker thread hasn't picked up data yet" << ende;    
	retval = PlnRet_NotReadyYet;      
      } else {
	actionlog(60) << "4LP: Feeding data to worker thread" << ende;    
	//Give the worker thread the data
	if (timeLeft != TL_Abort) {
	  actionlog(80) << "4LP: resetting cyclesUsed because last was not abort" << ende;
	  cyclesUsed = 0;
	}
	timeLeft = TL_Normal;      
	pthread_mutex_lock(&mutex_input);
	*pCurrProb = *pTheProblem;
	//Get the starting location
	pGetPD->SetPositions(pCurrPlayDist);
	pthread_mutex_unlock(&mutex_input);
      
	//wake up the worker thread
	pthread_mutex_lock(&mutex_data_ready);
	data_ready = true;    
	pthread_cond_signal(&cv_data_ready);
	pthread_mutex_unlock(&mutex_data_ready);
	retval = PlnRet_NotReadyYet;      
	pTheProblem->fresh = false;
      }
      break;
    
    case TS_SettingUp:
    case TS_RunningWaypoint:
    case TS_RunningRoleMaker:
    case TS_RunningSTNCompilation:
    case TS_RunningAgentInstantiation:
      actionlog(60) << "4LP: Still working on answer: " << thrdStatus << ende;    
      retval = PlnRet_NotReadyYet;
      break;    

    case TS_Complete:
      //Yay! We got an answer
      actionlog(60) << "4LP: Answer complete" << ende;    
      pthread_mutex_lock(&mutex_outputPlan);
      pThePlan->copyFrom(pOutputPlan);    
      pthread_mutex_unlock(&mutex_outputPlan);
      thrdStatus = TS_WaitingForData; //set the thread for the next bit of work    
      retval = PlnRet_Complete;
      break;

    case TS_Error:
      actionlog(60) << "4LP: Thread returned error" << ende;    
      thrdStatus = TS_WaitingForData; //set the thread for the next bit of work    
      retval = PlnRet_Failure;
      break;
    
    default:
      errorlog << "What is the thrdStatus? " << thrdStatus << ende;
      retval = PlnRet_Failure;    
      break;    
    } ;
    pthread_mutex_unlock(&mutex_thrdStatus);

  } //not a new problem
  

  //check if we need to adjust timeleft
  if (retval == PlnRet_NotReadyYet) {
    cyclesUsed++;
  
    if (cyclesUsed >= timeLimit) {
      //Abort the computation, and return failure
      actionlog(30) << "4LP: did not complete in the time limit!" << ende;
      errorlog << "FourLevelPlanner did not complete!!!" << ende;
      timeLeft = TL_Abort; 
      //we set this to -1 so that we know the previous problem was abort
      cyclesUsed = -1; //this abort needs to reset this for next planning sessions
      retval = PlnRet_Failure;
    } else if (cyclesUsed >= timeLimit - timeLimitBuffer) {
      actionlog(30) << "4LP: telling thread to hurry" << ende;
      timeLeft = TL_Hurry; //this should get the answer out quickly    
    } else if (cyclesUsed >= (timeLimit - timeLimitBuffer) * CoachParam::instance()->getSppFracToFocusBest()) {
      actionlog(30) << "4LP: telling thread that we're partway" << ende;
      timeLeft = TL_MidWay1; 
    }
    
  }
  
  return retval;  
}


void* FourLevelPlanner::asynch_worker()
{
  //cout << "Starting" << endl;

  Problem prob;
  PlayerDistribution playDist;
  WaypointList waypoints;
  AgentBallMovements moves;
  bool error;  
    
  actionlog(70) << "4LPworker: starting" << ende;
    
  /* The first time through is the only time that the worker changes it's status
     to TS_WaitingForData */
  pthread_mutex_lock(&mutex_thrdStatus);
  thrdStatus = TS_WaitingForData;    
  pthread_mutex_unlock(&mutex_thrdStatus);
    
  while(1) {
    error = false;
    
    actionlog(70) << "4LPworker: about to start waiting" << ende;

    pthread_mutex_lock(&mutex_data_ready);
    while (!data_ready) {
      pthread_cond_wait(&cv_data_ready, &mutex_data_ready);
    }
    pthread_mutex_unlock(&mutex_data_ready);
    
    //acquire data mutex
    pthread_mutex_lock(&mutex_input);
    //copy data
    prob = *pCurrProb;
    playDist.copyFrom(pCurrPlayDist);    
    //release data mutex
    pthread_mutex_unlock(&mutex_input);

    actionlog(70) << "4LPworker: Got Data" << ende;

    //Indicate that we are running *before* we actually set data_ready to false
    pthread_mutex_lock(&mutex_thrdStatus);
    thrdStatus = TS_SettingUp;    
    pthread_mutex_unlock(&mutex_thrdStatus);

    //indicate that more data could be put in now
    pthread_mutex_lock(&mutex_data_ready);
    data_ready = false;    
    pthread_mutex_unlock(&mutex_data_ready);

    /****** Here we do the real work *****/

    //Run the waypoint generator
    if (!error) {
      pthread_mutex_lock(&mutex_thrdStatus);
      thrdStatus = TS_RunningWaypoint;    
      pthread_mutex_unlock(&mutex_thrdStatus);
      error = !pWaypoint->run(&prob, &playDist, pGetPD, &waypoints);      
      actionlog(70) << "4LPworker: Completed generating waypoints , err=" <<  error << ende;
      waypoints.Log(70);
      if (CoachParam::instance()->getSppDrawPlannedWaypoints())
	waypoints.draw(CoachParam::instance()->getSppDrawPlannedWaypointsFStem() +
		       toString(prob.time),
		       CoachParam::instance()->getSppDrawPlannedWaypointsScale());
    }
    
    //Run the role/movement generator
    if (!error) {
      pthread_mutex_lock(&mutex_thrdStatus);
      thrdStatus = TS_RunningRoleMaker;    
      pthread_mutex_unlock(&mutex_thrdStatus);
      error = !pRoleMaker->run(&waypoints, &prob, &moves);      
      actionlog(70) << "4LPworker: Completed generating agent roles , err=" << error << ende;
      moves.Log(70);
    }
    
    //Run the STN compilation
    if (!error) {
      pthread_mutex_lock(&mutex_thrdStatus);
      thrdStatus = TS_RunningSTNCompilation;    
      pthread_mutex_unlock(&mutex_thrdStatus);
      //we have to aquire the output plan mutex- this shouldn't be a problem
      pthread_mutex_lock(&mutex_outputPlan);
      error = !pSTNCompilation->run(&moves, &prob, pOutputPlan);
      /* we release even though we to get it again in a sec to avoid a possible deadlock
	 with the thrdStatus mutex */	
      pthread_mutex_unlock(&mutex_outputPlan);
      actionlog(70) << "4LPworker: Completed STN compilation , err=" <<  error << ende;
    }
    
    //Run the Agent Instantiation
    if (!error) {
      pthread_mutex_lock(&mutex_thrdStatus);
      thrdStatus = TS_RunningAgentInstantiation;    
      pthread_mutex_unlock(&mutex_thrdStatus);
      pthread_mutex_lock(&mutex_outputPlan);
      error = !pAgentInstantiation->run(pOutputPlan, &prob);      	
      pthread_mutex_unlock(&mutex_outputPlan);
      actionlog(70) << "4LPworker: Completed Agent Instantiation , err=" << error << ende;
    }
    
    /****** Now the real work is over *****/
    if (!error) {
      //indicate that we finished
      pthread_mutex_lock(&mutex_thrdStatus);
      thrdStatus = TS_Complete;    
      pthread_mutex_unlock(&mutex_thrdStatus);
    } else {
      //indicate that an error happened
      pthread_mutex_lock(&mutex_thrdStatus);
      thrdStatus = TS_Error;    
      pthread_mutex_unlock(&mutex_thrdStatus);
    }
    actionlog(70) << "4LPworker: work completed , err=" << error << ende;
  }
}


/* Random code  */
#ifdef NEVER
    for (int i=0; i<10000; i++)
      for (int j=0; j<10000; j++)
	;
    
    ifstream infile("Plans/passdribpass");

    if (!infile)
      errorlog << "SMURF: could not open plan file" << ende;  

    pthread_mutex_lock(&mutex_outputPlan);
    if (!pOutputPlan->Read(infile))
      errorlog << "FixedPlanPlanner: couldn't read plan" << ende;
    pthread_mutex_unlock(&mutex_outputPlan);
#endif
