/* -*- Mode: C++ -*- */

/* This file contains the four modules for the planning architecture for the coach */

#ifndef _PLAN_MODULES_H_
#define _PLAN_MODULES_H_

#include <fstream.h>

#include "Monitor.h"
#include "Diagnosis.h"
#include "Planner.h"
#include "Executor.h"
#include "OppModelSet.h"
#include "DefOppModel.h"
#include "CoachMessageQueue.h"
#include "PlannerModules.h"
#include "Logger.h"

/* NOTE: all data structures that are passed around give coordinates like we
   are the left team. Each module is reposible for flipping things as needed */

/***********************************************************************************/
/*******************                Monitor                        *****************/
/***********************************************************************************/

//only watches for timeout
class TimeoutMonitor : public Monitor 
{
public:
  TimeoutMonitor(TeamSide my_side, const WorldHistory& history)
    : Monitor(), my_side(my_side), history(history) {};  

  monitor_return_t run(PlanExecution* plan_exec);

  void* asynch_worker() 
  { errorlog << "TimeoutMonitor: don't run me asynch!" << spades::ende; return NULL;}

private:
  TeamSide my_side;
  const WorldHistory& history;  
};

class OpponentPossessionMonitor : public Monitor 
{
public:
  OpponentPossessionMonitor(TeamSide my_side, const WorldHistory& history)
    : Monitor(), my_side(my_side), history(history) {};  

  monitor_return_t run(PlanExecution* plan_exec);

  void* asynch_worker() 
  { errorlog << "TimeoutMonitor: don't run me asynch!" << spades::ende; return NULL;}
private:
  TeamSide my_side;
  const WorldHistory& history;  
};

/* Note that this class will *not* delete the pDefOM that you give it */
class DefensiveModelMonitor : public Monitor 
{
public:
  DefensiveModelMonitor(TeamSide my_side, const WorldHistory& history)
    : Monitor(), my_side(my_side), history(history) { 
    pDefOM = NULL; 
    addInfo = true;
  };  
  //DEB: make this next pointer the right type
  DefensiveModelMonitor(TeamSide my_side, const WorldHistory& history, DefOppModel* p)
    : Monitor(), my_side(my_side), history(history) { 
    pDefOM = p; 
    addInfo = true;
  };  

  monitor_return_t run(PlanExecution* plan_exec);

  void* asynch_worker() 
  { errorlog << "DefensiveModelMonitor: don't run me asynch!" << spades::ende; return NULL;}

  //this is specific to this class
  void SetDefModel(DefOppModel* new_model){ 
    pDefOM = new_model; 
    addInfo = true;
  }

private:
  TeamSide my_side;
  const WorldHistory& history;  
private:
  //DefOppModel* pDefOM;
  DefOppModel* pDefOM;
  bool addInfo;

};

/* Note that this class does not delete the opp model set which is passed to it */
class OppModelSetMonitor : public Monitor 
{
public:
  OppModelSetMonitor(TeamSide my_side, const WorldHistory& history);
  OppModelSetMonitor(TeamSide my_side, const WorldHistory& history, OppModelSet* p, OppModelSet* pu);
  ~OppModelSetMonitor();

  monitor_return_t run(PlanExecution* plan_exec);

  void* asynch_worker();

  //this is specific to this class
  void SetModelSet(OppModelSet* new_model)
    { pOMSet = new_model; }

  static const int lenArrObs;

  int getDataReadyCount() { return data_ready_count; }

private:
  TeamSide my_side;
  const WorldHistory& history;  

private:
  OppModelSet* pOMSet;
  OppModelSet* pOMSetUniform;
  int data_ready_count;
  MovementObservation* arrObs;

  int read_idx;
  int write_idx;

  //PAT: fields I added
  PlayerDistribution* initial;
  PlayerDistribution* final;
  VecPosition* startPos;
  bool gotStartPos;
  BallMovement* currentBM;
  float prevBallDirection;
  int obsCount;
  bool watching;
  Unum prev_controller;
  int time_since_cont_change; //cycles since the ball controller changed

  ofstream outfile;
  ofstream outfile_uniform;
  ofstream outfile_dist;
  
};



/***********************************************************************************/
/*******************                Diagnosis                      *****************/
/***********************************************************************************/


/***********************************************************************************/
/*******************                Planner                        *****************/
/***********************************************************************************/

class FixedPlanPlanner : public Planner
{
 public:
  FixedPlanPlanner(TeamSide my_side, const WorldHistory& history);  
  
  planner_return_t run(Problem* theProblem, Plan* p);

  void* asynch_worker() 
  { errorlog << "FixedPlanPlanner: don't run me asynch!" << spades::ende; return NULL;}    
private:
  TeamSide my_side;
  const WorldHistory& history;  
 private:
  Plan* thePlan;  
};

/* this one loads a default plan with a minimal intelligence on selecting which
   one of a couple variants to choose */
#define NUM_PLANS_PER_MODE 2
class DefaultPlanPlanner : public Planner
{
 public:
  DefaultPlanPlanner(TeamSide my_side, const WorldHistory& history);  
  ~DefaultPlanPlanner();
  
  planner_return_t run(Problem* theProblem, Plan* p);

  void* asynch_worker() 
  { errorlog << "DefaultPlanPlanner: don't run me asynch!" << spades::ende; return NULL;}    

private:
  TeamSide my_side;
  const WorldHistory& history;  
 private:
  Plan* arrPlans[num_SetPlayMode][NUM_PLANS_PER_MODE];  

  bool FlipYCoordIfNeeded(Problem* theProblem, Plan* p);
  
};


/* Note that on destruction of this class, it will delete the pDefOM that 
   you give it */
class DefensivePlanner : public Planner
{
 public:
  DefensivePlanner(TeamSide my_side, const WorldHistory& history);  
  //DEB: put the right pointer here
  DefensivePlanner(TeamSide my_side, const WorldHistory& history, DefOppModel* p);  
  ~DefensivePlanner();
  
  planner_return_t run(Problem* theProblem, Plan* p);

  void* asynch_worker() 
  { errorlog << "DefensivePlanner: don't run me asynch!" << spades::ende; return NULL;}    

  //this is specific to this class
  void SetDefModel(DefOppModel* new_model);
  
private:
  TeamSide my_side;
  const WorldHistory& history;  
 private:
  /* DEB: here is where you put the variable for the defensive opponent model */
  //DefOppModel* pDefOM;
  //this char* is just temporary, delete it when you put in the real model
  DefOppModel* pDefOM;

};



/***********************************************************************************/
/*******************                Executor                       *****************/
/***********************************************************************************/

class StandardLanguageExecutor : public Executor
{
public:
  StandardLanguageExecutor(ModFormation* p, TeamSide my_side, const WorldHistory& history, CoachMessageQueue& q);
  ~StandardLanguageExecutor();
  
  executor_return_t run(PlanExecution* plan_exec);  

  void* asynch_worker() 
  { errorlog << "StandardLanguageExecutor: don't run me asynch!" << spades::ende; return NULL;}    

private:
  TeamSide my_side;
  const WorldHistory& history;  
private:
  CoachMessageQueue& mqueue;
  GreedyByRoleAI* pAI;

};

#endif
