/* This file contains the four modules for the planning architecture for the coach */

#include "ContModules.h"

#include "soccer_utils.h"
#include "ModCMRep.h"
#include "ServerParam.h"
#include "CoachParam.h"
#include "Logger.h"
using namespace spades;

#define DEB_MSG(x) 

#define OUTPUT(x) x
#define OUTPUT2(x) 

/***********************************************************************************/
/*******************                Monitor                        *****************/
/***********************************************************************************/

monitor_return_t TimeoutMonitor::run(PlanExecution* plan_exec)
{
  actionlog(210) << "TimeoutMonitor: status "
		 << plan_exec << " " << plan_exec->GetUnderlyingPlan() << ende;
  if (plan_exec->GetUnderlyingPlan() == NULL) {
    actionlog(50) << "Monitor: underlying plan is NULL, normal" << ende;    
    return MR_Normal; //it's always normal for a null plan  
  }  
  if (history.getWorldState().getPlayMode() == PM_PlayOn &&
      history.getWorldState().getTime() - history.getLastPlayModeChangeTime() > CoachParam::instance()->getSppTimeLimit()) {
    actionlog(50) << "Monitor: timeout, so Deviation" << ende;   
    return MR_Deviation;
  }  
  return MR_Normal;  
}

monitor_return_t OpponentPossessionMonitor::run(PlanExecution* plan_exec) 
{
  if (plan_exec->GetUnderlyingPlan() == NULL) {
    actionlog(50) << "OppPossMonitor: underlying plan is NULL, normal" << ende;    
    return MR_Normal; //it's always normal for a null plan  
  }  
  if (plan_exec->GetUnderlyingProblem() == NULL ||
      !IsSetPlayOffensive(plan_exec->GetUnderlyingProblem()->mode)) {
    //only monitor offensive
    actionlog(50) << "OppPossMonitor: only monitoring offensive" << ende;    
    return MR_Normal;
  }
  
  double dist;
  Unum controller = history.getWorldState().getClosestPlayerToPoint(TS_Both, history.getWorldState().getBall().getPos(), &dist);
  if (dist <= ServerParam::instance()->getSPKickableArea()) {
    if ((my_side == TS_Left && controller < 0) ||
	(my_side == TS_Right && controller > 0)) {
      actionlog(120) << "OpponentPossessionMonitor: opponents got ball " << controller << ende;
      return MR_Deviation;
    }
  }
  return MR_Normal;
}


monitor_return_t DefensiveModelMonitor::run(PlanExecution* plan_exec)
{
  if (pDefOM == NULL) {
    errorlog << "DefensiveModelMonitor can not run without a defensive opponent model!" << ende;
    return MR_Deviation;
  }

  if (history.getWorldState().getPlayMode() != PM_PlayOn) {
    actionlog(50) << "DefensiveModelMonitor: not play on, nothing to do" << ende;   
    return MR_Normal;
  }

  if ((plan_exec == NULL) || (plan_exec->GetUnderlyingProblem() == NULL)) {
    DEB_MSG(cout <<"def mod mon: plan_exec or underlying problem is null\n");
    addInfo = true;
    return MR_Normal;
  }
  
  if (!IsSetPlayDefensive(plan_exec->GetUnderlyingProblem()->mode)) {
    DEB_MSG(cout <<"def mod mon: not a defensive problem\n");
    addInfo = true;
    return MR_Normal;
  }

  if (addInfo) {
    bool added = false;
    added = pDefOM->addInfo(plan_exec->GetUnderlyingProblem(), 
			    history.getWorldState().getTime() - history.getLastPlayModeChangeTime(),
			    history.getWorldState().getBall().getPos());
    if (added) addInfo = false;
  }
  
  return MR_Normal;
}


const int OppModelSetMonitor::lenArrObs = 10;

OppModelSetMonitor::OppModelSetMonitor(TeamSide my_side, const WorldHistory& history)
  : Monitor(), my_side(my_side), history(history) 
{ 
  data_ready_count = 0;
  read_idx = 0;
  write_idx = 0;
  arrObs = new MovementObservation[lenArrObs];

  pOMSet = NULL;
  pOMSetUniform = NULL;

  initial = new PlayerDistribution;
  final = new PlayerDistribution;
  startPos = new VecPosition[ServerParam::instance()->getSPTeamSize()];
  gotStartPos = false;
  currentBM = new BallMovement;
  //VecPosition tmp;
  //tmp.x = 0;
  //tmp.y = 0;
  //currentBM->addBallPosition(tmp,0);
  prevBallDirection = 0;
    

  StartAsynchRun();
}  

OppModelSetMonitor::OppModelSetMonitor(TeamSide my_side, const WorldHistory& history, OppModelSet* p, OppModelSet* pu)
  : Monitor(), my_side(my_side), history(history) 
{ 
  data_ready_count = 0;
  read_idx = 0;
  write_idx = 0;
  arrObs = new MovementObservation[lenArrObs];

  pOMSet = p;
  pOMSetUniform = pu;

  initial = new PlayerDistribution;
  final = new PlayerDistribution;
  startPos = new VecPosition[ServerParam::instance()->getSPTeamSize()];
  gotStartPos = false;
  currentBM = new BallMovement;
  //VecPosition tmp;
  //tmp.x = 0;
  //tmp.y = 0;
  //currentBM->addBallPosition(tmp,0);
  prevBallDirection = 0;
  obsCount = 0;

  if (!CoachParam::instance()->getSppOmWeightOutputFN().empty()) {
    char tmpfn[255];
    outfile.open(CoachParam::instance()->getSppOmWeightOutputFN().c_str());
    strcpy(tmpfn, CoachParam::instance()->getSppOmWeightOutputFN().c_str());
    strcat(tmpfn, ".uniform");
    outfile_uniform.open(tmpfn);
    if (!outfile) {
      errorlog << "Could not open weight output file '" << CoachParam::instance()->getSppOmWeightOutputFN() << "'" << ende;
    }
    if (!outfile_uniform) {
      errorlog << "Could not open uniform weight output file '" << tmpfn << "'" << ende;
    }
  }
  if (!CoachParam::instance()->getSppOmDistOutputFN().empty()) {
    outfile_dist.open(CoachParam::instance()->getSppOmDistOutputFN().c_str());
    if (!outfile_dist) {
      errorlog << "Could not open dist output file '" << CoachParam::instance()->getSppOmDistOutputFN() << "'" << ende;
    }
  }

  StartAsynchRun();
}  

OppModelSetMonitor::~OppModelSetMonitor() 
{
  outfile.close();
  outfile_uniform.close();
  outfile_dist.close();
  delete [] arrObs;
  delete initial;
  delete final;
  delete []startPos;
  delete currentBM;
}

monitor_return_t OppModelSetMonitor::run(PlanExecution* plan_exec)
{
  if (history.getWorldState().getPlayMode() != PM_PlayOn) {
    gotStartPos = false;
    watching = true;
    prev_controller = Unum_Unknown;
    currentBM->clear();
    return MR_Normal;
  }

  else if ((plan_exec == NULL) || (plan_exec->GetUnderlyingProblem() == NULL))
    return MR_Normal;
  
  else if (!IsSetPlayOffensive(plan_exec->GetUnderlyingProblem()->mode))
    return MR_Normal;
  
  OUTPUT2(cout << history.getWorldState().getTime() << ": MSM running" << endl);

  //DEB: First do anything which has to be done this cycle
  if (history.getWorldState().getPlayMode() == PM_PlayOn && watching) {

    OUTPUT2(cout << history.getWorldState().getTime() << ": watching" << endl);
    
    double dist;

    if (history.getWorldState().getTime() - history.getLastPlayModeChangeTime() > CoachParam::instance()->getSppTimeLimit()) {
      watching = false;
      actionlog(120) << "OppModelSetMonitor: time limit reached, stopped watching" << ende;
      return MR_Normal;
    }

    if (!gotStartPos) {
      int i = 0;
      DEB_MSG(cout << "setting starting positions\n");
      actionlog(120) << "setting starting positions" << ende;
      for (i = 0; i < ServerParam::instance()->getSPTeamSize(); ++i) {
	if (history.getWorldState().getPlayer(getOppositeSide(my_side), i+1)) {
	  startPos[i] = history.getWorldState().getPlayer(getOppositeSide(my_side), i+1)->getPos();	
	  DEB_MSG(cout << history.getWorldState().getPlayer(getOppositeSide(my_side), i+1)->getPos() << "\n");
	  actionlog(140) << "player " << i << "'s position: "
			 << history.getWorldState().getPlayer(getOppositeSide(my_side), i+1)->getPos() << ende;
	} else {
	  startPos[i] = VecPosition(-1000, -1000);
	}
      }
      prev_controller = history.getWorldState().getClosestPlayerToPoint(TS_Both, history.getWorldState().getBall().getPos(), &dist);
      time_since_cont_change = 0;
      currentBM->addBallPosition(history.getWorldState().getBall().getPos(), 0); //this is the starting pos
      gotStartPos = true;
    }

    /* Now look to see if we need to add something to BallMovement */
    Unum controller = history.getWorldState().getClosestPlayerToPoint(TS_Both, history.getWorldState().getBall().getPos(), &dist);
    if (dist <= ServerParam::instance()->getSPKickableArea()) {
      /* first check to see if the opponents are controlling the ball, which makes us not
	 watch anymore */
      bool opp_control = 
	(my_side == TS_Left && controller < 0) ||
	(my_side == TS_Right && controller > 0);
      
      /* Now if this is a new controller, then add a ball movement and package observation*/
      if (controller != prev_controller && 
	  time_since_cont_change >= CoachParam::instance()->getSppMinObsTime() &&
	  (!opp_control || CoachParam::instance()->getSppObsOnLostBall())) {
	currentBM->addBallPosition(history.getWorldState().getBall().getPos(), time_since_cont_change);
	if (currentBM->getNumPosition() > 2) 
	  errorlog << "Why is currentBM so long? " << currentBM->getNumPosition() << ende;
	VecPosition endPos[ServerParam::instance()->getSPTeamSize()];
	for (int j = 0; j < ServerParam::instance()->getSPTeamSize(); ++j) {
	  if (history.getWorldState().getPlayer(getOppositeSide(my_side), j+1)) {
	    endPos[j] = history.getWorldState().getPlayer(getOppositeSide(my_side), j+1)->getPos();
	    DEB_MSG(cout << history.getWorldState().getPlayer(getOppositeSide(my_side), j+1)->getPos() << "\n");
	  } else {
	    endPos[j] = VecPosition(-1000, -1000);
	  }
	}
	DEB_MSG(cout << "writing observation to array");
	arrObs[write_idx].setLocations(startPos, endPos);
	arrObs[write_idx].setBallMovement(currentBM);
	arrObs[write_idx].setTime(currentBM->getTotalTime());
	DEB_MSG(cout << "....done\n");

	OUTPUT(cout << history.getWorldState().getTime() << ": packaged observation" << endl);
	while (data_ready_count >= lenArrObs) {
	  sched_yield();
	}
	pthread_mutex_lock(&mutex_data_ready);
	data_ready_count++;    
	pthread_cond_signal(&cv_data_ready);
	pthread_mutex_unlock(&mutex_data_ready);
	
	write_idx = (write_idx + 1) % lenArrObs;
	
	for (int i=0; i<ServerParam::instance()->getSPTeamSize(); i++)
	  startPos[i] = endPos[i];
	prev_controller = controller;
	currentBM->removeBallPosition(0);
	time_since_cont_change = 0;
      } 

      if (opp_control) {
	actionlog(120) << "OppModelSetMonitor: opponents got ball " << controller << ", not watching" << ende;
	watching = false;
	DEB_MSG(cout << "not watching anymore\n");
	return MR_Normal;
      }
    }
    else time_since_cont_change++;
  }  
  return MR_Normal;
}

void* OppModelSetMonitor::asynch_worker()
{
  int read_idx = 0;

  while(1) {
    pthread_mutex_lock(&mutex_data_ready);
    while (data_ready_count == 0) {
      pthread_cond_wait(&cv_data_ready, &mutex_data_ready);
    }
    pthread_mutex_unlock(&mutex_data_ready);
    //Do stuff here
    //pOMSet->calculate(arrObs[read_idx], ???, ???);
    DEB_MSG(cout << "processing observation\n");
    actionlog(100) << "processing observation" << ende;
    if (outfile_dist)
      pOMSet->calculate(&arrObs[read_idx], initial, final, &outfile_dist);
    else
      pOMSet->calculate(&arrObs[read_idx], initial, final);
    pOMSetUniform->setToUniform();
    pOMSetUniform->calculate(&arrObs[read_idx], initial, final);
    
    //indicate that we have processed the data
    actionlog(100) << "processed an observation" << ende;
    DEB_MSG(cout << ". . . . . . done\n");
    OUTPUT(pOMSet->printProbs());
    pOMSet->logProbs(100);
    if (outfile) {
      outfile << "Time: " << history.getWorldState().getTime() << endl;
      pOMSet->printCompactProbs(outfile);
      //outfile << "Test" << endl;
      actionlog(100) << "tried to write to file" << ende;
    } else {
      if (CoachParam::instance()->getSppOmWeightOutputFN()[0])
        errorlog << "Did not write to outfile" << ende;
    }
    if (outfile_uniform) {
      outfile_uniform << "Time: " << history.getWorldState().getTime() << endl;
      pOMSetUniform->printCompactProbs(outfile_uniform);
      //outfile << "Test" << endl;
      actionlog(100) << "tried to write to file (uniform)" << ende;
    } else {
      if (CoachParam::instance()->getSppOmWeightOutputFN()[0])
        errorlog << "Did not write to outfile (uniform)" << ende;
    }

    pthread_mutex_lock(&mutex_data_ready);
    data_ready_count--;
    pthread_mutex_unlock(&mutex_data_ready);

    //advance the read_idx
    read_idx = (read_idx + 1) % lenArrObs;
  }
  return NULL;
}




/***********************************************************************************/
/*******************                Diagnosis                      *****************/
/***********************************************************************************/


/***********************************************************************************/
/*******************                Planner                        *****************/
/***********************************************************************************/

/************* FixedPlanPlanner **************/

FixedPlanPlanner::FixedPlanPlanner(TeamSide my_side, const WorldHistory& history)
  : Planner(), my_side(my_side), history(history)
{
  //ifstream infile("../shared/Plans/example2");
  //ifstream infile("../shared/Plans/justinit");
  //ifstream infile("../shared/Plans/justsend");
  //ifstream infile("../shared/Plans/sendgo");
  //ifstream infile("../shared/Plans/1pass");
  //ifstream infile("../shared/Plans/1passsend");
  //ifstream infile("../shared/Plans/justdribble");
  //ifstream infile("../shared/Plans/passdribpass");
  ifstream infile("../shared/Plans/all_players");

  if (!infile)
    errorlog << "FixedPlanPlanner: could not open plan file" << ende;  

  thePlan = new Plan;
  if (!thePlan)
    errorlog << "FixedPlanPlanner: plan failed to allocate" << ende;  

  if (!thePlan->Read(infile))
    errorlog << "FixedPlanPlanner: couldn't read plan" << ende;

  if (!(my_side == TS_Left || my_side == TS_Right))
    errorlog << "FixedPlanPlanner: don't know what side I'm on" << ende;
  
  //the fixed plan is given for left side. If we are right, we have to flip
  if (my_side == TS_Right)
    thePlan->FlipCoords();

  cout << "Original Plan:" << endl;
  thePlan->Print(cout);
  cout << "message: " << thePlan->PrintStrCompact() << endl;  

  /*
  thePlan->ComputeAllPairsDistances();
  cout << "All Pair distances:" << endl;
  thePlan->PrintOnlyEdgeLens(cout);
  cout << "message: " << thePlan->PrintStrCompact() << endl;  

  thePlan->MakeMinimalDispatchable();  
  cout << "Min Dispatch:" << endl;
  thePlan->PrintOnlyEdgeLens(cout);
  cout << "message: " << thePlan->PrintStrCompact() << endl;  
  */
}

  
planner_return_t FixedPlanPlanner::run(Problem* theProblem, Plan* p)
{
  if (int_random(20) == 0) {
    actionlog(30) << "FixedPlanPlanner returning plan" << ende;  
    p->copyFrom(thePlan);
    return PlnRet_Complete;  
  } else {
    return PlnRet_NotReadyYet;
  }
}

/************* DefaultPlanPlanner **************/

DefaultPlanPlanner::DefaultPlanPlanner(TeamSide my_side, const WorldHistory& history)
  : Planner(), my_side(my_side), history(history)
{
  for (int spm=0; spm<num_SetPlayMode; spm++) {
    const char* fn_to_load[NUM_PLANS_PER_MODE];
    switch ((SetPlayMode)spm) {
    case SPM_No_Mode:
      fn_to_load[0] = fn_to_load[1] = NULL;
      break;

    case SPM_My_Kick_Off:
      fn_to_load[0] = CoachParam::instance()->getSppDefMyKickOff1().c_str();
      fn_to_load[1] = CoachParam::instance()->getSppDefMyKickOff2().c_str();
      break;
    case SPM_My_Kick_In:
      fn_to_load[0] = CoachParam::instance()->getSppDefMyKickIn1().c_str();
      fn_to_load[1] = CoachParam::instance()->getSppDefMyKickIn2().c_str();
      break;
    case SPM_My_Corner_Kick:
      fn_to_load[0] = CoachParam::instance()->getSppDefMyCornerKick1().c_str();
      fn_to_load[1] = CoachParam::instance()->getSppDefMyCornerKick2().c_str();
      break;
    case SPM_My_Goal_Kick:
      fn_to_load[0] = CoachParam::instance()->getSppDefMyGoalKick1().c_str();
      fn_to_load[1] = CoachParam::instance()->getSppDefMyGoalKick2().c_str();
      break;
    case SPM_My_Free_Kick:
      fn_to_load[0] = CoachParam::instance()->getSppDefMyFreeKick1().c_str();
      fn_to_load[1] = CoachParam::instance()->getSppDefMyFreeKick2().c_str();
      break;
    case SPM_My_Goalie_Catch:
      fn_to_load[0] = CoachParam::instance()->getSppDefMyGoalieCatch1().c_str();
      fn_to_load[1] = CoachParam::instance()->getSppDefMyGoalieCatch2().c_str();
      break;

    case SPM_Their_Kick_Off:
      fn_to_load[0] = CoachParam::instance()->getSppDefTheirKickOff().c_str();
      fn_to_load[1] = NULL;
      break;
    case SPM_Their_Kick_In:
      fn_to_load[0] = CoachParam::instance()->getSppDefTheirKickIn().c_str();
      fn_to_load[1] = NULL;
      break;
    case SPM_Their_Corner_Kick:
      fn_to_load[0] = CoachParam::instance()->getSppDefTheirCornerKick().c_str();
      fn_to_load[1] = NULL;
      break;
    case SPM_Their_Goal_Kick:
      fn_to_load[0] = CoachParam::instance()->getSppDefTheirGoalKick().c_str();
      fn_to_load[1] = NULL;
      break;
    case SPM_Their_Free_Kick:
      fn_to_load[0] = CoachParam::instance()->getSppDefTheirFreeKick().c_str();
      fn_to_load[1] = NULL;
      break;
    case SPM_Their_Goalie_Catch:
      fn_to_load[0] = CoachParam::instance()->getSppDefTheirGoalieCatch().c_str();
      fn_to_load[1] = NULL;
      break;
    default:
      errorlog << "DefaultPlanPlanner: What mode is this? " << spm << ende;
      fn_to_load[0] = fn_to_load[1] = NULL;
    } //switch

    for (int i=0; i<NUM_PLANS_PER_MODE; i++) {
      if (fn_to_load[i] == NULL) {
	arrPlans[spm][i] = NULL;
	continue;
      }
    
      arrPlans[spm][i] = new Plan;
      if (!arrPlans[spm][i]) {
	errorlog << "DefaultPlanPlanner: Failed to allocate mode " << spm << ", num " << i << ende;
	return;
      }
      
      ifstream infile;
      char fn[MAX_FILE_LEN];
      sprintf(fn, "%s/%s", CoachParam::instance()->getSppPlansDir().c_str(), fn_to_load[i]);
      infile.open(fn);
      if (!infile) {
	errorlog << "Could not open file '" << fn << "' for mode " << spm << ", num " << i << ende;
	delete arrPlans[spm][i];
	arrPlans[spm][i] = NULL;
	continue;
      }
      
      if (!arrPlans[spm][i]->Read(infile)) {
	errorlog << "Plan read from '" << fn << "' failed for mode " << spm << ", num " << i << ende;
	delete arrPlans[spm][i];
	arrPlans[spm][i] = NULL;
	continue;
      }

      if (my_side == TS_Right)
	arrPlans[spm][i]->FlipCoords();	
    }
  } //mode loop
}

DefaultPlanPlanner::~DefaultPlanPlanner()
{
  for (int spm=0; spm<num_SetPlayMode; spm++) {
    for (int i=0; i<NUM_PLANS_PER_MODE; i++) {
      if (arrPlans[spm][i])
	delete arrPlans[spm][i];
    }
  }  
}

bool DefaultPlanPlanner::FlipYCoordIfNeeded(Problem* theProblem, Plan* p)
{
  if ((my_side == TS_Left && theProblem->ball_pos.getY() < 0) ||
      (my_side == TS_Right && theProblem->ball_pos.getY() > 0)) {
    p->FlipCoordsY(); 
    return true;
  }
  return false;  
}

//The plan which comes out of here must be in coords for left team
planner_return_t DefaultPlanPlanner::run(Problem* theProblem, Plan* p)
{
  switch (theProblem->mode) {
    
  case SPM_My_Kick_Off:
  case SPM_My_Goalie_Catch:
    p->copyFrom(arrPlans[theProblem->mode][int_random(NUM_PLANS_PER_MODE)]);
    break;

  case SPM_My_Kick_In:
  case SPM_My_Corner_Kick:
  case SPM_My_Goal_Kick:
    p->copyFrom(arrPlans[theProblem->mode][int_random(NUM_PLANS_PER_MODE)]);
    FlipYCoordIfNeeded(theProblem, p);
    p->ToAbsoluteFrame(theProblem->ball_pos);
    break;

  case SPM_My_Free_Kick:
    if (theProblem->ball_pos.getY() < 0)
      p->copyFrom(arrPlans[theProblem->mode][0]);
    else
      p->copyFrom(arrPlans[theProblem->mode][1]);
    p->ToAbsoluteFrame(theProblem->ball_pos);
    break;

  case SPM_Their_Kick_Off:
  case SPM_Their_Goalie_Catch:
    p->copyFrom(arrPlans[theProblem->mode][0]);
    break;

  case SPM_Their_Corner_Kick:
  case SPM_Their_Goal_Kick:
    //plan is for left_positive_y / right_negative_y side
    FlipYCoordIfNeeded(theProblem, p);
    p->copyFrom(arrPlans[theProblem->mode][0]);
    break;

  case SPM_Their_Free_Kick:
    p->copyFrom(arrPlans[theProblem->mode][0]);
    FlipYCoordIfNeeded(theProblem, p);
    p->ToAbsoluteFrame(theProblem->ball_pos);
    //Make sure all locations are in field
    ((InitialPosPN*)p->GetPlanNode(0))->AdjustPosToWithin(&ServerParam::instance()->getSPFieldRectangle());
    break;
    
  case SPM_Their_Kick_In:
    //there's only one position here, but we need to set it!
    p->copyFrom(arrPlans[theProblem->mode][0]);
    p->ToAbsoluteFrame(theProblem->ball_pos); //just to make sure    
    p->GetInitPos()->SetPlayerPos(1,
				  VecPosition::getPointInBetween(theProblem->ball_pos,
								 getGoalPosForSide(my_side),
								 ServerParam::instance()->getSPOffsideKickMargin()+1));
    //this is now in absolute coords, left side
    break;

  case SPM_No_Mode:
  default:
    errorlog << "DefaultPlanPlanner: What mode is this? " << theProblem->mode << ende;
    return PlnRet_Failure;
  }
  
  return PlnRet_Complete;  
}

/************* DefensivePlanner **************/

DefensivePlanner::DefensivePlanner(TeamSide my_side, const WorldHistory& history)
  : Planner(), my_side(my_side), history(history)
{
  pDefOM = NULL;  
}

DefensivePlanner::DefensivePlanner(TeamSide my_side, const WorldHistory& history, DefOppModel* p)
  : Planner(), my_side(my_side), history(history)
{
  pDefOM = p;
}

DefensivePlanner::~DefensivePlanner()
{
  if (pDefOM != NULL)
    delete pDefOM;
}

void DefensivePlanner::SetDefModel(DefOppModel* new_model)
{ 
  if (pDefOM)
    delete pDefOM;
  
  pDefOM = new_model; 
}
  
planner_return_t DefensivePlanner::run(Problem* theProblem, Plan* p)
{
  if (pDefOM == NULL) {
    errorlog << "DefensivePlanner can not run without a defensive opponent model!" << ende;
    return PlnRet_Failure;
  }

  if (!IsSetPlayDefensive(theProblem->mode)) {
    actionlog(60) << "Defensive Planner only for defensive plays" << ende;
    return PlnRet_Failure;
  }

  VecPosition loc[NUM_PLAYERS];
  int num_set;
  //VAR: you could chnage this buffer
  Rectangle shrunkField = ServerParam::instance()->getSPFieldRectangle().shrink(1);

  DEB_MSG(cout << "DefensivePlanner called: " << history.getWorldState().getTime() << "\n");
  actionlog(110) << "defensive planner called" << ende;

  DEB_MSG(cout << "calling set defense: " << history.getWorldState().getTime() << "\n");
  actionlog(120) << "calling set defense" << ende;
  num_set = pDefOM->setDefense(theProblem, loc);
  if (num_set == -1) {
    actionlog(120) << "setDefense called for nondefensive play" << ende;
    return PlnRet_Failure;
  }
  else if (num_set == -2) errorlog << "check parameters for number of positions to return" << ende;
  else if (num_set == 0) errorlog << "setDefense returned 0 points" << ende;
  DEB_MSG(cout << "done calling set defense: " << history.getWorldState().getTime() << "\n");
  actionlog(120) << "done calling set defense" << ende;

  InitialPosPN* pIP = new InitialPosPN;

  p->Clear();

  FieldImage fi(5, CoachParam::instance()->getSppDrawPlannedWaypointsScale());
  fi.addFieldLines();
  
  int currPlayer = 1;
  for (int i=0; i<num_set; i++) {
    loc[i] = shrunkField.adjustToWithin(loc[i]);
    switch (theProblem->mode) {
    case SPM_Their_Kick_Off:
    case SPM_Their_Kick_In:
    case SPM_Their_Free_Kick:
    case SPM_Their_Corner_Kick:
      //move out of forbidden circle area
      if (theProblem->ball_pos.getDistanceTo(loc[i]) < ServerParam::instance()->getSPOffsideKickMargin() + CoachParam::instance()->getSppForbidAreaBuffer()) {
	VecPosition rel_loc = loc[i] - theProblem->ball_pos;
	rel_loc = rel_loc.scaleTo(ServerParam::instance()->getSPOffsideKickMargin() + CoachParam::instance()->getSppForbidAreaBuffer());
	loc[i] = theProblem->ball_pos + rel_loc;
      }
      if (!shrunkField.isInside(loc[i])) {
	actionlog(80) << "DefP: After pushing out, we are outside of field " << loc[i] << ende;
	loc[i] = shrunkField.adjustToWithin(loc[i]);
	if (theProblem->ball_pos.getDistanceTo(loc[i]) < ServerParam::instance()->getSPOffsideKickMargin() + CoachParam::instance()->getSppForbidAreaBuffer()) {
	  actionlog(80) << "DefP: After pushing out and field adjustment, we are too close "
			<< loc[i] << ende;
	  float xdiff = loc[i].getX() - theProblem->ball_pos.getX();
	  float ydiff = loc[i].getY() - theProblem->ball_pos.getY();
	  float sol1, sol2;
	  int num_sol = QuadraticFormula(1, 2*fabs(xdiff), Sqr(xdiff) + Sqr(ydiff) - 
					 Sqr(ServerParam::instance()->getSPOffsideKickMargin() + CoachParam::instance()->getSppForbidAreaBuffer()),
					 &sol1, &sol2);
	  if (num_sol == 0) {
	    errorlog << "FKDefAdjustment: How can there be no solutions" << ende;
	  } else if (num_sol == 1) {
	    loc[i].setX( loc[i].getX() + signf(xdiff)*sol1);
	  } else if (num_sol == 2) {
	    if (sol1 > 0) {
	      loc[i].setX( loc[i].getX() + signf(xdiff)*sol1);	      
	    } else if (sol2 > 0) {
	      loc[i].setX( loc[i].getX() + signf(xdiff)*sol2);
	    } else {
	      errorlog << "DefAdjustment: how are both solutions negative?" << ende;
	    }
	  }
	}
      }
      break;
    case SPM_Their_Goal_Kick:
    case SPM_Their_Goalie_Catch:
      //move out of pen box
      loc[i] = ServerParam::instance()->getSPPenaltyAreaRectangle(my_side == TS_Right).adjustToOutside(loc[i], CoachParam::instance()->getSppForbidAreaBuffer());
      break;
    default:
      errorlog << "DefensivePlanner: what is the mode? " << theProblem->mode << ende;
      break;
    }

    if (!shrunkField.isInside(loc[i]))
      warninglog(10) << "Defensive Plan: after adjustment, I'm still off field, ignoring pos "
		     << loc[i] << ende;
    else
      {
        pIP->SetPlayerPos(currPlayer++, loc[i]);
        if (CoachParam::instance()->getSppDrawPlannedWaypoints())
          fi.addPoint(loc[i]);
      }
    
  }

  if (CoachParam::instance()->getSppDrawPlannedWaypoints())
    fi.writeTo((CoachParam::instance()->getSppDrawPlannedWaypointsFStem() +
                toString(theProblem->time)).c_str());

  p->SetPlanNode(0, pIP);
  
  return PlnRet_Complete;
}



/***********************************************************************************/
/*******************                Executor                       *****************/
/***********************************************************************************/
StandardLanguageExecutor::StandardLanguageExecutor(ModFormation* p, TeamSide my_side, const WorldHistory& history, CoachMessageQueue& q)
  : Executor(), my_side(my_side), history(history), mqueue(q)
{
  pAI = new GreedyByRoleAI(p, my_side, history.getWorldState().getGoalieNum(my_side));
}

StandardLanguageExecutor::~StandardLanguageExecutor()
{
  delete pAI;
}



executor_return_t StandardLanguageExecutor::run(PlanExecution* plan_exec)
{
  if (plan_exec->GetUnderlyingPlan() == NULL)
    return ER_Normal;
  
  if (plan_exec->fresh_plan) {

    /* SMURF: this is actually a little dangerous because we will be changing the
       plan underneath the PlanExecution, which could conceivably cause bugs.
       However, we are only changing the number, so it should be okay */
    pAI->run(plan_exec->GetUnderlyingPlan(), plan_exec->GetUnderlyingProblem());
    
    //we need to sure that we send out absolute coords
    if (plan_exec->GetUnderlyingPlan()->GetCoordFrame() == CF_BallRelative) {
      actionlog(100) << "changing to absolute coordinates" << ende;    
      plan_exec->GetUnderlyingPlan()->ToAbsoluteFrame(history.getWorldState().getBall().getPos());	
    }
    /*
      plan_exec->GetUnderlyingPlan()->Print(cout);
    */

    //the underlying plan assumes we are left. Flip before sending to players
    if (my_side == TS_Right) {
      actionlog(100) << "flipping coords for right team" << ende;    
      plan_exec->GetUnderlyingPlan()->FlipCoords();
    }
    
    if (!plan_exec->ConvertToStandardLanguage(&mqueue))
      errorlog << "Conversion to standard language failed" << ende;

    plan_exec->fresh_plan = false;
  }
  
  return ER_Normal;
}

