// -*-c++-*-

/***************************************************************************
                                 gzfstream.h
                      Compression and decompression file streams
                             -------------------
    begin                : 31-OCT-2001
    copyright            : (C) 2001, 2002 by The RoboCup Soccer Server 
                           Maintenance Group.
    email                : sserver-admin@lists.sourceforge.net
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU LGPL as published by the Free Software  *
 *   Foundation; either version 2 of the License, or (at your option) any  *
 *   later version.                                                        *
 *                                                                         *
 ***************************************************************************/

#ifndef GZFSTREAM_H
#define GZFSTREAM_H

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif


#ifdef HAVE_LIBZ
#include <zlib.h>
#include <iostream>

#ifdef HAVE_SSTREAM
#include <sstream>
#else
#include <strstream>
#endif

#ifndef EOF
#define EOF (-1)
#endif

namespace rcss
{
  namespace gz
  {
    class gzfstreambuf
      : public std::streambuf
    {
    public:
      typedef int int_type;
      typedef char char_type;

      static const int DEFAULT_BUFSIZE = 512;
      
      gzfstreambuf ( int bufsize = DEFAULT_BUFSIZE )
        : M_open ( false ),
          M_gzfile ( NULL ),
          bufsize_(bufsize),
          inbuf_(NULL),
          outbuf_(NULL),
          remained_(0)
      {
      }

      ~gzfstreambuf ()
      {
        if ( is_open () )
          close ();

        delete [] inbuf_;
        delete [] outbuf_;
      }

      bool is_open () const
      { 
        return M_open;
      }  

      bool
      openOut( const char* path, 
	       int level = Z_DEFAULT_COMPRESSION,
	       int strategy = Z_DEFAULT_STRATEGY )
      {
#ifdef HAVE_SSTREAM
        std::ostringstream mode_str;
#else
        std::ostrstream mode_str;
#endif

        mode_str << "wb" << level;
    
        if ( strategy == Z_FILTERED )
          mode_str << "f";
        else if ( strategy == Z_HUFFMAN_ONLY )
          mode_str << "h";

#ifndef HAVE_SSTREAM    
        mode_str << std::ends;
#endif

#ifdef HAVE_SSTREAM    
        M_gzfile = gzopen ( path, mode_str.str().c_str() );
#else
        M_gzfile = gzopen ( path, mode_str.str() );
        mode_str.freeze( false );
#endif
        M_open = ( M_gzfile != NULL );
        return M_open;
      }

      bool
      openOut ( int fd,
		int level = Z_DEFAULT_COMPRESSION,
		int strategy = Z_DEFAULT_STRATEGY )
      {
#ifdef HAVE_SSTREAM
        std::ostringstream mode_str;
#else
        std::ostrstream mode_str;
#endif
    
        mode_str << "wb" << level;
    
        if ( strategy == Z_FILTERED )
          mode_str << "f";
        else if ( strategy == Z_HUFFMAN_ONLY )
          mode_str << "h";
    
#ifndef HAVE_SSTREAM    
        mode_str << std::ends;
#endif
    
#ifdef HAVE_SSTREAM    
        M_gzfile = gzdopen ( fd, mode_str.str().c_str() );
#else
        M_gzfile = gzdopen ( fd, mode_str.str() );
        mode_str.freeze( false );
#endif
        M_open = ( M_gzfile != NULL );
        return M_open;
      }

      bool
      openIn( const char* path )
      {
#ifdef HAVE_SSTREAM
        std::ostringstream mode_str;
#else
        std::ostrstream mode_str;
#endif

        mode_str << "rb";
    
#ifndef HAVE_SSTREAM    
        mode_str << std::ends;
#endif

#ifdef HAVE_SSTREAM    
        M_gzfile = gzopen ( path, mode_str.str().c_str() );
#else
        M_gzfile = gzopen ( path, mode_str.str() );
        mode_str.freeze( false );
#endif
        M_open = ( M_gzfile != NULL );
        return M_open;
      }

      bool
      openIn ( int fd )
      {
#ifdef HAVE_SSTREAM
        std::ostringstream mode_str;
#else
        std::ostrstream mode_str;
#endif
    
        mode_str << "rb";

#ifndef HAVE_SSTREAM    
        mode_str << std::ends;
#endif
    
#ifdef HAVE_SSTREAM    
        M_gzfile = gzdopen ( fd, mode_str.str().c_str() );
#else
        M_gzfile = gzdopen ( fd, mode_str.str() );
        mode_str.freeze( false );
#endif
        M_open = ( M_gzfile != NULL );
        return M_open;
      }

      void close ()
      {
        _flush ();
        gzclose ( M_gzfile );
        M_gzfile = NULL;
        M_open = false;
      }

    protected:
      void _flush()
      {
        int size = (pptr () - outbuf_) * sizeof(char_type);
        if ( size > 0 )
          gzwrite ( M_gzfile, outbuf_, size );
      }

      int_type overflow(int_type c = EOF)
      {
	if (!M_open)
	  return EOF;
	
        // this method is supposed to flush the put area of the buffer
        // to the I/O device

        // if the buffer was not already allocated nor set by user,
        // do it just now
        if (pptr() == NULL)
          {
            outbuf_ = new char_type[bufsize_];
          }
        else
          {
            _flush();
          }
        setp(outbuf_, outbuf_ + bufsize_);
        if (c != EOF)
          sputc(c);
        return 0;
      }

      int sync()
      {
        // just flush the put area
        if (pptr() != NULL)
          {
            _flush();
            setp(outbuf_, outbuf_ + bufsize_);
            return 0;
          }
        return 0;
      }

      int_type underflow()
      {
	if (!M_open)
	  return EOF;

        // this method is supposed to read some bytes from the I/O device

        // if the buffer was not already allocated nor set by user,
        // do it just now
        if (gptr() == NULL)
          {
            inbuf_ = new char_type[bufsize_];
          }

        if (remained_ != 0)
          inbuf_[0] = remainedchar_;

        int readn = gzread ( M_gzfile, (void*)(inbuf_ + remained_),
                             bufsize_ * sizeof(char_type) - remained_);

        // if (readn == 0 && remained_ != 0)
        // error - there is not enough bytes for completing
        // the last character before the end of the stream
        // - this can mean error on the remote end

        if (readn == 0)
          return EOF;

        int totalbytes = readn + remained_;
        setg(inbuf_, inbuf_,
             inbuf_ + totalbytes / sizeof(char_type));

        remained_ = totalbytes % sizeof(char_type);
        if (remained_ != 0)
          remainedchar_ = inbuf_[totalbytes / sizeof(char_type)];

        return sgetc();
      }

      virtual pos_type 
      seekoff(off_type off, std::ios_base::seekdir dir,
	      std::ios_base::openmode mode = std::ios_base::in | std::ios_base::out)
      {
	//cerr << "seekpos called" << off << " " << dir << " " << mode << endl;
	//TODO: We should probably check the mode somehow
	pos_type ret =  pos_type(off_type(-1)); 
	if (!M_open)
	  return ret;

	if (dir == std::ios_base::cur && off == 0)
	  {
	    ret = gzseek(M_gzfile, 0, SEEK_CUR);
	    ret -= _M_in_cur ? _M_in_cur - _M_in_beg : 0;
	    //cerr << "seekpos; just query " << ret << " " << _M_in_cur - _M_in_beg << endl;
	  }
	else
	  {
#ifdef BROKEN_CODE
	    //pfr: I started working on this, copying and modifying from filebuf.
	    // I never got it to work right, but I'm leaving the code here in case someone find it useful
	    bool testget = _M_in_cur && _M_in_beg < _M_in_end;
	    bool testput = _M_out_cur && _M_out_beg < _M_out_end;
	    off_type real_offset = off;
	    _M_pback_destroy();

	    if (testput)
	      this->sync();
	    if (testget && dir == std::ios_base::cur)
	      real_offset += _M_in_cur - _M_in_beg;

	    //forget what's in our read buffer
	    setg(_M_in_beg, NULL, _M_in_end);
	
	    int whence;
	    switch (dir)
	      {
	      case std::ios_base::beg: whence = SEEK_SET; break;
	      case std::ios_base::cur: whence = SEEK_CUR; break;
	      case std::ios_base::end:
		// This is not actually supported. Should we do something?
		whence = SEEK_END;
		break;
	      }
	
	    ret = gzseek(M_gzfile, real_offset, whence);
	    cerr << "seekpos; hard case " << ret << " " << real_offset << endl;
#endif
	  }
	
	return ret;
      }
      
      /* The following virual methods from streambuf may also need to be redefined
	 void imbue(const locale& __loc) ;
	 gzfstreambuf* setbuf(char_type*, streamsize);
	 pos_type seekpos(pos_type, ios_base::openmode __mode = ios_base::in | ios_base::out);
	 streamsize showmanyc();
	 streamsize xsgetn(char_type* __s, streamsize __n);

	 uflow is also a method, but it's default definition seems fine
	 int_type pbackfail(int_type __c  = traits_type::eof());
	 streamsize xsputn(const char_type* __s, streamsize __n)
      */
      
    private:

      // not for use
      gzfstreambuf(const gzfstreambuf&)
        : std::streambuf()
      {}

      gzfstreambuf& operator=(const gzfstreambuf&)
      { return *this; }

      bool M_open;
      gzFile M_gzfile;
      std::streamsize bufsize_;
      char_type *inbuf_;
      char_type *outbuf_;
      int remained_;
      char_type remainedchar_;
    };

    class gzifstream
      : public std::istream
    {
    public:
      gzifstream( int buffer_size = gzfstreambuf::DEFAULT_BUFSIZE, const char* filename = NULL)
        : std::istream( &M_gzfstreambuf ),
	  M_gzfstreambuf( buffer_size )
      { if (filename) open(filename); }
      gzifstream( const char* filename)
        : std::istream( &M_gzfstreambuf ),
	  M_gzfstreambuf( gzfstreambuf::DEFAULT_BUFSIZE )
      { if (filename) open(filename); }
      ~gzifstream() {}

      bool is_open() const { return M_gzfstreambuf.is_open(); }
      gzfstreambuf* rdbuf() { return &M_gzfstreambuf; }
      void open(const char* filename)
      {
	if (!M_gzfstreambuf.openIn(filename))
	  setstate(std::ios_base::badbit);
      }
      void close() { return M_gzfstreambuf.close(); }

    private:
      // not for use
      gzifstream(const gzifstream&);
      gzifstream& operator=(const gzifstream&);

      gzfstreambuf  M_gzfstreambuf;
    };

    class gzofstream
      : public std::ostream
    {
    public:
      gzofstream( int buffer_size = gzfstreambuf::DEFAULT_BUFSIZE, const char* filename = NULL)
        : std::ostream( &M_gzfstreambuf ),
	  M_gzfstreambuf( buffer_size )
      { if (filename) open(filename); }
      gzofstream( const char* filename)
        : std::ostream( &M_gzfstreambuf ),
	  M_gzfstreambuf( gzfstreambuf::DEFAULT_BUFSIZE )
      { if (filename) open(filename); }
      ~gzofstream() {}

      bool is_open() const { return M_gzfstreambuf.is_open(); }
      gzfstreambuf* rdbuf() { return &M_gzfstreambuf; }
      void open(const char* filename)
      {
	if (!M_gzfstreambuf.openOut(filename))
	  setstate(std::ios_base::badbit);
      }
      void open(const char* filename, int level)
      {
	if (!M_gzfstreambuf.openOut(filename, level))
	  setstate(std::ios_base::badbit);
      }
      void open(const char* filename, int level, int strategy)
      {
	if (!M_gzfstreambuf.openOut(filename, level, strategy))
	  setstate(std::ios_base::badbit);
      }
      void close() { return M_gzfstreambuf.close(); }

    private:
      // not for use
      gzofstream(const gzofstream&);
      gzofstream& operator=(const gzofstream&);

      gzfstreambuf  M_gzfstreambuf;
    };
    
  } // namespace gz
} // namespace rcss
#endif //HAVE_LIBZ

#endif 
