/* -*- Mode: C++ -*- */

/*
    libscoach: Soccer Coach library for use with the SoccerServer system 

    Copyright (C) 2001  Patrick Riley, Paul Carpenter, Gal Kaminka, Manuela Veloso
    Copyright (C) 1998,1999,2000 Patrick Riley, Peter Stone, Manuela Veloso

    This library is free software; you can redistribute it and/or
    modify it under the terms of the GNU Lesser General Public
    License as published by the Free Software Foundation; either
    version 2.1 of the License, or (at your option) any later version.

    This library is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
    Lesser General Public License for more details.

    You should have received a copy of the GNU Lesser General Public
    License along with this library; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

*/
/* Classes dealing with statistical stuff from data points */

#ifndef _DATA_H_
#define _DATA_H_

#include <math.h>
#include <iostream>
#include <vector>

//Min and Max are not currently written to or read from a file.
//Maybe some day I'll fix that
class SingleDataSummary {

public:
  SingleDataSummary();

  void reset();
  
  void addPoint(double x);
  double getMean() const;
  double getVariance() const;
  double getStdDev() const;
  double getMin() const { return min; }
  double getMax() const { return max; }
  int getNumPoints() const;
  double getSum() const { return sum; }

  void combineWith(const SingleDataSummary* sds);

  void setPrecision(int prec) { precision = prec; }
  
      
  int  getPrecision() const { return precision; }
  
  void writeInfoToFile(std::ostream& output_file, bool WithNumPoints = false) const;
  void writeCompactInfoToFile(std::ostream& output_file, bool WithNumPoints = false) const;
  void writeCompactMeanStdevToFile(std::ostream& output_file, bool WithNumPoints = false) const;

  /* if num_points is given, it should NOT be in the file */
  bool readInfoFromFile(std::istream& infile, int num_points = -1);
  bool readCompactInfoFromFile(std::istream& infile, int num_points = -1);

  friend std::ostream& operator<<(std::ostream& os, const SingleDataSummary& sds)
  { sds.writeInfoToFile(os, true); return os; }
    
 private:
  int n;
  double sum;
  double sum_squares;
  double min;
  double max;

  int precision;
};

//Only maintains info for max and mean
class MaxMeanSummary 
{
public:
  MaxMeanSummary() { reset(); }
  MaxMeanSummary(int new_n, double new_sum, double new_max)
  { n = new_n; sum = new_sum; max = new_max; }

  void reset();
  
  void addPoint(double x);
  double getMean();
  double getMax() { return max; }
  int getNumPoints() { return n; }

private:
  int n;
  double sum;
  double max;
} ;

class MinMaxSummary
{
public:
  MinMaxSummary() { reset(); }

  void reset() { min = HUGE_VAL; max = -HUGE_VAL; curr_index = 0; mindex = maxdex = -1;}
  
  void addPoint(double x);
  double getMin() { return min; }
  double getMax() { return max; }
  int getMinIndex() { return mindex; }
  int getMaxIndex() { return maxdex; }

private:
  double min;
  double max;
  int mindex;
  int maxdex;
  int curr_index;
} ;

//you give it a series of void* pointers and score and then you can get out the best N
class RecordBestN 
{
public:
  RecordBestN(int N);
  ~RecordBestN();
  
  void Clear();
  
  void AddPoint (void* data, float score);
  //0th best is the best
  void* GetBest(int idx = 0) { return (idx < num_stored) ? arrData[idx] : NULL; }
  float GetBestScore(int idx = 0) { return (idx < num_stored) ? arrScore[idx] : 0.0; }
  int GetNumStored() { return num_stored; }
  
private:
  int N;
  int num_stored;
  void** arrData;
  float* arrScore;
  
} ;

class IntBucket
{
public:
  typedef std::vector<int> Storage;
  
public:
  // min_val means all pts are >= min_val
  IntBucket(int min_val = 0, int bucket_width = 1);
  ~IntBucket() {}

  void clear() { counts.clear(); }

  void addPoint(int val);

  double getMean(int skip_initial = 0) const;

  friend std::ostream& operator<<(std::ostream& os, const IntBucket& bucket);
  
private:
  int getBIdxForVal(int val) const { return (val - min_val) / bucket_width; }
  int getBeginValForBIdx(int bidx) const { return bidx * bucket_width + min_val; }
  // one past the end of this bucket;
  int getEndValForBIdx(int bidx) const { return getBeginValForBIdx(bidx + 1); }
  
  int min_val;
  int bucket_width;
  Storage counts;
};



#endif
