/* -*- Mode: C++ -*- */

/*
    libscoach: Soccer Coach library for use with the SoccerServer system 

    Copyright (C) 2001  Patrick Riley, Paul Carpenter, Gal Kaminka, Manuela Veloso
    Copyright (C) 1998,1999,2000 Patrick Riley, Peter Stone, Manuela Veloso

    This library is free software; you can redistribute it and/or
    modify it under the terms of the GNU Lesser General Public
    License as published by the Free Software Foundation; either
    version 2.1 of the License, or (at your option) any later version.

    This library is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
    Lesser General Public License for more details.

    You should have received a copy of the GNU Lesser General Public
    License along with this library; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

*/
/* Classes dealing with statistical stuff from data points */


#include <iomanip>
#include <math.h>
#include "data.h"
#include "misc.h"
#include "Logger.h"

using namespace spades;

SingleDataSummary::SingleDataSummary()
{
  reset();
  precision = 2;
}

void
SingleDataSummary::reset() 
{
  n = 0;
  sum = 0.0;
  sum_squares = 0.0;
  min = HUGE;
  max = -HUGE;
}

void
SingleDataSummary::addPoint(double x)
{
  n++;
  sum += x;
  sum_squares += x*x;
  min = (x < min) ? x : min;
  max = (x > max) ? x : max;
}

double
SingleDataSummary::getMean() const
{
  if (n==0)
    return 0.0;
  else
    return sum/n;
}
				 
double
SingleDataSummary::getVariance() const
{
  if (n==1 || n==0)
    return 0;
  else
    return ((double)n*sum_squares - sum*sum) / ((double)n * (n-1));
}


double
SingleDataSummary::getStdDev() const
{
  if (round(getVariance(), -5) < 0) {
    errorlog << "How is variance negative: " << getVariance() << ende;
  }  
  return (sqrt(round(getVariance(),-5)));
}


int
SingleDataSummary::getNumPoints() const
{
  return n;
}

void
SingleDataSummary::combineWith(const SingleDataSummary* sds)
{
  n += sds->n;
  sum += sds->sum;
  sum_squares += sds->sum_squares;
}




void
SingleDataSummary::writeInfoToFile(std::ostream& output_file,
				   bool WithNumPoints) const
{
  if (WithNumPoints)
    output_file << "NumPoints: " << getNumPoints() << "\t";
  output_file << std::setprecision(precision) << "Mean: " << getMean()
	      << std::setprecision(precision) << "\tVariance: " << getVariance()
	      << std::setprecision(precision) << "\tStdDev: " << getStdDev()
	      << std::endl;
}

void
SingleDataSummary::writeCompactInfoToFile(std::ostream& output_file,
					  bool WithNumPoints) const
{
  if (WithNumPoints)
    output_file << getNumPoints() << "\t";
  output_file << std::setprecision(precision) << getMean() << "\t"
	      << std::setprecision(precision) << getVariance() << "\t"
	      << std::setprecision(precision) << getStdDev();
}

void
SingleDataSummary::writeCompactMeanStdevToFile(std::ostream& output_file,
					       bool WithNumPoints) const
{
  if (WithNumPoints)
    output_file << getNumPoints() << "\t";
  output_file << std::setprecision(precision) << getMean() << "\t"
	      << std::setprecision(precision) << getStdDev();
}


bool
SingleDataSummary::readInfoFromFile(std::istream& infile, int num_points)
{
  if (num_points == -1)
    {
      infile.ignore(1000, ':');
      if (!infile)
	return false;
      infile >> num_points;
    }

  n = num_points;
  double mean, var, stdev;
  infile.ignore(1000, ':');
  if (!infile) return false;
  infile >> mean;
  infile.ignore(1000, ':');
  if (!infile) return false;
  infile >> var;
  infile.ignore(1000, ':');
  if (!infile) return false;
  infile >> stdev;

  if (infile.fail())
    return false;

  sum = mean * n;
  sum_squares = (var * n * (n-1) + sum*sum) / n;

  return true;
}

bool
SingleDataSummary::readCompactInfoFromFile(std::istream& infile, int num_points)
{
  if (num_points == -1)
    infile >> num_points;

  n = num_points;
  double mean, var, stdev;
  infile >> mean >> var >> stdev;

  if (infile.fail())
    return false;
  
  sum = mean * n;
  sum_squares = (var * n * (n-1) + sum*sum) / n;

  /*printf("line: %s, I got: %d %g %g %g, %g %g %g\n", line,
	 n, mean, var, stdev, 
	 getMean(), getVariance(), getStdDev());*/
  
  return true;
}

/***********************************************************/


//Only maintains info for max and mean
void
MaxMeanSummary::reset()
{
  sum = 0.0;
  n = 0;
  max = -HUGE;
}
  
void
MaxMeanSummary::addPoint(double x)
{
  sum += x;
  n++;
  if (x > max)
    max = x;
}

double
MaxMeanSummary::getMean()
{
  if (n == 0) {
    errorlog << "Tried to get mean with 0 points!" << ende;
    return 0.0;
  } 
  return sum / n;
}

/***********************************************************/

void
MinMaxSummary::addPoint(double x)
{
  if (x < min) {
    min = x;
    mindex = curr_index;
  }
  if (x > max) {
    max = x;
    maxdex = curr_index;
  }
  curr_index++;
}

/****************************************************************************************/


RecordBestN::RecordBestN(int N)
{
  this->N = N;
  num_stored = 0;
  arrData = new void*[N];
  arrScore = new float[N];
}

RecordBestN::~RecordBestN()
{
  delete [] arrData;
  delete [] arrScore;
}

  
void RecordBestN::Clear()
{
  num_stored = 0;
}

//0th best is the best
void RecordBestN::AddPoint (void* data, float score)
{
  bool found_spot = false;
  void* held_data = data;
  float held_score = score;
  
  for (int i=0; i<num_stored; i++) {
    if (score > arrScore[i])
      found_spot = true;
    
    if (found_spot) {
      void* tmp_data = arrData[i];
      float tmp_score = arrScore[i];
      arrData[i] = held_data;
      arrScore[i] = held_score;
      held_data = tmp_data;
      held_score = tmp_score;
    }
  }

  if (num_stored < N) {
    arrData[num_stored] = held_data;
    arrScore[num_stored] = held_score;
    num_stored++;
  }
}

/****************************************************************************************/

IntBucket::IntBucket(int min_val, int bucket_width)
  : min_val(min_val), bucket_width(bucket_width), counts()
{
  if (bucket_width <= 0)
    {
      errorlog << "IntBucket: bad width " << bucket_width << ende;
      bucket_width = 1;
    }
}


void
IntBucket::addPoint(int val)
{
  int bidx = getBIdxForVal(val);
  if ((signed)counts.size() < (bidx+1))
    {
      counts.resize(bidx+1);
    }
  counts[bidx]++;
}


double
IntBucket::getMean(int skip_initial) const
{
  int cnt = 0;
  double total = 0;
  for (int bidx = skip_initial;
       bidx < (signed)counts.size();
       ++bidx)
    {
      cnt += counts[bidx];
      float bucket_val = (getBeginValForBIdx(bidx) + getEndValForBIdx(bidx) - 1) / 2.0;
      total += counts[bidx] * bucket_val;
    }
  return total / cnt;
}


//friend
std::ostream&
operator<<(std::ostream& os, const IntBucket& bucket)
{
  for (int bidx = 0;
       bidx < (signed)bucket.counts.size();
       ++bidx)
    {
      os << bucket.getBeginValForBIdx(bidx) << ' '
         << bucket.getEndValForBIdx(bidx) << ' '
         << bucket.counts[bidx]
         << std::endl;
    }

  return os;
}

