/* This is the main client program for the coach */

#include <iostream>
#include <signal.h>
#include "CoachParam.h"
#include "CoachTagFunction.h"
#include "version.h"
#include "ModuleRegistry.h"
#include "test.h"
//Now all the modules
#include "ModTest.h"
#include "ModCMRep.h"
#include "ModFormation.h"
#include "ModSetplayPlan.h"
#include "ModStatImage.h"
#include "ModAbstract.h"
#include "ModFeatures.h"
#include "ModTrainer.h"
#include "ModGlobalAdapt.h"
//Now the runners
#include "SingleCallRunner.h"
#include "OnlineRunner.h"
#include "LogfileRunner.h"

using namespace spades;
using namespace std;

// this is static so that signal handlers can get to it
static Runner* pRunner = NULL;

void
sigterm_handler(int) 
{
  warninglog(10) << "Got some sort of term signal, shutting down" << ende;
  if (pRunner)
    pRunner->initiateShutdown();
}

void
init_handler_term()
{
  struct sigaction sigact;
  sigact.sa_flags = 0;

  //This makes sure that the handler is only called once for each signal
  sigact.sa_flags |= SA_RESETHAND;
  
  sigemptyset(&sigact.sa_mask);
  sigact.sa_handler = &sigterm_handler;
  sigaction(SIGINT, &sigact, NULL);
  sigaction(SIGQUIT, &sigact, NULL);
  sigaction(SIGTERM, &sigact, NULL);
}



bool
initializeAllModules(ModuleRegistry* pReg)
{
  ModTest::initialize(pReg);
  ModFeatures::initialize(pReg);
  ModTrainer::initialize(pReg);
  ModGlobalAdapt::initialize(pReg);
  ModFormation::initialize(pReg);
  ModSetplayPlan::initialize(pReg);
  ModStatImage::initialize(pReg);
  ModAbstract::initialize(pReg);
  // we want ModCMRep to be late so that it gets messages from everyone
  ModCMRep::initialize(pReg);

  return true;
}

/*******************************************************************/
int
main (int argc, const char* const* argv)
{
  
  cout << "OWL Coach, version " << g_VersionString << endl;
  cout << g_CopyrightString << endl;

  init_handler_term();
  
  CoachParam* pParam;
  pParam = new CoachParam;
  pParam->getOptions(argc, argv);

  seedRandom();
  
  CoachTagFunction *pTagFunc = new CoachTagFunction;

  Logger::instance()->setTagFunction(pTagFunc);
  Logger::instance()->setMaxActionLogLevel(CoachParam::instance()->getActionLogLevel());
  Logger::instance()->openFiles(CoachParam::instance()->getLogfileDir(),
				CoachParam::instance()->getActionLogFN());

  //testError(); return 0;
  //testErrBoundValue(); return 0;
  //testVectorIO(); return 0;
  //testFormationIO(); return 0;
  //testFieldImage(); return 0;
  //testDrawPoints(); return 0;
  //testDrawRegions(); return 0;
  //testGD(); return 0;
  //testPlayerOccupancy(); return 0;
  //testRegions(); return 0;
  //testBallGridFactor(); return 0;
  //testMarkovChain(); return 0;
  //testMarkovChainReIdx(); return 0;
  //testMarkovChainValue(); return 0;
  //testMarkovChainValueLearn(); return 0;
  //testFactorOr(); return 0;
  //testGraphWriter(); return 0;
  //testASDiffPattern(); return 0;
  //testASDiffFeature(); return 0;
  //testASDiffClassifier(); return 0;
  //testMDP(); return 0;
  //testTransitionSorter(); return 0;
  //testQTable(); return 0;
  //testRectBetween(); return 0;
  //testStateValue(); return 0;
  //testAdviceTree(); return 0;
  //testAdviceTreeCreation(); return 0;
  //testCircleIntersect(); return 0;
  //testCircleRectIntersect(); return 0;
  //testBinaryFile(); return 0;
  //testBinQTable(); return 0;
  //testPOElem(); return 0;
  //testBinMDP(); return 0;
  //testTypeInfo(); return 0;
  //testIntBucket(); return 0;
  //testDrawMisc(); return 0;
  
  ModuleRegistry* pReg = new ModuleRegistry;
  if (initializeAllModules(pReg))
    {

      switch (CoachParam::instance()->getRunMode())
	{
	case RM_Invalid:
	  errorlog << "Got run mode invalid" << ende;
	  break;
	case RM_Null:
	  actionlog(10) << "Got Null run mode, so doing nothing" << ende;
	  break;
	case RM_Single:
	  actionlog(10) << "Running in single call mode" << ende;
	  pRunner = new SingleCallRunner(pReg);
	  break;
	case RM_Online:
	  actionlog(10) << "Running in online mode" << ende;
	  pRunner = new OnlineRunner(pReg);
	  break;
	case RM_Logfile:
	  actionlog(10) << "Running in logfile mode" << ende;
	  pRunner = new LogfileRunner(pReg);
	  break;
	}

      if (!pRunner)
	{
	  errorlog << "Did not get a runner allocated, exiting" << ende;
	}
      else
	{
	  if (pRunner->dependencyCheck(pReg))
	    {
	      cout << "All modules initialized" << endl;

	      pRunner->run();

	      cout << "Runner exited, cleaning up" << endl;
	      actionlog(100) << "Runner exited, cleaning up" << ende;
	    }
	  else
	    {
	      errorlog << "Failed in dependency check of modules, exiting" << ende;
	    }
      
	  delete pRunner;
	}

    }
  else
    {
      errorlog << "Failed in initialization of modules, exiting" << ende;
    }
  
  delete pReg;
  
  delete CoachParam::instance();

  Logger::removeInstance();
  
  return 0;

}
