/* -*- Mode: C++ -*- */

/* These classes describe the state of the world */


#ifndef _WORLD_STATE_H_
#define _WORLD_STATE_H_

#include <map>
#include "types.h"
#include "Geometry.h"
#include "ErrBoundValue.h"
#include "soccer_utils.h"
#include "types.h"
#include "FieldImage.h"

// to prevent everyone needing "region.h"
namespace rcss
{
  namespace clang
  {
    class Region;
  }
}

/*******************************************************************************/

class ObjectInfo
{
public:
  ObjectInfo() : pos_x(), pos_y(), vel_x(), vel_y() {}
  ObjectInfo(VecPosition pos, VecPosition vel)
    : pos_x(pos.getX()), pos_y(pos.getY()), vel_x(vel.getX()), vel_y(vel.getY()) {}
  ObjectInfo(VecPosition pos, VecPosition pos_err, VecPosition vel, VecPosition vel_err)
    : pos_x(pos.getX(), pos_err.getX()), pos_y(pos.getY(), pos_err.getY()),
      vel_x(vel.getX(), vel_err.getX()), vel_y(vel.getY(), vel_err.getY())
  {}
  virtual ~ObjectInfo() {}

  VecPosition getPos() const { return VecPosition(pos_x.getVal(), pos_y.getVal()); }
  VecPosition getVel() const { return VecPosition(vel_x.getVal(), vel_y.getVal()); }

  virtual void clear() { pos_x.clear(); pos_y.clear(); vel_x.clear(); vel_y.clear(); }

  friend std::ostream & operator << (std::ostream & o, const ObjectInfo& i)
  { o << "Object[pos=(" << i.pos_x << ", " << i.pos_y << ")"
      << ", vel=(" << i.vel_x << ", " << i.vel_y << ")" << "]"; 
  return o; }

private:
  ErrBoundValue pos_x;
  ErrBoundValue pos_y;
  ErrBoundValue vel_x;
  ErrBoundValue vel_y;
};

/*******************************************************************************/

class BallInfo
  : public ObjectInfo
{
public:
  BallInfo() : ObjectInfo() {}
  BallInfo(VecPosition pos, VecPosition vel)
    : ObjectInfo(pos, vel) {}
  BallInfo(VecPosition pos, VecPosition pos_err, VecPosition vel, VecPosition vel_err)
    : ObjectInfo(pos, pos_err, vel, vel_err) {}
  ~BallInfo() {}

private:
};

/*******************************************************************************/

class PlayerInfo
  : public ObjectInfo
{
public:
  PlayerInfo()
    : ObjectInfo(), side(TS_None), num(-1), body_ang(), neck_ang(), goalie(false) {}
  PlayerInfo(TeamSide side, Unum num,
	     VecPosition pos, VecPosition pos_err, VecPosition vel, VecPosition vel_err,
	     ErrBoundValue body_ang, ErrBoundValue neck_ang,
	     bool goalie = false)
    : ObjectInfo(pos, pos_err, vel, vel_err),
      side(side), num(num), body_ang(body_ang), neck_ang(neck_ang),
      goalie(goalie)
  {}
  ~PlayerInfo() {}

  TeamSide getSide() const { return side; }
  Unum     getNum() const { return num; }
  ErrBoundValue getBodyAng() const { return body_ang; }
  ErrBoundValue getNeckAng() const { return neck_ang; }
  bool          getGoalie() const { return goalie; }

  void clear();

  friend std::ostream & operator << (std::ostream & o, const PlayerInfo& i)
  { o << "Player(s=" << i.side << ",n=" << i.num << ", " << (ObjectInfo)i
      << ",body=" << i.body_ang << ",neck=" << i.neck_ang << ",g=" << i.goalie;
  return o; }

private:
  TeamSide         side;
  Unum             num;
  ErrBoundValue    body_ang;
  ErrBoundValue    neck_ang; /* relative to body angle */
  bool             goalie;
};

/*******************************************************************************/

class WorldState
{
public:
  WorldState();
  ~WorldState() {}

  typedef std::map<Unum, PlayerInfo> PlayerStorage;

  //Accessor methods
  Time getTime() const { return time; }
  PlayMode getPlayMode() const { return play_mode; }

  //returns NULL if no info for that player
  int getNumPlayers(TeamSide team) const;
  const PlayerInfo* getPlayer(TeamSide s, Unum n) const;
  const PlayerInfo* getPlayer(int idx) const;

  const BallInfo& getBall() const { return ball; }
  int getLeftScore() const { return left_score; }
  int getRightScore() const { return right_score; }
  int getScore(TeamSide ts) const;
  
  //various "synthesis" info
  Unum getGoalieNum(TeamSide s) const;
  // if TS_Both is use, right players are negative
  Unum getClosestPlayerToPoint(TeamSide side, VecPosition point, double* pdist) const;
  int countPlayersInRegion(TeamSide side,
			   rcss::clang::Region* r,
			   TeamSide my_side, //need for relative points
			   const VarBindings& bindings) const;
  
  //Clearing information
  void clearTransientValues() { clearTime(); clearPlayers(); clearBall(); }
  void clearTime() { time = -1; }
  void clearPlayers() { left_players.clear(); right_players.clear(); }
  void clearBall() { ball.clear(); }

  //Setting of stuff
  void setTime(Time t) { time = t; }
  void setPlayMode(PlayMode pm) { play_mode = pm; }
  void setPlayer(const PlayerInfo& p);
  void setBall(const BallInfo& b) { ball = b; }
  void setScore(TeamSide ts, int s);
  void setScores(int left, int right) { setLeftScore(left); setRightScore(right); }
  void setLeftScore(int s) { left_score = s; }
  void setRightScore(int s) { right_score = s; }

  // Other
  void setDefaultBeginGame();

  // Drawing
  void draw(FieldImage* pfi) const 
  { draw(pfi, DEFAULT_LEFT_COLOR, DEFAULT_RIGHT_COLOR, DEFAULT_BALL_COLOR); }
  void draw(FieldImage* pfi, FieldImage::Color c_left, FieldImage::Color c_right) const
  { draw(pfi, c_left, c_right, DEFAULT_BALL_COLOR); }    
  void draw(FieldImage* pfi, FieldImage::Color c_left, FieldImage::Color c_right,
	    FieldImage::Color c_ball) const;
  
  // friend functions
  friend std::ostream & operator << (std::ostream & o, const WorldState& ws);
  
private:
  const PlayerInfo* getPlayer(const PlayerStorage& store, Unum n) const;
  PlayerInfo* getPlayer(PlayerStorage& store, Unum n);
  Unum getGoalieNum(const PlayerStorage& s) const;

  int countPlayersInRegion(const PlayerStorage& store,
			   rcss::clang::Region* r,
			   TeamSide my_side, //need for relative points
			   const VarBindings& bindings) const;

  Time        time;
  PlayMode    play_mode;
  PlayerStorage left_players;
  PlayerStorage right_players;
  BallInfo    ball;
  int         left_score;
  int         right_score;

  static const FieldImage::Color DEFAULT_LEFT_COLOR;
  static const FieldImage::Color DEFAULT_RIGHT_COLOR;
  static const FieldImage::Color DEFAULT_BALL_COLOR;
};



#endif
