/* -*- Mode: C++ -*- */

/* These classes describe the state of the world */

#include <sstream>
#include "WorldState.h"
#include "region.h"
#include "WorldStateWMI.h"
#include "Logger.h"

using namespace spades;
using namespace std;


/*************************************************************************/

void
PlayerInfo::clear()
{
  ObjectInfo::clear();
  side = TS_None;
  num = -1;
  body_ang.clear();
  neck_ang.clear();
  goalie = false;
}


/*************************************************************************/

WorldState::WorldState()
  : time(-1), play_mode(PM_Null), left_players(), right_players(), ball(),
    left_score(-1), right_score(-1)
{
}


int
WorldState::getNumPlayers(TeamSide team) const
{
  int cnt = 0;
  if (team == TS_Left || team == TS_Both)
    cnt += left_players.size();
  if (team == TS_Right || team == TS_Both)
    cnt += right_players.size();

  return cnt;
}

const PlayerInfo*
WorldState::getPlayer(TeamSide s, Unum n) const
{
  switch (s)
    {
    case TS_Left:
      return getPlayer(left_players, n);
    case TS_Right:
      return getPlayer(right_players, n);
    default:
      errorlog << "getPlayer: what is side? " << s << ende;
    }
  
  return NULL;
}

const PlayerInfo*
WorldState::getPlayer(int idx) const
{
  int left_cnt = left_players.size();
  if (idx < left_cnt)
    {
      PlayerStorage::const_iterator iter = left_players.begin();
      advance(iter, idx);
      return &(iter->second);
    }
  int right_cnt = right_players.size();
  if (idx < left_cnt + right_cnt)
    {
      PlayerStorage::const_iterator iter = right_players.begin();
      advance(iter, idx - left_cnt);
      return &(iter->second);
    }
  return NULL;
}

Unum
WorldState::getGoalieNum(TeamSide s) const
{
  switch (s)
    {
    case TS_Left: return getGoalieNum(left_players);
    case TS_Right: return getGoalieNum(right_players);
    default:
      errorlog << "Asked to get goalie num for bad side: " << s << ende;
    }
  return -1;
}

const PlayerInfo*
WorldState::getPlayer(const PlayerStorage& store, Unum n) const
{
  PlayerStorage::const_iterator iter = store.find(n);
  if (iter == store.end())
    return NULL;
  return &(iter->second);
}

PlayerInfo*
WorldState::getPlayer(PlayerStorage& store, Unum n)
{
  PlayerStorage::iterator iter = store.find(n);
  if (iter == store.end())
    return NULL;
  return &(iter->second);
}

Unum
WorldState::getGoalieNum(const PlayerStorage& s) const
{
  for (PlayerStorage::const_iterator iter = s.begin();
       iter != s.end();
       iter++)
    {
      if (iter->second.getGoalie())
	return iter->second.getNum();
    }
  return -1;
}

int
WorldState::getScore(TeamSide ts) const
{
  switch (ts)
    {
    case TS_Left: return left_score;
    case TS_Right: return right_score;
    default:
      errorlog << "WorldState::getScore: I don't understand side " << ts << ende;
    }
  return -1;
}


Unum
WorldState::getClosestPlayerToPoint(TeamSide side, VecPosition point, double* pdist) const
{
  *pdist = HUGE;
  Unum player = Unum_Unknown;
  float tmp_dist;

  if (side == TS_Left || side == TS_Both)
    {
      for (PlayerStorage::const_iterator iter = left_players.begin();
	   iter != left_players.end();
	   iter++)
	{
	  tmp_dist = iter->second.getPos().getDistanceTo(point);  
	  if ( tmp_dist < *pdist )
	    {
	      *pdist = tmp_dist;
	      player = iter->second.getNum();
	    }
	}
    }
  if (side == TS_Right || side == TS_Both)
    {
      for (PlayerStorage::const_iterator iter = left_players.begin();
	   iter != left_players.end();
	   iter++)
	{
	  tmp_dist = iter->second.getPos().getDistanceTo(point);  
	  if ( tmp_dist < *pdist )
	    {
	      *pdist = tmp_dist;
	      player = iter->second.getNum();
	      if (side == TS_Both)
		player *= -1;  // If getting both, then right players are negative
	    }
	}
    }
    
  return player;
}

  
int
WorldState::countPlayersInRegion(TeamSide side,
				 rcss::clang::Region* r,
				 TeamSide my_side, //need for relative points
				 const VarBindings& bindings) const
{
  int count = 0;
  switch (side)
    {
    case TS_None:
      break;
    case TS_Left:
      count += countPlayersInRegion(left_players, r, my_side, bindings);
      break;
    case TS_Right:
      count += countPlayersInRegion(right_players, r, my_side, bindings);
      break;
    case TS_Both:
      count += countPlayersInRegion(left_players, r, my_side, bindings);
      count += countPlayersInRegion(right_players, r, my_side, bindings);
      break;
    default:
      errorlog << "countPlayersInRegion: what is side? " << side << ende;
      break;
    }
  return count;
}

int
WorldState::countPlayersInRegion(const PlayerStorage& store,
				 rcss::clang::Region* r,
				 TeamSide my_side, //need for relative points
				 const VarBindings& bindings) const
{
  int count = 0;
  //Note also that the regions are always assumed to be in left-hand coordinates!
  // Therefore, when my_side is TS_Right, the WorldStateRWMI flips the coordinates
  // and tells the region processing functions to flip themselves also
  WorldStateRWMI rwmi(*this, my_side);
  for (PlayerStorage::const_iterator iter = store.begin();
       iter != store.end();
       iter++)
    {
      VecPosition pos = iter->second.getPos();
      if (my_side == TS_Right)
	pos = pos.flipCoords();
      if (r->isPtIn(pos, &rwmi, bindings))
	count++;
    }
  return count;
}



void
WorldState::setPlayer(const PlayerInfo& p)
{
  switch (p.getSide())
    {
    case TS_Left:
      left_players[p.getNum()] = p;
      break;
    case TS_Right:
      right_players[p.getNum()] = p;
      break;
    default:
      errorlog << "setPlayer: what is side? " << p.getSide() << ende;
      break;
    }
}

void
WorldState::setScore(TeamSide ts, int s)
{
  switch (ts)
    {
    case TS_Left: left_score = s; break;
    case TS_Right: right_score = s; break;
    default:
      errorlog << "WorldState::setScore: I don't understand side " << ts << ende;
    }
}

void
WorldState::setDefaultBeginGame()
{
  time = 0;
  play_mode = PM_BeforeKickOff;
  left_players.clear();
  right_players.clear();
  ball = BallInfo(VecPosition(0,0), 0.0, VecPosition(0,0), 0.0);
  left_score = right_score = 0;
}

class PlayerDrawer
{
public:
  PlayerDrawer(FieldImage* pfi, FieldImage::Color& c) : pfi(pfi), c(c) {}

  void operator() (const std::pair<const Unum, PlayerInfo>& pi)
  {
    pfi->addPoint(pi.second.getPos(), c);
    std::ostringstream o;
    o << pi.first;
    pfi->addText(o.str().c_str(), pi.second.getPos());
  }
  
  FieldImage* pfi;
  FieldImage::Color c;
};

void
WorldState::draw(FieldImage* pfi,
		 FieldImage::Color c_left, FieldImage::Color c_right,
		 FieldImage::Color c_ball) const
{
  PlayerDrawer drawer_left(pfi, c_left);
  for_each(left_players.begin(), left_players.end(), drawer_left);
  PlayerDrawer drawer_right(pfi, c_right);
  for_each(right_players.begin(), right_players.end(), drawer_right);

  // The 2 makes the ball a little bigger
  pfi->addPoint(getBall().getPos(), c_ball, 2);

  ostringstream ostr_pmode;
  ostr_pmode << getPlayMode();
  pfi->addLegendLine(ostr_pmode.str().c_str());

  ostringstream ostr_score;
  ostr_score << getLeftScore() << " - " << getRightScore();
  pfi->addLegendLine(ostr_score.str().c_str());
}


std::ostream&
operator << (std::ostream & o, const WorldState& ws)
{
  o << "WorldState:" << endl
    << "\ttime=" << ws.time << endl
    << "\tplay_mode=" << ws.play_mode << endl
    << "\tplayers:" << endl;
  const PlayerInfo* p;
  for (int idx = 0;
       (p = ws.getPlayer(idx)) != NULL;
       idx++)
    {
      o << "\t\t" << *p << endl;
    }
  o << "\tball=" << ws.ball << endl
    << "\tscores=" << ws.left_score << ' ' << ws.right_score << endl;

  return o;  
}

const FieldImage::Color WorldState::DEFAULT_LEFT_COLOR("#ffccoo");
const FieldImage::Color WorldState::DEFAULT_RIGHT_COLOR("#ff0000");
const FieldImage::Color WorldState::DEFAULT_BALL_COLOR("#0000cc");
