/* -*- Mode: C++ -*- */

/* This class provides storage for an arbitrary value associated with a state */

#ifndef _STATE_VALUE_H_
#define _STATE_VALUE_H_

#include <iostream>
#include <algorithm>
#include "utility.h"

/* Since this is templated, we have to include the implementation here now */
/* values are read and written sorted so you don't have to do that work again
   if you don't change anything */
//Value must be assignable, comparable with <, and support the iostream << and >> ops
template<class Value>
class StateValue
{
public:
  class StateValuator
  {
  public:
    StateValuator() {}
    virtual ~StateValuator() {}

    virtual Value getStateValue(int sidx) = 0;
  };
    
  static StateValue<Value>* create(int num_states, StateValuator* f)
  {
    StateValue<Value>* p = new StateValue<Value>(num_states);
    for (int sidx = 0; sidx < num_states; sidx++)
      p->setValue(sidx, f->getStateValue(sidx));
    return p;
  }
  
public:
  StateValue(int num_states = 0)
    : sorted(false), storage(num_states) {}
  ~StateValue() {}

  void setNumStates(int num_states)
  { storage.resize(num_states); sorted = false; }
  int getNumStates()
  { return storage.size(); }
  
  void setValue(int sidx, const Value& v)
  { storage[sidx] = v; sorted = false; }
  const Value& getValue(int sidx) const
  { return storage[sidx].getValue(); }

  // returns the rank of the state; 0 is the lowest value
  int getRank(int sidx)
  { checkSort(); return storage[sidx].getRank(); }

  
private:
  class SortElement;
  class Element
  {
  public:
    Element()
      : val(), rank(-1) {}
    Element(const Value& val, int rank = -1)
      : val(val), rank(rank) {}
    Element(SortElement& e)
      : val(e.val), rank(e.rank) {}
    ~Element() {}

    const Value& getValue() const { return val; }
    Value& getValue() { return val; }
    void setValue(const Value& v) { val = v; }

    int getRank() const { return rank; }
    void setRank(int r) { rank = r; }
    
    bool operator<(const Element& e) const { return rank < e.rank; }

    friend std::ostream& operator<<(std::ostream& os, const Element& e)
    {
      os << e.val << ' ' << e.rank;
      return os;
    }
    friend std::istream& operator>>(std::istream& is, Element& e)
    {
      is >> e.val >> e.rank;
      return is;
    }
    
  private:
    Value val;
    int rank;
  };

  typedef std::vector<Element> Storage;

  class SortElement
  {
  public:
    SortElement(int sidx, Element& e) : sidx(sidx), val(e.getValue()), rank(-1) {}
    SortElement() : sidx(-1), val(), rank(-1) {}

    bool operator<(const SortElement& e) const { return val < e.val; }
    
    friend std::ostream& operator<<(std::ostream& os, const SortElement& e)
    {
      os << e.sidx << ' ' << e.val;
      return os;
    }
    friend std::istream& operator>>(std::istream& is, SortElement& e)
    {
      is >> e.sidx >> e.val;
      return is;
    }

    int sidx;
    Value val;
    int rank;
  };
  
  typedef std::vector<SortElement> SortStorage;

  void doSort(SortStorage& sort_storage)
  {
    sort_storage.clear();
    for (typename Storage::iterator iter = storage.begin();
	 iter != storage.end();
	 iter++)
      sort_storage.push_back(SortElement(iter - storage.begin(), *iter));

    std::sort(sort_storage.begin(), sort_storage.end());
  }
  
  
  void checkSort()
  {
    if (sorted)
      return;

    SortStorage sort_storage;
    doSort(sort_storage);
    for (typename SortStorage::iterator iter = sort_storage.begin();
	 iter != sort_storage.end();
	 iter++)
      storage[iter->sidx].setRank(iter - sort_storage.begin());

    sorted = true;
  }
    
public:
  friend std::ostream& operator<<(std::ostream& os, StateValue& sv)
  {
    os << "# This represents a StateValue object" << std::endl;
    os << "# The elements here are in rank order by value, lowest to highest" << std::endl;
    os << "# Format: <state idx> <value>" << std::endl;
    sv.checkSort();
    SortStorage sort_storage;
    sv.doSort(sort_storage);
    os << sort_storage.size() << std::endl;
    std::copy(sort_storage.begin(), sort_storage.end(), std::ostream_iterator<SortElement>(os, "\n"));
    os << std::endl;
    return os;
  }
  
  friend std::istream& operator>>(std::istream& is, StateValue& sv)
  {
    sv.setNumStates(0);
    int size;
    if (!spades::skip_to_non_comment(is))
      return is;
    is >> size;
    if (is.fail())
      return is;
    sv.storage.resize(size);
    for (int i = 0; i < size; i++)
      {
	SortElement se;
	is >> se;
	if (is.fail())
	  return is;
	se.rank = i;
	Element e(se);
	sv.storage[se.sidx] = e;
      }
    sv.sorted = true;
    return is;
  }

private:
  bool sorted;
  Storage storage;
};


#endif

