/* -*- Mode: C++ -*- */

/* These is a spceialization of MDP for Soccer */

#ifndef _SOCCER_MDP_H_
#define _SOCCER_MDP_H_

#include "MDP.h"
#include "AbstractState.h"
#include "clangaction.h"

class BallGridFactor; // defined in AbstractStateElement
class AdviceTreeAction;

/*****************************************************************************/
/*****************************************************************************/
/*****************************************************************************/

class SoccerMDPAction;
class SoccerMDPActionTest;
class SoccerMDPActionHoldBall;
class SoccerMDPActionPassTo;
class SoccerMDPActionDribbleTo;
class SoccerMDPActionClear;
class SoccerMDPActionShoot;
class SoccerMDPActionCross;
class SoccerMDPActionTheirAct;
class SoccerMDPActionNull;

enum TypesSoccerMDPAction {
  SMDPA_Test,
  SMDPA_HoldBall,
  SMDPA_PassTo,
  SMDPA_DribbleTo,
  SMDPA_Clear,
  SMDPA_Shoot,
  SMDPA_Cross,
  SMDPA_TheirAct,
  SMDPA_Null,
  SMDPA_MAX
};

  
// a virtual base class to represent an action
class SoccerMDPAction
  : public MDPAction
{
public:
  SoccerMDPAction() : MDPAction() {}
  virtual ~SoccerMDPAction() {}

  //we're leaving the print method virtual

  virtual void readParams(std::istream& is) = 0;
  virtual bool readParams(BinaryFileReader& reader) = 0;

  // This allows the action to examine the AbstractStateDescription and pull
  // whatever information is necessary out of it
  virtual void processDescription(AbstractStateDescription* pdesc) {}
  
  // a return of false means this transition is not a valid "primary"
  // transition
  virtual bool setParamsFromTransition(AbstractStateDescription* pdesc,
				       int first_state,
				       int second_state) { return true; }

  // setParams should have already been called before this
  // This should be used for the secondary transitions 
  virtual bool acceptTran(AbstractStateDescription* pdesc,
			  int first_state,
			  int second_state)
  { return true; }
  
  virtual bool tryUnify(SoccerMDPAction* pact) = 0;

  virtual bool doesCreateCLangAction() = 0;
  virtual rcss::clang::Action* createAction(AbstractStateDescription* pdesc,
					    int state_idx) = 0;
  AdviceTreeAction* createAdviceTreeAction(AbstractStateDescription* pdesc,
					   int state_idx);
  
  
public:
  // This is a double dispatch scheme
  virtual bool tryUnifyDispatch2(SoccerMDPActionTest* pact) { return false; }
  virtual bool tryUnifyDispatch2(SoccerMDPActionHoldBall* pact) { return false; }
  virtual bool tryUnifyDispatch2(SoccerMDPActionPassTo* pact) { return false; }
  virtual bool tryUnifyDispatch2(SoccerMDPActionDribbleTo* pact) { return false; }
  virtual bool tryUnifyDispatch2(SoccerMDPActionClear* pact) { return false; }
  virtual bool tryUnifyDispatch2(SoccerMDPActionShoot* pact) { return false; }
  virtual bool tryUnifyDispatch2(SoccerMDPActionCross* pact) { return false; }
  virtual bool tryUnifyDispatch2(SoccerMDPActionTheirAct* pact) { return false; }
  virtual bool tryUnifyDispatch2(SoccerMDPActionNull* pact) { return false; }

protected:
  // add anything besides the basic action from above
  virtual void addToAdviceTreeAction(AbstractStateDescription* pdesc,
				     int state_idx,
				     AdviceTreeAction* pact)
  {}

public:

  static SoccerMDPAction* createFromStream(std::istream& is);
  static SoccerMDPAction* createFrom(BinaryFileReader& reader);
  static SoccerMDPAction* createByName(const char* name);
  static SoccerMDPAction* createByType(TypesSoccerMDPAction t);
};

/*****************************************************************************/

class SoccerMDPActionTest
  : public SoccerMDPAction
{
public:
  SoccerMDPActionTest() : SoccerMDPAction() {}
  ~SoccerMDPActionTest() {}

  void print (std::ostream& os) const { os << "Test"; }
  bool writeTo(BinaryFileWriter& writer) { return writer.writeChar(SMDPA_Test); }

  void readParams(std::istream& is) {}
  bool readParams(BinaryFileReader& reader) { return true; }

  bool tryUnify(SoccerMDPAction* pact) { return pact->tryUnifyDispatch2(this); }

  bool doesCreateCLangAction() { return false; }
  rcss::clang::Action* createAction(AbstractStateDescription* pdesc,
				    int state_idx)
  { return NULL; }

protected:
  bool tryUnifyDispatch2(SoccerMDPActionTest* pact) { return true; }
};

/*****************************************************************************/

class SoccerMDPActionHoldBall
  : public SoccerMDPAction
{
public:
  SoccerMDPActionHoldBall() : SoccerMDPAction() {}
  ~SoccerMDPActionHoldBall() {}

  void print (std::ostream& os) const { os << "HoldBall"; }
  bool writeTo(BinaryFileWriter& writer) { return writer.writeChar(SMDPA_HoldBall); }

  void readParams(std::istream& is) {}
  bool readParams(BinaryFileReader& reader) { return true; }

  bool tryUnify(SoccerMDPAction* pact) { return pact->tryUnifyDispatch2(this); }

  bool doesCreateCLangAction() { return true; }
  rcss::clang::Action* createAction(AbstractStateDescription* pdesc,
				     int state_idx)
  { return new rcss::clang::ActHold(); }

protected:
  bool tryUnifyDispatch2(SoccerMDPActionHoldBall* pact) { return true; }
};

/*****************************************************************************/
// This is a virtual base class for PassTo and DribbleTo
class SoccerMDPActionBallMovement
  : public SoccerMDPAction
{
public:
  SoccerMDPActionBallMovement(int grid_idx = -1)
    : SoccerMDPAction(), pFac(NULL), grid_idx(grid_idx) {}
  ~SoccerMDPActionBallMovement() {}

  void readParams(std::istream& is);
  bool readParams(BinaryFileReader& reader);

  // THis just reads the BallGridFactor from the description;
  // This is a bit of a hack!
  void processDescription(AbstractStateDescription* pdesc);

  bool setParamsFromTransition(AbstractStateDescription* pdesc,
			       int first_state,
			       int second_state);

  // setParams should have already been called before this
  // This should be used for the secondary transitions 
  bool acceptTran(AbstractStateDescription* pdesc,
		  int first_state,
		  int second_state);

  bool doesCreateCLangAction() { return true; }

protected:
  BallGridFactor* getBallGridFactor() { return pFac; }
  const BallGridFactor* getBallGridFactor() const { return pFac; }
  int getGridIdx() const { return grid_idx; }
  
private:
  BallGridFactor* pFac;
  int grid_idx;
  
};

/*****************************************************************************/

class SoccerMDPActionPassTo
  : public SoccerMDPActionBallMovement
{
public:
  SoccerMDPActionPassTo(int grid_idx = -1)
    : SoccerMDPActionBallMovement(grid_idx) {}
  ~SoccerMDPActionPassTo() {}

  void print (std::ostream& os) const { os << "PassTo (" << getGridIdx() << ")"; }
  bool writeTo(BinaryFileWriter& writer)
  { return writer.writeChar(SMDPA_PassTo) && writer.writeIntAsShort(getGridIdx()); }

  bool tryUnify(SoccerMDPAction* pact) { return pact->tryUnifyDispatch2(this); }

  rcss::clang::Action* createAction(AbstractStateDescription* pdesc,
				    int state_idx);

protected:
  bool tryUnifyDispatch2(SoccerMDPActionPassTo* pact)
  { return getGridIdx() == pact->getGridIdx(); }

  void addToAdviceTreeAction(AbstractStateDescription* pdesc,
			     int state_idx,
			     AdviceTreeAction* pact);
  
};

/*****************************************************************************/

class SoccerMDPActionDribbleTo
  : public SoccerMDPActionBallMovement
{
public:
  SoccerMDPActionDribbleTo(int grid_idx = -1)
    : SoccerMDPActionBallMovement() {}
  ~SoccerMDPActionDribbleTo() {}

  void print (std::ostream& os) const { os << "DribbleTo (" << getGridIdx() << ")"; }
  bool writeTo(BinaryFileWriter& writer)
  { return writer.writeChar(SMDPA_DribbleTo) && writer.writeIntAsShort(getGridIdx()); }

  bool tryUnify(SoccerMDPAction* pact) { return pact->tryUnifyDispatch2(this); }

  rcss::clang::Action* createAction(AbstractStateDescription* pdesc,
				    int state_idx);

protected:
  bool tryUnifyDispatch2(SoccerMDPActionDribbleTo* pact)
  { return getGridIdx() == pact->getGridIdx(); }

};

/*****************************************************************************/

// We'll do a little bit of a funky thing here and clear the square around the
// ball. We really should be doing the grid idx, but oh well
class SoccerMDPActionClear
  : public SoccerMDPAction
{
public:
  SoccerMDPActionClear() : SoccerMDPAction() {}
  ~SoccerMDPActionClear() {}

  void print (std::ostream& os) const { os << "Clear"; }
  bool writeTo(BinaryFileWriter& writer) { return writer.writeChar(SMDPA_Clear); }

  void readParams(std::istream& is) {}
  bool readParams(BinaryFileReader& reader) { return true; }

  bool tryUnify(SoccerMDPAction* pact) { return pact->tryUnifyDispatch2(this); }

  bool doesCreateCLangAction() { return true; }
  rcss::clang::Action* createAction(AbstractStateDescription* pdesc,
				    int state_idx);

protected:
  bool tryUnifyDispatch2(SoccerMDPActionClear* pact) { return true; }
};

/*****************************************************************************/

class SoccerMDPActionShoot
  : public SoccerMDPAction
{
public:
  SoccerMDPActionShoot() : SoccerMDPAction() {}
  ~SoccerMDPActionShoot() {}

  void print (std::ostream& os) const { os << "Shoot"; }
  bool writeTo(BinaryFileWriter& writer) { return writer.writeChar(SMDPA_Shoot); }

  void readParams(std::istream& is) {}
  bool readParams(BinaryFileReader& reader) { return true; }

  bool tryUnify(SoccerMDPAction* pact) { return pact->tryUnifyDispatch2(this); }

  bool doesCreateCLangAction() { return true; }
  rcss::clang::Action* createAction(AbstractStateDescription* pdesc,
				    int state_idx)
  { return new rcss::clang::ActShoot(); }

protected:
  bool tryUnifyDispatch2(SoccerMDPActionShoot* pact) { return true; }
};

/*****************************************************************************/

class SoccerMDPActionCross
  : public SoccerMDPAction
{
public:
  SoccerMDPActionCross() : SoccerMDPAction() {}
  ~SoccerMDPActionCross() {}

  void print (std::ostream& os) const { os << "Cross"; }
  bool writeTo(BinaryFileWriter& writer) { return writer.writeChar(SMDPA_Cross); }

  void readParams(std::istream& is) {}
  bool readParams(BinaryFileReader& reader) { return true; }

  bool tryUnify(SoccerMDPAction* pact) { return pact->tryUnifyDispatch2(this); }

  bool doesCreateCLangAction() { return true; }
  rcss::clang::Action* createAction(AbstractStateDescription* pdesc,
				    int state_idx);

protected:
  bool tryUnifyDispatch2(SoccerMDPActionCross* pact) { return true; }
};

/*****************************************************************************/

class SoccerMDPActionTheirAct
  : public SoccerMDPAction
{
public:
  SoccerMDPActionTheirAct() : SoccerMDPAction() {}
  ~SoccerMDPActionTheirAct() {}

  void print (std::ostream& os) const { os << "TheirAct"; }
  bool writeTo(BinaryFileWriter& writer) { return writer.writeChar(SMDPA_TheirAct); }

  void readParams(std::istream& is) {}
  bool readParams(BinaryFileReader& reader) { return true; }

  bool tryUnify(SoccerMDPAction* pact) { return pact->tryUnifyDispatch2(this); }

  bool doesCreateCLangAction() { return false; }
  rcss::clang::Action* createAction(AbstractStateDescription* pdesc,
				    int state_idx)
  { return NULL; }

protected:
  bool tryUnifyDispatch2(SoccerMDPActionTheirAct* pact) { return true; }
};

/*****************************************************************************/

/* Transitions/actions we can't/won't express in CLang */
class SoccerMDPActionNull
  : public SoccerMDPAction
{
public:
  SoccerMDPActionNull() : SoccerMDPAction() {}
  ~SoccerMDPActionNull() {}

  void print (std::ostream& os) const { os << "Null"; }
  bool writeTo(BinaryFileWriter& writer) { return writer.writeChar(SMDPA_Null); }

  void readParams(std::istream& is) {}
  bool readParams(BinaryFileReader& reader) { return true; }

  bool tryUnify(SoccerMDPAction* pact) { return pact->tryUnifyDispatch2(this); }

  bool doesCreateCLangAction() { return false; }
  rcss::clang::Action* createAction(AbstractStateDescription* pdesc,
				    int state_idx) 
  { return NULL; }

protected:
  bool tryUnifyDispatch2(SoccerMDPActionNull* pact) { return true; }
};

/*****************************************************************************/
// May want to consider an intercept action for times when they have the ball

/*****************************************************************************/
/*****************************************************************************/
/*****************************************************************************/

class SoccerMDP
  : public MDP
{
public:
  SoccerMDP(int num_states = 0);
  ~SoccerMDP();

  MDPAction* createAction(std::istream& is);
  MDPAction* createAction(BinaryFileReader& reader);

  // This processes the AbstractStateDescription so that all the actions
  // can load whatever information they need
  void processDescription(AbstractStateDescription* pdesc);

protected:
  bool writeHeader(BinaryFileWriter& writer) const;
  bool readHeader(BinaryFileReader& reader);
  
private:

  static const char* BIN_FILE_MAGIC;
  static const int BIN_FILE_VERSION;
};

/*****************************************************************************/
/*****************************************************************************/
/*****************************************************************************/

#endif
