/* -*- Mode: C++ -*- */

/* These is a spceialization of MDP for Soccer */

#include <algorithm>
#include <cstring>
#include "SoccerMDP.h"
#include "AbstractStateDifference.h"
#include "AbstractStateElements.h"
#include "AdviceTree.h"
#include "ServerParam.h"
#include "CoachParam.h"
#include "utility.h"
#include "misc.h"
#include "Logger.h"

using namespace spades;

//static
SoccerMDPAction*
SoccerMDPAction::createFromStream(std::istream& is)
{
  std::string name;
  is >> name;
  if (is.fail())
    return NULL;

  SoccerMDPAction* pact = createByName(name.c_str());
  if (!pact)
    return NULL;
  pact->readParams(is);
  return pact;
}

//static
SoccerMDPAction*
SoccerMDPAction::createFrom(BinaryFileReader& reader)
{
  char c_type;
  if (!reader.readChar(&c_type)) return NULL;
  
  SoccerMDPAction* pact = createByType(static_cast<TypesSoccerMDPAction>(c_type));
  if (!pact)
    return NULL;
  pact->readParams(reader);
  return pact;
}
SoccerMDPAction*
SoccerMDPAction::createByName(const char* name)
{
  std::string strname(name);
  std::transform(strname.begin(), strname.end(), strname.begin(), ::tolower);
  
  SoccerMDPAction* pact = NULL;
  if (strname == "test")
    pact = new SoccerMDPActionTest;
  else if (strname == "holdball")
    pact = new SoccerMDPActionHoldBall;
  else if (strname == "passto")
    pact = new SoccerMDPActionPassTo;
  else if (strname == "dribbleto")
    pact = new SoccerMDPActionDribbleTo;
  else if (strname == "clear")
    pact = new SoccerMDPActionClear;
  else if (strname == "shoot")
    pact = new SoccerMDPActionShoot;
  else if (strname == "cross")
    pact = new SoccerMDPActionCross;
  else if (strname == "theiract")
    pact = new SoccerMDPActionTheirAct;
  else if (strname == "null")
    pact = new SoccerMDPActionNull;
  else
    {
      errorlog << "Did not understand SoccerMDPAction name '" << name << "'" << ende;
    }
  return pact;
}

//static
SoccerMDPAction*
SoccerMDPAction::createByType(TypesSoccerMDPAction t)
{
  switch (t)
    {
    case SMDPA_Test:      return new SoccerMDPActionTest;
    case SMDPA_HoldBall:  return new SoccerMDPActionHoldBall;
    case SMDPA_PassTo:    return new SoccerMDPActionPassTo;
    case SMDPA_DribbleTo: return new SoccerMDPActionDribbleTo;
    case SMDPA_Clear:     return new SoccerMDPActionClear;
    case SMDPA_Shoot:     return new SoccerMDPActionShoot;
    case SMDPA_Cross:     return new SoccerMDPActionCross;
    case SMDPA_TheirAct:  return new SoccerMDPActionTheirAct;
    case SMDPA_Null:      return new SoccerMDPActionNull;
    default:
      errorlog << "SoccerMDPAction::createByType: bad type "
	       << static_cast<int>(t) << ende;
      return NULL;
    }
  errorlog << "SoccerMDPAction::createByType: How did I get here? " << ende;
  return NULL;
}



/*****************************************************************************/

AdviceTreeAction*
SoccerMDPAction::createAdviceTreeAction(AbstractStateDescription* pdesc,
					int state_idx)
{
  if (!doesCreateCLangAction())
    return NULL;
  
  //It's actually not clear that we want to say do our {0}
  rcss::clang::UNumSet uset;
  uset.add(rcss::clang::UNum::uAll);
  rcss::clang::DirComm* pdir = new rcss::clang::DirComm(true, true, uset);
  // this createAction should always be a valid action since we
  // checked doesCreateCLangAction above
  pdir->push_back( std::auto_ptr<rcss::clang::Action>(createAction(pdesc, state_idx)));

  AdviceTreeAction* pact = new AdviceTreeAction;
  pact->addDirective(pdir);

  addToAdviceTreeAction(pdesc, state_idx, pact);

  return pact;
}

/*****************************************************************************/

void
SoccerMDPActionBallMovement::readParams(std::istream& is)
{
  if (!spades::skip_white_space(is))
    return;
  if (is.get() != '(')
    {
      is.setstate(std::ios::failbit);
      return;
    }
  is >> grid_idx;
  if (is.fail())
    return;
  if (!spades::skip_white_space(is))
    return;
  if (is.get() != ')')
    {
      is.setstate(std::ios::failbit);
      return;
    }
  return;
}

bool
SoccerMDPActionBallMovement::readParams(BinaryFileReader& reader)
{
  return reader.readShort(&grid_idx);
}


void
SoccerMDPActionBallMovement::processDescription(AbstractStateDescription* pdesc)
{
  BallPosDecoder decoder;
  ASDiffPatternElement* ppat = decoder.constructPattern();
  if (!ppat->matchTo(pdesc))
    errorlog << "processDescription: how did pattern not match?" << ende;
  // HACK: I always get the second pattern, which in my current abstract state space
  // is the one that happens in play on mode
  // This is a safe cast by the magic of my pattern matching
  pFac = (BallGridFactor*)decoder.getSecondPattern()->getMatchingFactor();
  delete ppat;
}

bool
SoccerMDPActionBallMovement::setParamsFromTransition(AbstractStateDescription* pdesc,
						     int first_state_idx,
						     int second_state_idx)
{
  BallPosDecoder decoder;
  //AbstractState first_state(pdesc);
  AbstractState second_state(pdesc);
  //first_state.setStateIdx(first_state_idx);
  second_state.setStateIdx(second_state_idx);
  ASDiffPatternElement* ppat = decoder.constructPattern();
  if (!ppat->matchTo(pdesc))
    errorlog << "setParamFromTransition: how did pattern not match?" << ende;
  if (!decoder.getValidFactor(&second_state, &pFac, &grid_idx))
    errorlog << "setParamFromTransition: how did decoder not extract?" << ende;
  delete ppat;
  return true;
}

// setParams should have already been called before this
// This should be used for the secondary transitions 
/* What we'll do here is verify that the ball pos in second_state
   is on the way between first_state and grid_idx (which is the overall
   target of the pass) */
bool
SoccerMDPActionBallMovement::acceptTran(AbstractStateDescription* pdesc,
					int first_state_idx,
					int second_state_idx)
{
  BallPosDecoder decoder;
  AbstractState first_state(pdesc);
  AbstractState second_state(pdesc);
  int first_grid_idx;
  int second_grid_idx;
  first_state.setStateIdx(first_state_idx);
  second_state.setStateIdx(second_state_idx);
  ASDiffPatternElement* ppat = decoder.constructPattern();
  if (!ppat->matchTo(pdesc))
    errorlog << "acceptTran: how did pattern not match?" << ende;
  BallGridFactor* pFac;
  if (!decoder.getValidFactor(&first_state, &pFac, &first_grid_idx))
    errorlog << "acceptTran: how did decoder not extract?" << ende;
  if (!decoder.getValidFactor(&second_state, &pFac, &second_grid_idx))
    errorlog << "acceptTran: how did decoder not extract?" << ende;
  delete ppat;

  // Now that we have all this stuff, we can ask the BallGridFactor
  // whether this is ok.
  if (pFac->isLocBetween(second_grid_idx, first_grid_idx, grid_idx))
    {
      actionlog(60) << "SoccerMDPActionBallMovement::acceptTran: accepting grid_idx of "
		     << second_grid_idx << "; pass from " << first_grid_idx
		     << " to " << grid_idx << ende;
      return true;
    }
  else
    {
      actionlog(60) << "SoccerMDPActionBallMovement::acceptTran: rejecting grid_idx of "
		     << second_grid_idx << "; pass from " << first_grid_idx
		     << " to " << grid_idx << ende;
      return false;
    }
}

/*****************************************************************************/
rcss::clang::Action*
SoccerMDPActionPassTo::createAction(AbstractStateDescription* pdesc,
				    int state_idx)
{
  if (getBallGridFactor() == NULL)
    {
      errorlog << "SoccerMDPActionPassTo::createAction: ball grid factor is Null!" << ende;
      return NULL;
    }

  rcss::clang::RegNamed* pReg =
    new rcss::clang::RegNamed(getBallGridFactor()->getGridName(getGridIdx()));
  return new rcss::clang::ActPassReg( std::auto_ptr<rcss::clang::Region>(pReg) );
}

void
SoccerMDPActionPassTo::addToAdviceTreeAction(AbstractStateDescription* pdesc,
					     int state_idx,
					     AdviceTreeAction* p_advice_tree_act)
{
  if (!CoachParam::instance()->getAbstractMDPGeneratePassPosAdvice())
    return;

  //we'll make a condition to pick players near the target grid rectangle
  // the amount we expand the grid. An affine function of the distance
  // from source to target
  AbstractState state(pdesc);
  state.setStateIdx(state_idx);
  BallPosDecoder decoder;
  ASDiffPatternElement* ppat = decoder.constructPattern();
  if (!ppat->matchTo(pdesc))
    errorlog << "SoccerMDPActionPassTo::addToAdviceTreeAction: how did pattern not match?" << ende;
  BallGridFactor* pStartFac;
  int start_grid_idx;
  if (!decoder.getValidFactor(&state, &pStartFac, &start_grid_idx))
    errorlog << "SoccerMDPActionPassTo::addToAdviceTreeAction: how did decoder fail?"
	     << ende;

  Rectangle start_rect = pStartFac->getGridRectangle(start_grid_idx);
  Rectangle end_rect = getBallGridFactor()->getGridRectangle(getGridIdx());
  float dist = start_rect.getCenter().getDistanceTo(end_rect.getCenter());
  float expand_amt = CoachParam::instance()->getAbstractMDPPassExpandYInt()
    + CoachParam::instance()->getAbstractMDPPassExpandSlope() * dist;
  expand_amt = Max(expand_amt, 0.0);

  Rectangle target = end_rect.expand(expand_amt);

  actionlog(200) << "addToAdviceTreeAction:passTo: dist=" << dist
		 << ", expand_amt=" << expand_amt
		 << ", target_rect=" << target
		 << ende;
  
  //now we'll tell those players to go to the grid region
  const char* varname = "X";
  rcss::clang::UNumSet uset;
  uset.add(rcss::clang::UNum( varname ));
  rcss::clang::CondPlayerPos* pCond =
    new rcss::clang::CondPlayerPos( true,
				    uset,
				    0,
				    ServerParam::instance()->getSPTeamSize(),
				    std::auto_ptr<rcss::clang::Region>(new rcss::clang::RegRec(target)));
  p_advice_tree_act->addCondition(pCond);

  rcss::clang::ActPos* pAct =
    new rcss::clang::ActPos( std::auto_ptr<rcss::clang::Region>( new rcss::clang::RegNamed( getBallGridFactor()->getGridName(getGridIdx()) )));

  rcss::clang::DirComm* pDir =
    new rcss::clang::DirComm( true, true, uset);
  pDir->add (std::auto_ptr<rcss::clang::Action>(pAct));

  p_advice_tree_act->addDirective(pDir);

  delete ppat;
}

/*****************************************************************************/
rcss::clang::Action*
SoccerMDPActionDribbleTo::createAction(AbstractStateDescription* pdesc,
				       int state_idx)
{
  if (getBallGridFactor() == NULL)
    {
      errorlog << "SoccerMDPActionPassTo::createAction: ball grid factor is Null!" << ende;
      return NULL;
    }

  rcss::clang::RegNamed* pReg =
    new rcss::clang::RegNamed(getBallGridFactor()->getGridName(getGridIdx()));
  return new rcss::clang::ActDribble( std::auto_ptr<rcss::clang::Region>(pReg) );
}

/*****************************************************************************/
rcss::clang::Action*
SoccerMDPActionClear::createAction(AbstractStateDescription* pdesc,
				   int state_idx)
{
  BallPosDecoder decoder;
  AbstractState state(pdesc);
  int grid_idx;
  state.setStateIdx(state_idx);
  ASDiffPatternElement* ppat = decoder.constructPattern();
  if (!ppat->matchTo(pdesc))
    errorlog << "SoccerMDPActionClear::createAction: how did pattern not match?" << ende;
  BallGridFactor* pFac;
  if (!decoder.getValidFactor(&state, &pFac, &grid_idx))
    errorlog << "SoccerMDPActionClear::createAction: how did decoder not extract?" << ende;
  delete ppat;
  rcss::clang::RegNamed* pReg =
    new rcss::clang::RegNamed(pFac->getGridName(grid_idx));
  return new rcss::clang::ActClear( std::auto_ptr<rcss::clang::Region>(pReg) );
}

/*****************************************************************************/
rcss::clang::Action*
SoccerMDPActionCross::createAction(AbstractStateDescription* pdesc,
				   int state_idx)
{
  // I'll define a cross as a pass anywhere in the penalty box
  //SMURF: I would sort of like to name a thingy for this, but I don't
  // have the right calls set up right now
  rcss::clang::RegRec* pReg =
    new rcss::clang::RegRec(ServerParam::instance()->getSPPenaltyAreaRectangle(false));
  return new rcss::clang::ActPassReg( std::auto_ptr<rcss::clang::Region>(pReg) );
}


/*****************************************************************************/
/*****************************************************************************/
/*****************************************************************************/
//static
const char* SoccerMDP::BIN_FILE_MAGIC = "PFRMDP";
//static
const int SoccerMDP::BIN_FILE_VERSION = 0;

SoccerMDP::SoccerMDP(int num_states)
  : MDP(num_states)
{
}

SoccerMDP::~SoccerMDP()
{
}


MDPAction*
SoccerMDP::createAction(std::istream& is)
{
  return SoccerMDPAction::createFromStream(is);
}
MDPAction*
SoccerMDP::createAction(BinaryFileReader& reader) 
{
  return SoccerMDPAction::createFrom(reader);
}


// This processes the AbstractStateDescription so that all the actions
// can load whatever information they need
void
SoccerMDP::processDescription(AbstractStateDescription* pdesc)
{
  for (int sidx = getNumStates() - 1; sidx >= 0; sidx--)
    {
      for (int aidx = getNumActionsInState(sidx) - 1; aidx >= 0; aidx--)
	{
	  // This is a safe cast because we were in charge of creating all
	  // these actions anyway
	  ((SoccerMDPAction*)(getAction(sidx, aidx)->getAction()))->processDescription(pdesc);
	}
    }
}

bool
SoccerMDP::writeHeader(BinaryFileWriter& writer) const
{
  if (!writer.writeMagicHeader(BIN_FILE_MAGIC)) return false;

  if (!writer.writeChar(BIN_FILE_VERSION)) return false;

  return true;
}

bool
SoccerMDP::readHeader(BinaryFileReader& reader)
{
  if (!reader.checkMagicHeader(BIN_FILE_MAGIC)) return false;

  char version;
  if (!reader.readChar(&version)) return false;
  if (version != BIN_FILE_VERSION)
    {
      warninglog(10) << "Trying to read different version SoccerMDP: saw="
		     << version << ", exp=" << BIN_FILE_VERSION
		     << ende;
      return false;
    }

  return true;
}

