/* -*- Mode: C++ -*- */

/* This set of classes represents information about the locations of players on the
   fields. It uses clang Regions for a pretty flexible way of working with this */

#ifndef _PLAYER_OCCUPANCY_H_
#define _PLAYER_OCCUPANCY_H_

#include <iostream>
#include <fstream>
#include "region.h"
#include "WorldState.h"
#include "FileReader.h"
#include "CoachMessageQueue.h"

class AbstractStateFactor; // defined in AbstractState.h


class PlayerOccupancyElement;
class PlayerOccupancyState;
class PlayerOccupancySet;
class POSetFileReader;

/****************************************************************************/

/* The elements are
 * a side: relative (so mine or theirs) None is invalid
 * a region of interest: players are either inside or outside the region
 * a list of cutoffs. This will be translatex to an index. If the cutoff are c_1 .. c_n,
   then the index values correspond to [0, c_1), [c_1, c_2) .. [c_n, \infty)
*/
class PlayerOccupancyElement
{
public:
  typedef std::set<int> CutOffStorage;

  // used to specify which players we are looking at in the region
  enum SideType {
    ST_Invalid,
    ST_Mine,
    ST_Theirs,
    ST_MineBigger,
    ST_TheirsBigger
  };
  
public:
  PlayerOccupancyElement()
    : reg(NULL), side(ST_Invalid), cutoffs() {}
  PlayerOccupancyElement(rcss::clang::Region* reg, SideType side, CutOffStorage& cutoffs)
    : reg(reg), side(side), cutoffs(cutoffs) {}
  PlayerOccupancyElement(const PlayerOccupancyElement& p)
    : reg( (p.reg==NULL) ? p.reg : p.reg->deepCopy().release()), side(p.side), cutoffs(p.cutoffs) {}
  ~PlayerOccupancyElement() { clear(); }

  const PlayerOccupancyElement& operator=(const PlayerOccupancyElement& p);
  
  rcss::clang::Region* getRegion() { return reg; }
  const rcss::clang::Region* getRegion() const { return reg; }
  void setRegion(rcss::clang::Region* r) { if (reg) delete reg; reg = r; }

  SideType getSide() const { return side; }
  void setSide(SideType s) { side = s; }
  bool isAMySide() const { return side == ST_Mine || side == ST_MineBigger; }
  
  CutOffStorage& getStorage() { return cutoffs; }
  const CutOffStorage& getStorage() const { return cutoffs; }
  void clearCutoffs() { cutoffs.clear(); }
  void addCutoffs(int x) { cutoffs.insert(x); }

  void clear() { if (reg) delete reg; side = ST_Invalid; clearCutoffs(); }
  
  int getIdxForWorldState(const WorldState& ws, TeamSide my_side);
  // how many possible indices can be returned
  int getNumIdx() const { return cutoffs.size() + 1; }

  void draw(FieldImage* pfi, bool verbose,
	    FieldImage::Color c_border, FieldImage::Color c_insider,
	    RegionWorldModelInterface* wmi,
	    const VarBindings& bindings) const;

  // used to define regions and conditions and such
  // The idx is an index in the set; used to establish unique names
  void createInitialAdvice(CoachMessageQueue& mqueue, int elem_idx) const;

  // elem_idx is the index in the set (used for names)
  // my_idx is the idx for this state
  rcss::clang::Cond* createCondition(int elem_idx, int my_idx) const;
  
  friend std::ostream& operator << (std::ostream &os, const PlayerOccupancyElement& p);
  friend std::istream& operator >> (std::istream &is, PlayerOccupancyElement& p);

private:
  std::string getRegionName(int elem_idx) const;
  std::string getCondName(int elem_idx, int cutoff_idx) const;
  
  rcss::clang::Region* reg;
  SideType side;
  // for each element x, the number < x (and >= previous element) is one value
  CutOffStorage cutoffs;

private:
  static const char* REG_PREFIX; // add element idx
  static const char* COND_PREFIX; // add element idx, then idx in cutoffs
  
};

std::ostream & operator << (std::ostream & o, const PlayerOccupancyElement::SideType& st);
std::istream & operator >> (std::istream & i, PlayerOccupancyElement::SideType& st);
  

/****************************************************************************/
/* This class represents a particular arrangment of occupancy of the players
   in a region. It only makes sense in the context of a PlayerOccupancySet
   (which is why you have to give it in the constructor) */
class PlayerOccupancyState
{
public:
  // the idx is the value of the PlayerOccupancyElement
  static void getColorsFor(int idx, bool my_side,
			   FieldImage::Color* pborder,
			   FieldImage::Color* pinside);

public:
  PlayerOccupancyState(const PlayerOccupancySet* pposet)
    : pposet(pposet), idx_storage() { updateStorageSize(); }
  PlayerOccupancyState() {}

  const PlayerOccupancySet* getPOSet() const { return pposet; }
  void setPOSet(const PlayerOccupancySet* p) { pposet = p; updateStorageSize(); }

  // -1 on error
  int getElementIdx(int elem) const;
  void setElementIdx(int elem, int idx);

  int getOverallIdx() const;
  void setOverallIdx(int idx);
  
  void updateStorageSize();
  
  void draw(FieldImage* pfi, bool verbose,
	    RegionWorldModelInterface* wmi,
	    const VarBindings& bindings) const;

  rcss::clang::Cond* createCondition() const;

  friend std::ostream& operator << (std::ostream &os, const PlayerOccupancyState& s);

private:
  const PlayerOccupancySet* pposet;
  
  // Stores the indexes of the elements of the set for a particular world state
  typedef std::vector<int> IdxStorage;
  IdxStorage idx_storage;

  //we,ll just adjust the alpha channel for the inside
  static const FieldImage::Color OUR_COLORS[];
  static const FieldImage::Color OPP_COLORS[];
  static const float INSIDE_ALPHA_VALUE;
};


/****************************************************************************/

class PlayerOccupancySet
{
public:
  typedef std::vector<PlayerOccupancyElement> POStorage;
  // The int is the multiplication factor for converting a PlayerOccupancy to an int
  typedef std::vector<int> MultFactorStorage;
  
public:
  PlayerOccupancySet();
  ~PlayerOccupancySet();

  friend std::ostream& operator << (std::ostream &os, const PlayerOccupancySet& p);

  unsigned getNumElements() const { return storage.size(); }

  int getNumIdx() const
  { return mult_factor.empty() ? 0 : (*(mult_factor.end()-1) * (storage.end()-1)->getNumIdx()); }

  // If pstate is NULL, a new one it allocated
  PlayerOccupancyState* 
  getPOStateForWorldState(const WorldState& ws,
			  TeamSide my_side,
			  PlayerOccupancyState* pstate);

  void draw(FieldImage* pfi, bool verbose,
	    RegionWorldModelInterface* wmi,
	    const VarBindings& bindings) const;

  // used to define regions and conditions and such
  void createInitialAdvice(CoachMessageQueue& mqueue) const;

  AbstractStateFactor* createAbstractStateFactor() const;
  
protected:
  friend class POSetFileReader;
  friend class PlayerOccupancyState;
  
  // This is protected because of the caching thing we do with the number
  // of allowable indices
  POStorage& getStorage() { return storage; }
  const POStorage& getStorage() const { return storage; }
  MultFactorStorage& getMultFactor() { return mult_factor; }
  const MultFactorStorage& getMultFactor() const { return mult_factor; }

  void updateMultFactor();

private:
  /* The low-order element is stored first */
  POStorage storage;
  MultFactorStorage mult_factor; // the mult factor used to convert to a single index

  static const FieldImage::Color DEFAULT_OUR_COLOR;
  static const FieldImage::Color DEFAULT_OUR_INSIDE_COLOR;
  static const FieldImage::Color DEFAULT_OPP_COLOR;
  static const FieldImage::Color DEFAULT_OPP_INSIDE_COLOR;

};


class POSetFileReader
  : public spades::FileReader
{
public:
  POSetFileReader(PlayerOccupancySet* pposet) : FileReader(), pposet(pposet) {}
  ~POSetFileReader() {}
  
  void readFile (const char* path) { FileReader::readFile(path, 1.0); }

protected:
  bool processLine(std::istrstream& line,
		   const char* fileid,
		   const char* path,
		   float version);
  
private:
  PlayerOccupancySet* pposet;
};

/****************************************************************************/

class PlayerOccupancyTracker
{
public:
  // if my_side is TS_Both, every state is counted as us being both left and right sides
  // per_cycle_fn is NULL or empty to turn that off
  PlayerOccupancyTracker(TeamSide my_side, const PlayerOccupancySet& poset,
			 const char* per_cycle_fn);
  ~PlayerOccupancyTracker() {}

  TeamSide getMySide() const { return my_side; }
  void setMySide(TeamSide s) { my_side = s; }

  const PlayerOccupancySet& getPOSet() const { return poset; }
  
  void writeSummaryInfo(std::ostream& os);

  void processWorldState(const WorldState& ws);
  
private:
  void processWorldState(const WorldState& ws, TeamSide side_to_use);

  typedef std::vector<unsigned short> POStateSummary;

  TeamSide my_side;
  PlayerOccupancySet poset;
  bool write_per_cycle;
  std::ofstream os_per_cycle;
  POStateSummary state_summary;

  // even though this is transient date, we have it here to avoid reallocation
  PlayerOccupancyState state;
};

#endif
