/* -*- Mode: C++ -*- */

#include <iterator>
#include <sstream>
#include "PlayerOccupancy.h"
#include "StoreRegionMsgBuilder.h"
#include "clangparser.h"
#include "constants.h"
#include "utility.h"
#include "AbstractState.h"
#include "AbstractStateElements.h"
#include "ServerParam.h"
#include "Logger.h"

using namespace spades;

#define USE_SANITY_CHECKS

/********************** PlayerOccupancyElement::SideType *****************************/
static const char* SIDE_TYPE_STRINGS[] = {"None", "Mine", "Theirs", "MineBigger", "TheirsBigger"};

std::ostream&
operator << (std::ostream & o, const PlayerOccupancyElement::SideType& st)
{
  o << SIDE_TYPE_STRINGS[(int)st];
  return o;
}

std::istream&
operator >> (std::istream & is, PlayerOccupancyElement::SideType& st)
{
  std::string str;
  is >> str;
  if (is.fail())
    return is;
  for (int i=0; i<4; i++)
    {
      if (strcasecmp(str.c_str(), SIDE_TYPE_STRINGS[i]) == 0)
	{
	  st = (PlayerOccupancyElement::SideType)i;
	  return is;
	}
    }
  is.setstate(std::ios::failbit);
  return is;
}


/********************** PlayerOccupancyElement **************************************/
const char* PlayerOccupancyElement::REG_PREFIX = "PR"; // add element idx
const char* PlayerOccupancyElement::COND_PREFIX = "PC"; // add element idx, then idx in cutoffs

const PlayerOccupancyElement&
PlayerOccupancyElement::operator=(const PlayerOccupancyElement& p)
{
  clear();
  reg = (p.reg == NULL) ? NULL : p.reg->deepCopy().release();
  side = p.side;
  cutoffs = p.cutoffs;
  return *this;
}


int
PlayerOccupancyElement::getIdxForWorldState(const WorldState& ws, TeamSide my_side)
{
  //SMURF: Do we need to have a real bindings structure?
  VarBindings bindings;

  int my_side_count = ws.countPlayersInRegion(relativeTeamSideToAbs(RTS_Mine, my_side),
					      reg, my_side, bindings);
  int their_side_count = ws.countPlayersInRegion(relativeTeamSideToAbs(RTS_Theirs, my_side),
						 reg, my_side, bindings);
  int my_idx = -1;
  int their_idx = -1;
  
  int idx = 0;
  for (CutOffStorage::const_iterator iter = cutoffs.begin();
       iter != cutoffs.end();
       iter++, idx++)
    {
      if (my_idx == -1 && my_side_count < *iter)
	my_idx = idx;
      if (their_idx == -1 && their_side_count < *iter)
	their_idx = idx;
    }
  if (my_idx == -1)
    my_idx = idx;
  if (their_idx == -1)
    their_idx = idx;

  actionlog(230) << "PlayerOccupancyElement::getIdxForWorldState: "
		 << side << ' '
		 << my_side_count << ' '
		 << their_side_count << ' '
		 << my_idx << ' '
		 << their_idx << ' '
		 << ende;
  
  if (side == ST_MineBigger && !(my_idx > their_idx))
    return 0;
  if (side == ST_TheirsBigger && !(their_idx > my_idx))
    return 0;

  if (side == ST_MineBigger || side == ST_Mine)
    return my_idx;
  if (side == ST_TheirsBigger || side == ST_Theirs)
    return their_idx;

  errorlog  << "PlayerOccupancyElement::getIdxForWorldState: how did I get here" << ende;
  return -1;
}

std::ostream&
operator<< (std::ostream &os, const PlayerOccupancyElement& p)
{
  os << p.side << "   ";
  copy(p.cutoffs.begin(), p.cutoffs.end(), std::ostream_iterator<int>(os, " "));
  os << ' ';
  if (p.reg == NULL)
    {
      rcss::clang::RegNull regnull;
      os << regnull;
    }
  else
    {
      os << *p.reg;
    }
  return os;
}

std::istream&
operator>> (std::istream &is, PlayerOccupancyElement& p)
{
  p.clear();
  
  is >> p.side;
  if (is.fail())
    return is;

  for (;;)
    {
      if (!spades::skip_white_space(is))
	{
	  is.setstate(std::ios::failbit);
	  return is;
	}
      if (!isdigit(is.peek()))
	break;
      int x;
      is >> x;
      if (is.fail())
	return is;
      p.cutoffs.insert(x);
    }

  is.clear();
  
  StoreRegionMsgBuilder msg_builder;
  rcss::clang::Parser clang_parser(msg_builder);

  try {
    // First we'll just try and read one line. If we get an EOF, we'll let the parser have all of it
    char line[MAX_SERVER_MSG_LEN];
    is.getline(line, MAX_SERVER_MSG_LEN);
    if (is.fail() && is.eof())
      {
	if (((rcss::Parser*)&clang_parser)->parse(is) != 0)
	  {
	    errorlog << "Could not parse region for PlayerOccupancy: " << line << ende;
	    is.setstate(std::ios::failbit);
	  }
	else
	  {
	    p.reg = msg_builder.getLastRegion();
	  }
      }
    else if (is.fail())
      {
	return is;
      }
    else
      {
	if (clang_parser.parse(line) != 0)
	  {
	    errorlog << "Could not parse region for PlayerOccupancy: " << line << ende;
	    is.setstate(std::ios::failbit);
	  }
	else
	  {
	    p.reg = msg_builder.getLastRegion();
	  }
      }
  }
  catch( rcss::clang::BuilderErr e)
    {
      errorlog << "Error parsing region for PlayerOccupancy: " << e.what() << ende;
      is.setstate(std::ios::failbit);
    }
  catch( rcss::util::NullErr e)
    {
      errorlog << "Error parsing region for PlayerOccupancy: " << e.what() << ende;
      is.setstate(std::ios::failbit);
    }
  
  return is;
}

void
PlayerOccupancyElement::draw(FieldImage* pfi, bool verbose,
			     FieldImage::Color c_border, FieldImage::Color c_inside,
			     RegionWorldModelInterface* wmi,
			     const VarBindings& bindings) const
{
  if (reg == NULL)
    pfi->addLegendLine("NULL region");
  else
    reg->draw(pfi, verbose, c_border, c_inside, wmi, bindings);

  if (verbose)
    {
      VecPosition pt = reg->getRandomPtIn(wmi, bindings);
      std::ostringstream label;
      // we'll not put the side in -- leave that to the color
      copy(cutoffs.begin(), cutoffs.end(), std::ostream_iterator<int>(label, " "));
      pfi->addText(label.str().c_str(), pt, c_border);
    }
}

// used to define regions and conditions and such
// The idx is an index in the set; used to establish unique names
void
PlayerOccupancyElement::createInitialAdvice(CoachMessageQueue& mqueue, int elem_idx) const
{
  // create the region
  std::string regname = getRegionName(elem_idx);
  mqueue.getDefineContainer().push(new rcss::clang::DefReg(regname,
							   std::auto_ptr<rcss::clang::Region>(reg->deepCopy())));

  // create the various conditions we could need
  int cutoff_idx = 0;
  int last_cutoff = 0;
  rcss::clang::UNumSet uset;
  uset.add(rcss::clang::UNum::uAll);

  for (CutOffStorage::const_iterator iter = cutoffs.begin();
       ; // we'll let the loop term happen at the end; we have an implict team_size at the end
       iter++, cutoff_idx++)
    {
      int this_cutoff =
	(iter != cutoffs.end()) ? *iter : (ServerParam::instance()->getSPTeamSize()+1);
      
      rcss::clang::Cond* pCond = NULL;
      
      rcss::clang::CondPlayerPos* pBaseCond =
	new rcss::clang::CondPlayerPos( (side == ST_Mine || side == ST_MineBigger),
					uset,
					last_cutoff,
					this_cutoff - 1,
					std::auto_ptr<rcss::clang::Region>(new rcss::clang::RegNamed(regname)));

      // Add a [0,last_cutoff-1] condition to the opposite side
      // This is the condition that the opposite team has a smaller cutoff value
      if (side == ST_MineBigger || side == ST_TheirsBigger)
	{
	  rcss::clang::CondAnd* pCondAnd = new rcss::clang::CondAnd;
	  pCondAnd->push_back(std::auto_ptr<rcss::clang::Cond>(pBaseCond));
	  pCondAnd->push_back(std::auto_ptr<rcss::clang::Cond>(
			      new rcss::clang::CondPlayerPos( !(side == ST_MineBigger),
							      uset,
							      0,
							      std::max(last_cutoff - 1, 0),
							      std::auto_ptr<rcss::clang::Region>(new rcss::clang::RegNamed(regname)))));
	  pCond = pCondAnd;
	}
      else
	{
	  pCond = pBaseCond;
	}
      
      std::string condname =
	getCondName(elem_idx, cutoff_idx);

      mqueue.getDefineContainer().push(new rcss::clang::DefCond(condname,
								std::auto_ptr<rcss::clang::Cond>(pCond)));

      last_cutoff = this_cutoff;

      if (iter == cutoffs.end())
	break;
    }
}

// elem_idx is the index in the set (used for names)
// my_idx is the idx for this state
rcss::clang::Cond*
PlayerOccupancyElement::createCondition(int elem_idx, int my_idx) const
{
  return new rcss::clang::CondNamed(getCondName(elem_idx, my_idx));
}



std::string
PlayerOccupancyElement::getRegionName(int elem_idx) const
{
  return std::string(REG_PREFIX) + toString(elem_idx);
}

std::string
PlayerOccupancyElement::getCondName(int elem_idx, int cutoff_idx) const
{
  return std::string(COND_PREFIX) + toString(elem_idx) + "_" + toString(cutoff_idx);
}


/********************** PlayerOccupancyState ************************************/

// We'll set alpha to 1.0 if the idx is 0, so we can make the first two
// the same
const FieldImage::Color
PlayerOccupancyState::OUR_COLORS[] = { FieldImage::Color("#0B0165"),
				       FieldImage::Color("#0B0165"),
				       FieldImage::Color("#1201A4"),
				       //FieldImage::Color("#1702D4"),
				       FieldImage::Color("#1A02F0"),
				       FieldImage::Color("#1B01FE"),
				       FieldImage::Color("#1B01FE"),
				       FieldImage::Color("#1B01FE"),
				       FieldImage::Color("#1B01FE") };
const FieldImage::Color
PlayerOccupancyState::OPP_COLORS[] = { FieldImage::Color("#620B01"),
				       FieldImage::Color("#620B01"),
				       //FieldImage::Color("#A91303"),
				       FieldImage::Color("#E71B04"),
				       FieldImage::Color("#FF1D05"),
				       FieldImage::Color("#FF1D05"),
				       FieldImage::Color("#FF1D05"),
				       FieldImage::Color("#FF1D05") };
const float PlayerOccupancyState::INSIDE_ALPHA_VALUE = 0.9;

//static
// the idx is the value of the PlayerOccupancyElement
void
PlayerOccupancyState::getColorsFor(int idx, bool my_side,
				   FieldImage::Color* pborder,
				   FieldImage::Color* pinside)
{
  *pborder = FieldImage::Color(my_side ? OUR_COLORS[idx] : OPP_COLORS[idx]);
  *pinside = FieldImage::Color(*pborder);
  // This is sort of a hack to make the 0 element clear
  pinside->setAlpha(idx == 0 ? 1.0 : INSIDE_ALPHA_VALUE);
}



// -1 on error
int
PlayerOccupancyState::getElementIdx(int elem) const
{
#ifdef USE_SANITY_CHECKS
  if (elem < 0 || (unsigned)elem >= idx_storage.size())
    {
      errorlog << "POState::getElementIdx: out of range: idx="
	       << elem << ", count=" << idx_storage.size() << ende;
      return -1;
    }
#endif
  return idx_storage[elem];
}

void
PlayerOccupancyState::setElementIdx(int elem, int idx)
{
#ifdef USE_SANITY_CHECKS
  if (elem < 0 || (unsigned)elem >= idx_storage.size())
    {
      errorlog << "POState::setElementIdx: elem out of range: idx="
	       << elem << ", count=" << idx_storage.size() << ende;
      return;
    }
  if (pposet == NULL)
    {
      errorlog << "POState::setElementIdx: NULL pposet" << ende;
      return;
    }
  if ((unsigned)elem >= pposet->getStorage().size())
    {
      errorlog << "POState::setElementIdx: in range for me, but not for Set!: idx="
	       << elem << ", count(me)=" << idx_storage.size()
	       << ", count(set)=" << pposet->getStorage().size()
	       << ende;
      return;
    }
  const PlayerOccupancyElement& poelem = pposet->getStorage()[elem];
  if (idx < 0 || idx >= poelem.getNumIdx())
    {
      errorlog << "POState::setElementIdx: idx out of range for elem " << elem << ":"
	       << "idx=" << idx << ", num=" << poelem.getNumIdx() << ende;
      return;
    }
#endif
  
  idx_storage[elem] = idx;
}


int
PlayerOccupancyState::getOverallIdx() const
{
#ifdef USE_SANITY_CHECKS
  if (pposet == NULL)
    {
      errorlog << "POState::getOverallIdx: NULL pposet" << ende;
      return -1;
    }
  if (idx_storage.size() != pposet->getStorage().size() ||
      idx_storage.size() != pposet->getMultFactor().size())
    {
      errorlog << "POState::getOverallIdx: size mismatch "
	       << idx_storage.size() << " "
	       << pposet->getStorage().size() << " "
	       << pposet->getMultFactor().size() << ende;
      return -1;
    }
#endif

  int result = 0;
  for (int i=idx_storage.size()-1; i>=0; i--)
    {
      result += idx_storage[i] * pposet->getMultFactor()[i];
    }
  return result;
}

void
PlayerOccupancyState::setOverallIdx(int idx)
{
#ifdef USE_SANITY_CHECKS
  if (pposet == NULL)
    {
      errorlog << "POState::setOverallIdx: NULL pposet" << ende;
      return;
    }
  if (idx < 0 || idx >= pposet->getNumIdx())
    {
      errorlog << "POState::setOverallIdx: idx out of range "
	       << "idx=" << idx << ", num=" << pposet->getNumIdx() << ende;
      return;
    }
  if (idx_storage.size() != pposet->getMultFactor().size())
    {
      errorlog << "POState::setOverallIdx: size mismatch"
	       << idx_storage.size() << " "
	       << pposet->getMultFactor().size() << ende;
      return;
    }
#endif

  for (int elem=idx_storage.size()-1; elem >= 0; elem--)
    {
#ifdef USE_SANITY_CHECKS
      setElementIdx(elem, idx / pposet->getMultFactor()[elem]);
#else
      idx_storage[elem] = idx / pposet->getMultFactor()[elem];
#endif
      idx %= pposet->getMultFactor()[elem];
    }
}

void
PlayerOccupancyState::updateStorageSize()
{
  idx_storage.resize( pposet ? pposet->getNumElements() : 0 );
}

void
PlayerOccupancyState::draw(FieldImage* pfi, bool verbose,
			   RegionWorldModelInterface* wmi,
			   const VarBindings& bindings) const
{
#ifdef USE_SANITY_CHECKS
  if (pposet == NULL)
    {
      errorlog << "POState::draw: NULL pposet" << ende;
      return;
    }
  if (idx_storage.size() != pposet->getStorage().size())
    {
      errorlog << "POState::draw: size mismatch"
	       << idx_storage.size() << " "
	       << pposet->getMultFactor().size() << ende;
      return;
    }
#endif

  int elem = 0;
  for (PlayerOccupancySet::POStorage::const_iterator iter = pposet->getStorage().begin();
       iter != pposet->getStorage().end();
       iter++, elem++)
    {
      FieldImage::Color c_border;
      FieldImage::Color c_inside;
      getColorsFor(idx_storage[elem],
		   iter->isAMySide(),
		   &c_border,
		   &c_inside);
      iter->draw(pfi, verbose,
		 c_border, c_inside,
		 wmi, bindings);
    }
  if (verbose)
    {
      std::ostringstream label;
      label << "PlayerOccupancyState idx=" << getOverallIdx() << ":" << *this;
      pfi->addLegendLine(label.str().c_str());
    }
}

rcss::clang::Cond*
PlayerOccupancyState::createCondition() const
{
#ifdef USE_SANITY_CHECKS
  if (pposet == NULL)
    {
      errorlog << "POState::createCondition: NULL pposet" << ende;
      return NULL;
    }
  if (idx_storage.size() != pposet->getStorage().size())
    {
      errorlog << "POState::createCondition: size mismatch"
	       << idx_storage.size() << " "
	       << pposet->getMultFactor().size() << ende;
      return NULL;
    }
#endif

  rcss::clang::CondAnd* pAnd = new rcss::clang::CondAnd;
  for (int elem_idx = 0;
       elem_idx < (signed)idx_storage.size();
       elem_idx++)
    {
      rcss::clang::Cond* p =
	pposet->getStorage()[elem_idx].createCondition(elem_idx, idx_storage[elem_idx]);
      if (p)
	pAnd->push_back(std::auto_ptr<rcss::clang::Cond>(p));
    }

  if (pAnd->getConds().empty())
    {
      delete pAnd;
      pAnd = NULL;
    }
  return pAnd;
}



std::ostream&
operator<< (std::ostream &os, const PlayerOccupancyState& s)
{
  copy(s.idx_storage.begin(), s.idx_storage.end(),
       std::ostream_iterator<int>(os, " "));
  return os;			
}


/********************** PlayerOccupancySet **************************************/

const FieldImage::Color PlayerOccupancySet::DEFAULT_OUR_COLOR("#1A38FE");
const FieldImage::Color PlayerOccupancySet::DEFAULT_OUR_INSIDE_COLOR("#1A38FE", 0.7);
const FieldImage::Color PlayerOccupancySet::DEFAULT_OPP_COLOR("#FF2817");
const FieldImage::Color PlayerOccupancySet::DEFAULT_OPP_INSIDE_COLOR("#FF2817", 0.7);

PlayerOccupancySet::PlayerOccupancySet()
{
}

PlayerOccupancySet::~PlayerOccupancySet()
{
}

void
PlayerOccupancySet::updateMultFactor()
{
  int fac = 1;

  mult_factor.resize(storage.size());

  for (POStorage::iterator iter = storage.begin();
       iter != storage.end();
       iter++)
    {
      mult_factor[iter-storage.begin()] = fac;
      fac *= iter->getNumIdx();
    }
}

PlayerOccupancyState*
PlayerOccupancySet::getPOStateForWorldState(const WorldState& ws,
					    TeamSide my_side,
					    PlayerOccupancyState* pstate)
{
  if (pstate == NULL)
    pstate = new PlayerOccupancyState(this);
  
#ifdef USE_SANITY_CHECKS
  if (pstate->getPOSet() != this)
    {
      errorlog << "getPOStateForWorldState: different sets "
	       << pstate->getPOSet() << " " << this
	       << ende;
      return pstate;
    }
#endif

  for (int elem=storage.size() - 1; elem>=0; elem--)
    {
      int idx = storage[elem].getIdxForWorldState(ws, my_side);
      pstate->setElementIdx(elem, idx);
    }
  return pstate;
}

void
PlayerOccupancySet::draw(FieldImage* pfi, bool verbose,
			 RegionWorldModelInterface* wmi,
			 const VarBindings& bindings) const
{
  for (POStorage::const_iterator iter = storage.begin();
       iter != storage.end();
       iter++)
    {
      iter->draw(pfi, verbose,
		 (iter->isAMySide()) ? DEFAULT_OUR_COLOR : DEFAULT_OPP_COLOR,
		 (iter->isAMySide()) ? DEFAULT_OUR_INSIDE_COLOR : DEFAULT_OPP_INSIDE_COLOR,
		 wmi, bindings);
    }
  if (verbose)
    {
      std::ostringstream label;
      label << "Number of possible values: " << getNumIdx();
      pfi->addLegendLine(label.str().c_str());
    }
}

void
PlayerOccupancySet::createInitialAdvice(CoachMessageQueue& mqueue) const
{
  for (POStorage::const_iterator iter = storage.begin();
       iter != storage.end();
       iter++)
    {
      iter->createInitialAdvice(mqueue, iter - storage.begin());
    }
}

AbstractStateFactor*
PlayerOccupancySet::createAbstractStateFactor() const
{
  AbstractStateFactorAnd* pAnd = new AbstractStateFactorAnd();
  for (POStorage::const_iterator iter = storage.begin();
       iter != storage.end();
       iter++)
    {
      pAnd->addFactor(new POElementFactor(*iter));
    }
  return pAnd;
}


std::ostream&
operator << (std::ostream &os, const PlayerOccupancySet& p)
{
  os << "# This file represents a PlayerOccupancySet (from coach/PlayerOccupancy.[Ch])" << std::endl;
  os << "version : 1.0" << std::endl;

  copy(p.storage.begin(), p.storage.end(), std::ostream_iterator<PlayerOccupancyElement>(os, "\n"));

  return os;
}


bool
POSetFileReader::processLine(std::istrstream& line,
			     const char* fileid,
			     const char* path,
			     float version)
{
  PlayerOccupancyElement poelem;

  line >> poelem;

  if (line.fail())
    {
      errorlog << "Failed reading PlayerOccupancySet line from " << fileid << ende;
      return false;
    }
  
  pposet->getStorage().push_back(poelem);

  // This is somewhat inefficient because we keep doing this as we read lines
  // However, I don't think we'll be doing this much, so I'm not going to worry about it.
  pposet->updateMultFactor();
  
  return true;
}


/********************** PlayerOccupancyTracker **************************************/

PlayerOccupancyTracker::PlayerOccupancyTracker(TeamSide my_side,
					       const PlayerOccupancySet& poset,
					       const char* per_cycle_fn)
  :  my_side(my_side),
     poset(poset),
     write_per_cycle(per_cycle_fn != NULL && per_cycle_fn[0] != 0),
     os_per_cycle(),
     state_summary(poset.getNumIdx(), 0),
     state(&this->poset)
{
  if (write_per_cycle)
    {
      os_per_cycle.open(per_cycle_fn);
      if (!os_per_cycle)
	{
	  errorlog << "PlayerOccupancyTracker: could not open per cycle file '"
		   << per_cycle_fn << "'" << ende;
	  write_per_cycle = false;
	}
      else
	{
	  os_per_cycle << "# This is a per cycle file generated by PlayerOccupancyTracker " << std::endl;
	  os_per_cycle << "# Format: <time>: <overall index> " << std::endl;
	  os_per_cycle << "# The PlayerOccupancySet used was" << std::endl;
	  writeCommented(os_per_cycle, poset);
	}      
    }
}


void
PlayerOccupancyTracker::writeSummaryInfo(std::ostream& os)
{
  os << "# This file is the summary info of PlayerOccupancyTracker" << std::endl;
  os << "# Format: <State idx>: <num of times seen>" << std::endl;
  os << "# The PlayerOccupancySet used was: " << std::endl;
  writeCommented(os, poset);
  for (unsigned i=0; i < state_summary.size(); i++)
    os << i << ": " << state_summary[i] << std::endl;
}


void
PlayerOccupancyTracker::processWorldState(const WorldState& ws)
{
  switch (my_side)
    {
    case TS_None:
      actionlog(150) << "PlayerOccupancyTracker: got a TS_None, doing nothing" << ende;
      break;
    case TS_Left:
      processWorldState(ws, TS_Left);
      break;
    case TS_Right:
      processWorldState(ws, TS_Right);
      break;
    case TS_Both:
      processWorldState(ws, TS_Left);
      processWorldState(ws, TS_Right);
      break;
    default:
      errorlog << "PlayerOccupancyTracker::processWorldState: what is my_side? "
	       << my_side << ende;
      break;
    }
}


void
PlayerOccupancyTracker::processWorldState(const WorldState& ws, TeamSide side_to_use)
{
  poset.getPOStateForWorldState(ws, side_to_use, &state);
  int idx = state.getOverallIdx();
  if (write_per_cycle)
    os_per_cycle << ws.getTime() << ": " << idx << std::endl;
  state_summary[idx]++;
}

