/* -*- Mode: c++ -*- */

/*
 *Copyright:

    Copyright (C) 2002, 2003 Patrick Riley
    Copyright (C) 2001 Patrick Riley and Emil Talpes

    This file is part of the SPADES simulation system.

    The SPADES simulation system is free software; you can
    redistribute it and/or modify it under the terms of the GNU Lesser
    General Public License as published by the Free Software
    Foundation; either version 2 of the License, or (at your option)
    any later version.

    The SPADES simulation system is distributed in the hope that it
    will be useful, but WITHOUT ANY WARRANTY; without even the implied
    warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
    See the GNU Lesser General Public License for more details.

    You should have received a copy of the GNU Lesser General Public
    License along with the SPADES simulation system; if not, write to
    the Free Software Foundation, Inc., 59 Temple Place, Suite 330,
    Boston, MA 02111-1307 USA

 *EndCopyright:
*/

#ifndef PARAMREADER_H_
#define PARAMREADER_H_

#include <iostream>
#include <fstream>
#include <strstream>
#include <map>
#include <string>
#include <vector>
#include "FileReader.h"
#include "Logger.h"

namespace spades
{
  /****************************************************************************************/
  /* Helper Functions */
  /****************************************************************************************/

  class ParamReader
    : public FileReader
  {
  public:
    // A class that knows how to process parameters and store them
    // We'll subclass this below
    class ParamStorer
    {
    public:
      //ReadReturn values. These are ints so that sucess can be anything >= 0
      static const int RR_None = -3;
      static const int RR_NoMatch = -2;
      static const int RR_FormatErr = -1;
      // all values >= 0 indicate success
      
    public:
      ParamStorer() {}
      virtual ~ParamStorer() {}

      // return 0 on success
      // otherwise, see the RR_ values above
      virtual int readArgsFrom(const std::string& key, const char* fileid, const char* path,
			       std::istream& is, bool have_argument) = 0;
      // returns the number of args processed
      // otherwise see the RR_ values above
      virtual int readCmdLineArgs(const std::string& key,
				  int argc, const char* const* argv) = 0;

      // returns whether this storer set the data for this
      // you should only override the ones that make sense for you
      // return 0 on success,
      //        RR_FormatErr on type mismatch
      //        RR_NoMatch if key DNE
      virtual int setParam (const std::string & key, const int& value)
      { return RR_FormatErr; }
      virtual int setParam (const std::string & key, const short& value)
      { return setParam(key, static_cast<int>(value)); }
      virtual int setParam (const std::string & key, const bool& value)
      { return RR_FormatErr; }
      // Any time you override this method, you shoudl override the pair one below
      // as ParamReader may put the current directory as the second element for you
      virtual int setParam (const std::string & key, const std::string& value)
      { return RR_FormatErr; }
      // This variant is intended for a file path entry. The string is the value
      // and the const char* is a starting file/directory (if it is a directory, it
      // must end in / otherwise the final component will be removed)
      virtual int setParam (const std::string & key,
			    const std::pair<const std::string, const char*>& value)
      { return RR_FormatErr; }
      virtual int setParam (const std::string & key, const double& value)
      { return RR_FormatErr; }
      virtual int setParam (const std::string & key, const std::vector<int>& value)
      { return RR_FormatErr; }
      virtual int setParam (const std::string & key, const std::vector<double>& value)
      { return RR_FormatErr; }
      virtual int setParam (const std::string & key, const std::vector<std::string>& value)
      { return RR_FormatErr; }

      virtual void print(std::ostream& o) const = 0;
    };

  private:

    class ParamStorerByMap
      : public ParamStorer
    {
    public:
      //an abstract base class
      class ParamStorage
      {
      public:
	ParamStorage() {}
	virtual ~ParamStorage() {}

	// only success return should be 0
	// the key has already been read; it is provided solely to generate error messages
	// otherwise, see the RR_ values above, though RR_NoMatch should not be returned
	virtual int readArgsFrom(const std::string& key,
				 const char* fileid, const char* path,
				 std::istream& is, bool have_argument) = 0;
	// returns the number of args processed
	// the key has already been read; it is provided solely to generate error messages
	// otherwise, see the RR_ values above, though RR_NoMatch should not be returned
	virtual int readCmdLineArgs(const std::string& key,
				    int argc, const char* const* argv) = 0;

	// returns whether this storage accepted the data
	virtual bool setParam (const int& value)
	{ return false; }
	virtual bool setParam (const bool& value)
	{ return false; }
	// Any time you override this method, you shoudl override the pair one below
	// as ParamReader may put the current directory as the second element for you
	virtual bool setParam (const std::string& value)
	{ return false; }
	// This variant is intended for a file path entry. The string is the value
	// and the const char* is a starting file/directory (if it is a directory, it
	// must end in / otherwise the final component will be removed)
	virtual bool setParam (const std::pair<const std::string, const char*>& value)
	{ return false; }
	virtual bool setParam (const double& value)
	{ return false; }
	virtual bool setParam (const std::vector<int>& value)
	{ return false; }
	virtual bool setParam (const std::vector<double>& value)
	{ return false; }
	virtual bool setParam (const std::vector<std::string>& value)
	{ return false; }

	virtual void print(std::ostream& o) const = 0;
      };

    public:
      ParamStorerByMap();
      ~ParamStorerByMap();

      int readArgsFrom(const std::string& key, const char* fileid, const char* path,
		       std::istream& is, bool have_argument);
      // returns the number of args processed
      int readCmdLineArgs(const std::string& key,
			  int argc, const char* const* argv);

      // returns whether this storer set the data for this
      // return 0 on success,
      //        RR_FormatErr on type mismatch
      //        RR_NoMatch if key DNE
      int setParam (const std::string & key, const int& value);
      int setParam (const std::string & key, const bool& value);
      int setParam (const std::string & key, const std::string& value);
      int setParam (const std::string & key,
		    const std::pair<const std::string, const char*>& value);
      int setParam (const std::string & key, const double& value);
      int setParam (const std::string & key, const std::vector<int>& value);
      int setParam (const std::string & key, const std::vector<double>& value);
      int setParam (const std::string & key, const std::vector<std::string>& value);

      // returns whether a parameter already had this name
      bool addParam (const std::string& key, ParamStorage* p);
      
      void print(std::ostream& o) const ;
      
    private:
      template <class T> int templateSetParam (const std::string& key, const T& value);

      typedef std::map<std::string, ParamStorage*> MapParam;
      MapParam map_param;
    };

    template <class T>
    class BasicParamStorage
      : public ParamStorerByMap::ParamStorage
    {
    public:
      // do not take over memory
      BasicParamStorage(T* pval) : ParamStorage(), pval(pval) {}
      virtual ~BasicParamStorage() {}

      //note that the readArgsFrom and readCmdLineArgs are left virtual

      // returns whether this storage accepted the data
      virtual bool setParam (const T& value) { *pval = value; return true; }

      void print(std::ostream& o) const { o << *pval; }
      
    protected:
      T* pval;
    };

    typedef BasicParamStorage<int>         IntBasicParamStorage;
    typedef BasicParamStorage<bool>        BoolBasicParamStorage;
    typedef BasicParamStorage<std::string> StringBasicParamStorage;
    typedef BasicParamStorage<double>      DoubleBasicParamStorage;
    
    class IntParamStorage
      : public IntBasicParamStorage
    {
    public:
      // do not take over memory
      IntParamStorage(int* pval) : IntBasicParamStorage(pval) {}
      ~IntParamStorage() {}

      // only success return should be 0
      // the key has already been read
      // otherwise, see the RR_ values above, though RR_NoMatch should not be returned
      int readArgsFrom(const std::string& key,
		       const char* fileid, const char* path,
		       std::istream& is, bool have_argument);
      // returns the number of args processed
      // the key has already been read
      // otherwise, see the RR_ values above, though RR_NoMatch should not be returned
      int readCmdLineArgs(const std::string& key,
			  int argc, const char* const* argv);
    };

    class BoolParamStorage
      : public BoolBasicParamStorage
    {
    public:
      // do not take over memory
      BoolParamStorage(bool* pval) : BoolBasicParamStorage(pval) {}
      ~BoolParamStorage() {}

      // only success return should be 0
      // the key has already been read
      // otherwise, see the RR_ values above, though RR_NoMatch should not be returned
      int readArgsFrom(const std::string& key,
		       const char* fileid, const char* path,
		       std::istream& is, bool have_argument);
      // returns the number of args processed
      // the key has already been read
      // otherwise, see the RR_ values above, though RR_NoMatch should not be returned
      int readCmdLineArgs(const std::string& key,
			  int argc, const char* const* argv);

      bool setParam (const int& value)
      { *pval = static_cast<bool>(value); return true; }
    };

    class StringParamStorage
      : public StringBasicParamStorage
    {
    public:
      // do not take over memory
      StringParamStorage(std::string* pval) : StringBasicParamStorage(pval) {}
      virtual ~StringParamStorage() {}

      // only success return should be 0
      // the key has already been read
      // otherwise, see the RR_ values above, though RR_NoMatch should not be returned
      int readArgsFrom(const std::string& key,
		       const char* fileid, const char* path,
		       std::istream& is, bool have_argument);
      // returns the number of args processed
      // the key has already been read
      // otherwise, see the RR_ values above, though RR_NoMatch should not be returned
      int readCmdLineArgs(const std::string& key,
			  int argc, const char* const* argv);

      bool setParam (const std::string& val)
      { return set(val, "./"); }

      bool setParam (const std::pair<const std::string, const char*>& val)
      { return set(val.first, val.second); }
      
      virtual bool set(const std::string& val, const char* starting_dir)
      { *pval = val; return true; }
    };

    class FilePathParamStorage
      : public StringParamStorage
    {
    public:
      FilePathParamStorage(std::string* pval) : StringParamStorage(pval) {}
      ~FilePathParamStorage() {}

      bool set(const std::string& val, const char* starting_dir);
    };
    
    class DoubleParamStorage
      : public DoubleBasicParamStorage
    {
    public:
      // do not take over memory
      DoubleParamStorage(double* pval) : DoubleBasicParamStorage(pval) {}
      ~DoubleParamStorage() {}

      // only success return should be 0
      // the key has already been read
      // otherwise, see the RR_ values above, though RR_NoMatch should not be returned
      int readArgsFrom(const std::string& key,
		       const char* fileid, const char* path,
		       std::istream& is, bool have_argument);
      // returns the number of args processed
      // the key has already been read
      // otherwise, see the RR_ values above, though RR_NoMatch should not be returned
      int readCmdLineArgs(const std::string& key,
			  int argc, const char* const* argv);

      bool setParam (const int& value)
      { *pval = static_cast<double>(value); return true; }
    };

    template <class T>
    class VectorParamStorage
      : public ParamStorerByMap::ParamStorage
    {
    public:
      VectorParamStorage(std::vector<T>* pval, int min_cnt, int max_cnt)
	: ParamStorage(), pval(pval), min_cnt(min_cnt), max_cnt(max_cnt) {}
      ~VectorParamStorage() {}

      // only success return should be 0
      // the key has already been read
      // otherwise, see the RR_ values above, though RR_NoMatch should not be returned
      int readArgsFrom(const std::string& key,
		       const char* fileid, const char* path,
		       std::istream& is, bool have_argument)
      {
	if (!readVectorWBounds(is, *pval, min_cnt, max_cnt))
	  {
	    errorlog << fileid << ": Bad number of vector args " << pval->size()
		     << " (min=" << min_cnt << ")"
		     << " (max=" << max_cnt << ")" 
		     << ende;
	    return ParamStorer::RR_FormatErr;
	  }
	return 0; //success!
      }
      
      // returns the number of args processed
      // the key has already been read
      // otherwise, see the RR_ values above, though RR_NoMatch should not be returned
      int readCmdLineArgs(const std::string& key,
			  int argc, const char* const* argv)
      {
	int args_consumed = 0;
  
	std::vector<T> v;

	for (int arg_idx = 1; arg_idx < argc; arg_idx++)
	  {
	    const char* this_arg = *(argv + arg_idx);
	    if (this_arg[0] == '-' && this_arg[1] == '-')
	      break;
	    std::istrstream ifile(this_arg);
	    T val;
	    ifile >> val;
	    if (ifile.fail())
	      {
		return ParamStorer::RR_FormatErr;
	      }
	    v.push_back(val);
	    args_consumed++;
	  }

	if (static_cast<signed>(v.size()) < min_cnt ||
	    (static_cast<signed>(v.size()) > max_cnt &&
	     max_cnt > 0))
	  {
	    errorlog << "Error: Option '" << key << "': Bad number of vector args "
		     << v.size()
		     << " (min=" << min_cnt << ")"
		     << " (max=" << max_cnt << ")"
		     << ende;
	    return ParamStorer::RR_FormatErr;
	  }

	*pval = v;
	return args_consumed;
      }

      bool setParam (const std::vector<T>& value)
      {
	if (static_cast<signed>(value.size()) < min_cnt ||
	    (static_cast<signed>(value.size()) > max_cnt &&
	     max_cnt > 0))
	  {
	    errorlog << "Error: Bad number of vector args in setParam"
		     << value.size()
		     << " (min=" << min_cnt << ")"
		     << " (max=" << max_cnt << ")"
		     << ende;
	    return false;
	  }
	*pval = value;
	return true;
      }

      void print(std::ostream& o) const
      { o << *pval; }

    private:
      std::vector<T>* pval;
      int min_cnt;
      int max_cnt;
    };

    typedef VectorParamStorage<int> VIntParamStorage;
    typedef VectorParamStorage<double> VDoubleParamStorage;
    typedef VectorParamStorage<std::string> VStringParamStorage;
    
    /****************************************************************************/
    /****************************************************************************/
    /****************************************************************************/
  public:

    ParamReader (float max_version);
    virtual ~ParamReader ();

    // returns the directory of the executable.
    // Only valid after getOptions call or after setExecutableDir called
    const std::string& getExecutableDir() const;
    // You should only need to do this if you are not going to be calling
    // getOptions
    void setExecutableDir(const std::string& dir);

    // This class takes over the memory
    void addParamStorer(ParamStorer* p) { vec_param_storer.push_back(p); }
    
    void getOptions (const int &argc, const char *const *argv);

    bool setParam (const std::string & key, const int& value);
    bool setParam (const std::string & key, const bool& value);
    bool setParam (const std::string & key, const char* val)
    { return setParam(key, std::string(val)); }
    bool setParam (const std::string & key, const std::string& value,
		   const char* starting_dir = "./");
    bool setParam (const std::string & key, const double& value);
    bool setParam (const std::string & key, const std::vector<int>& value);
    bool setParam (const std::string & key, const std::vector<double>& value);
    bool setParam (const std::string & key, const std::vector<std::string>& value);
    
    void printAll(std::ostream& out);    
    
    /* This function will be called after the values have been read.
       It may be called several times!
       The only guaruntee that you get is that this function will be called after the last value
       has been read in */
    virtual void postReadProcessing() = 0;

  protected:

    static const int MaxArgLength;

    /* this is the call back for the FileReader */
    bool processLine(std::istrstream& line, const char* fileid, const char* path, float version);

    virtual void addAll2Maps () = 0;
    virtual void setDefaultValues () = 0;

    void add2Maps (const std::string & key, int *value);
    void add2Maps (const std::string & key, std::string *value, bool file_path = false);
    void add2Maps (const std::string & key, bool *value);
    void add2Maps (const std::string & key, double *value);
    void add2Maps (const std::string & key, std::vector<int> *value, int min, int max);
    void add2Maps (const std::string & key, std::vector<double> *value, int min, int max);
    void add2Maps (const std::string & key, std::vector<std::string> *value, int min, int max);
    
  private:
    template <class T> bool templateSetParam (const std::string& key, const T& value);

    // we make these private to insure that postReadProcessing is not called unneccsarily
    void readParams (const char *path) { readFile(path, max_version); }
    
    //getOptions sets up all the appropriate stuff
    void readOptions (const int &argc, const char *const *argv);

    typedef std::vector<ParamStorer*> VecParamStorer;
    VecParamStorer vec_param_storer;
    ParamStorerByMap* p_param_storer_by_map;
    
    //we just use this to make sure that the -file on the command line does not give an error
    std::string conf_file; 

    //stores the directory of the executable
    std::string exec_dir;
    bool exec_dir_set;
    
    float max_version;
  };

} //spades namespace

#endif
