/* -*- Mode: c++ -*- */

/*
 *Copyright:

    Copyright (C) 2002, 2003 Patrick Riley
    Copyright (C) 2001 Patrick Riley and Emil Talpes

    This file is part of the SPADES simulation system.

    The SPADES simulation system is free software; you can
    redistribute it and/or modify it under the terms of the GNU Lesser
    General Public License as published by the Free Software
    Foundation; either version 2 of the License, or (at your option)
    any later version.

    The SPADES simulation system is distributed in the hope that it
    will be useful, but WITHOUT ANY WARRANTY; without even the implied
    warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
    See the GNU Lesser General Public License for more details.

    You should have received a copy of the GNU Lesser General Public
    License along with the SPADES simulation system; if not, write to
    the Free Software Foundation, Inc., 59 Temple Place, Suite 330,
    Boston, MA 02111-1307 USA

 *EndCopyright:
*/

#include <string>
#include <cstdlib>
#include <cstdio>
#include <new>
#include <sstream>
#include <algorithm>
#include "ParamReader.h"
#include "utility.h"
#include "Logger.h"

using namespace std;
using namespace spades;

//a simple macro to turn on debugging since this is too early in the startup process to
// use the normal action logging stuff
#define DEBUG(x) 

const int
spades::  ParamReader::MaxArgLength  = 512;

spades::ParamReader::ParamReader (float max_version)
  : p_param_storer_by_map(new ParamStorerByMap),
    exec_dir(""),
    exec_dir_set(false),
    max_version(max_version)
{
  addParamStorer(p_param_storer_by_map);
}

spades::ParamReader::~ParamReader ()
{
  std::for_each(vec_param_storer.begin(),
		vec_param_storer.end(),
		deleteptr<ParamStorer>());
  //Note that we do NOT need to delete p_param_storer_by_map since
  // it was put into vec_param_storer, whose elements are deleted above
}

// returns the directory of the executable.
// Only valid after getOptions call or after setExecutableDir called
const std::string&
spades::ParamReader::getExecutableDir() const
{
  if (!exec_dir_set)
    {
      errorlog << "Tried to access ParamReader::getExecutableDir without the dir being set!"
	       << ende;
    }
  return exec_dir;
}

// You should only need to do this if you are not going to be calling
// getOptions
void
spades::ParamReader::setExecutableDir(const std::string& dir)
{
  exec_dir = dir;
  exec_dir_set = true;
}


void
spades::ParamReader::getOptions (const int &argc, const char *const *argv)
{
  addAll2Maps ();
  add2Maps("file", &conf_file);
  setDefaultValues ();

  /* process the executable name (which should be in argv[0] */
  exec_dir = resolvePathGivenStartingFile(argv[0], ".");
  exec_dir_set = true;
  
  /* first, search option '-file' */
  for (int i = 1; i < argc; i++)
    {
      if (strcmp (argv[i], "--file") == 0)
        {
          i++;
          readParams (argv[i]);
        }
    }

  readOptions (argc, argv);

  postReadProcessing();

}

/* this is the call back for the FileReader */
bool
spades::ParamReader::processLine(istrstream& line, const char* fileid,
				 const char* path, float version)
{
  string param;
  line >> param;
  if (param.empty ())
    {
      return true;
    }

  bool find_colon = true;
  if (param[param.length () - 1] == ':')
    {
      param.resize (param.length () - 1);
      find_colon = false;
    }

  bool have_argument;

  if (find_colon)
    {
      if (!skip_white_space_on_line(line))
	{
	  have_argument = false;
	  //we don't want the line to be marked as a failure state
	  line.clear();
	}
      else
	{
	  //we already skipped the white space
	  char c;
	  line.get(c);
	  if (line.fail() || c != ':')
	    {
	      line.putback(c);
	      have_argument = false;
	      //we don't want the line to be marked as a failure state
	      line.clear();
	    }
	  else
	    have_argument = true;
	}
    }
  else
    {
      //already found a colon, so we must have an argument
      have_argument = true;
    }

  int read_ret = ParamStorer::RR_NoMatch;
  for (VecParamStorer::iterator iter = vec_param_storer.begin();
       iter != vec_param_storer.end();
       iter++)
    {
      read_ret = (*iter)->readArgsFrom(param, fileid, path, line, have_argument);
      if (read_ret != ParamStorer::RR_NoMatch)
	break;
    }

  switch (read_ret)
    {
    case ParamStorer::RR_None:
      errorlog << fileid << ": How is read_ret RR_None?" << ende;
      return false;
    case ParamStorer::RR_NoMatch:
      errorlog << fileid << ": Unrecognized parameter in file '" << param << "'" << ende;
      return false;
    case ParamStorer::RR_FormatErr:
      errorlog << fileid << ": Badly formatted line for parameter '" << param << "'" << ende;
      return false;
    default:
      // this is success
      break;
    }

  return true;
}



void
spades::ParamReader::readOptions (const int &orig_argc, const char *const orig_argv[])
{
  int argc = orig_argc;
  const char *const *argv = orig_argv;
  char buf[MaxArgLength];

  argc--;
  argv++;

  while (argc > 0)
    {
      if (sscanf (*argv, "--%s", buf) != 1)
        {
          errorlog << "Error: Did not understand argument '" << *argv << "'" << ende;
          argv++;
          argc--;
          continue;
        }

      string param = buf;
      DEBUG(cout << "Processing '" << param.c_str() << "' from command line" << endl);

      int read_ret = ParamStorer::RR_NoMatch;
      for (VecParamStorer::iterator iter = vec_param_storer.begin();
	   iter != vec_param_storer.end();
	   iter++)
	{
	  read_ret = (*iter)->readCmdLineArgs(param, argc, argv);
	  if (read_ret != ParamStorer::RR_NoMatch)
	    break;
	}

      switch (read_ret)
	{
	case ParamStorer::RR_None:
	  errorlog << "ParamReader: How is read_ret RR_None?" << ende;
	  break;
	case ParamStorer::RR_NoMatch:
	  errorlog << "ParamReader: Unrecognized parameter on command line: " << param << ende;
	  break;
	case ParamStorer::RR_FormatErr:
	  errorlog << "ParamReader: Could not read value for parameter '" << param << "'" << ende;
	  break;
	default:
	  // this is success, move past the args processed
	  argv += read_ret;
	  argc -= read_ret;
	  break;
	}
      
      //skip the argument we just read
      argv++;
      argc--;
    }

}


// some temp test code
//namespace spades 
//{  
//std::ostream& operator<<(std::ostream& os, const std::pair<const std::string, const char*>& p)
//{
//  os << "pair(" << p.first << ", " << p.second << ")";
//  return os;
//}
//}


template <class T>
bool
spades::ParamReader::templateSetParam (const std::string& key, const T& value)
{
  //std::cout << "templateSetParam '" << key << "' '" << value << "'" << std::endl;
  for (VecParamStorer::iterator iter = vec_param_storer.begin();
       iter != vec_param_storer.end();
       iter++)
    {
      int ret = (*iter)->setParam(key, value);
      switch (ret)
	{
	case ParamStorer::RR_None:
	  errorlog << "How is setParam return RR_None?" << ende;
	  break;
	case ParamStorer::RR_NoMatch:
	  continue;
	case ParamStorer::RR_FormatErr:
	  errorlog << "Type mismatch or other format error in setParam for key '" << key
		   << "': type is " << typeid(value).name() << ende;
	  return false;
	case 0:
	  return true;
	default:
	  errorlog << "What is setParam returning? " << ret << ende;
	  return false;
	}
    }
  return false;
}


bool
spades::ParamReader::setParam (const std::string & key, const int& value)
{
  return templateSetParam(key, value);
}

bool
spades::ParamReader::setParam (const std::string & key, const bool& value)
{
  return templateSetParam(key, value);
}

bool
spades::ParamReader::setParam (const std::string & key,
			       const std::string& value,
			       const char* starting_dir)
{
  return templateSetParam(key, std::make_pair(value, starting_dir));
}

bool
spades::ParamReader::setParam (const std::string & key, const double& value)
{
  return templateSetParam(key, value);
}

bool
spades::ParamReader::setParam (const std::string & key, const std::vector<int>& value)
{
  return templateSetParam(key, value);
}

bool
spades::ParamReader::setParam (const std::string & key, const std::vector<double>& value)
{
  return templateSetParam(key, value);
}

bool
spades::ParamReader::setParam (const std::string & key, const std::vector<std::string>& value)
{
  return templateSetParam(key, value);
}


void
spades::ParamReader::printAll(ostream& out)
{
  for (VecParamStorer::iterator iter = vec_param_storer.begin();
       iter != vec_param_storer.end();
       iter++)
    {
      (*iter)->print(out);
    }
}

void
spades::ParamReader::add2Maps (const std::string & key, int *value)
{
  p_param_storer_by_map->addParam(key, new IntParamStorage(value));
}

void
spades::ParamReader::add2Maps (const std::string & key, std::string *value,
			       bool file_path)
{
  p_param_storer_by_map->addParam(key,
				  file_path ?
				  (new FilePathParamStorage(value)) :
				  (new StringParamStorage(value)));
}

void
spades::ParamReader::add2Maps (const std::string & key, bool *value)
{
  p_param_storer_by_map->addParam(key, new BoolParamStorage(value));
}

void
spades::ParamReader::add2Maps (const std::string & key, double *value)
{
  p_param_storer_by_map->addParam(key, new DoubleParamStorage(value));
}

void
spades::ParamReader::add2Maps (const std::string & key, std::vector<int> *value, int min, int max)
{
  p_param_storer_by_map->addParam(key, new VIntParamStorage(value, min, max));
}

void
spades::ParamReader::add2Maps (const std::string & key, std::vector<double> *value, int min, int max)
{
  p_param_storer_by_map->addParam(key, new VDoubleParamStorage(value, min, max));
}

void
spades::ParamReader::add2Maps (const std::string & key, std::vector<std::string> *value, int min, int max)
{
  p_param_storer_by_map->addParam(key, new VStringParamStorage(value, min, max));
}


/******************************************************************************************/
/******************************************************************************************/

spades::ParamReader::ParamStorerByMap::ParamStorerByMap()
  : ParamStorer(), map_param()
{
}

spades::ParamReader::ParamStorerByMap::~ParamStorerByMap()
{
  std::for_each(map_param.begin(), map_param.end(),
		deletemapptr<MapParam>());
  map_param.clear();
}


int
spades::ParamReader::ParamStorerByMap::readArgsFrom(const std::string& key,
						    const char* fileid,
						    const char* path,
						    std::istream& is,
						    bool have_argument)
{
  MapParam::iterator iter = map_param.find(key);
  if (iter == map_param.end())
    return RR_NoMatch;
  return iter->second->readArgsFrom(key, fileid, path, is, have_argument);
}

// returns the number of args processed
int
spades::ParamReader::ParamStorerByMap::readCmdLineArgs(const std::string& key,
						       int argc,
						       const char* const* argv)
{
  MapParam::iterator iter = map_param.find(key);
  if (iter == map_param.end())
    return RR_NoMatch;
  return iter->second->readCmdLineArgs(key, argc, argv);
}


// returns whether this storer set the data for this
// return 0 on success,
//        RR_FormatErr on type mismatch
//        RR_NoMatch if key DNE
template <class T>
int
spades::ParamReader::ParamStorerByMap::templateSetParam (const std::string& key,
							 const T& value)
{
  MapParam::iterator iter = map_param.find(key);
  if (iter == map_param.end())
    return RR_NoMatch;
  return iter->second->setParam(value) ? 0 : RR_FormatErr;
}

int
spades::ParamReader::ParamStorerByMap::setParam (const std::string & key,
						 const int& value)
{
  return templateSetParam(key, value);
}

int
spades::ParamReader::ParamStorerByMap::setParam (const std::string & key,
						 const bool& value)
{
  return templateSetParam(key, value);
}

int
spades::ParamReader::ParamStorerByMap::setParam (const std::string & key,
						 const std::string& value)
{
  return templateSetParam(key, value);
}

int
spades::ParamReader::ParamStorerByMap::setParam (const std::string & key,
						 const std::pair<const std::string, const char*>& value)
{
  return templateSetParam(key, value);
}

int
spades::ParamReader::ParamStorerByMap::setParam (const std::string & key,
						 const double& value)
{
  return templateSetParam(key, value);
}

int
spades::ParamReader::ParamStorerByMap::setParam (const std::string & key,
						 const std::vector<int>& value)
{
  return templateSetParam(key, value);
}

int
spades::ParamReader::ParamStorerByMap::setParam (const std::string & key,
						 const std::vector<double>& value)
{
  return templateSetParam(key, value);
}

int
spades::ParamReader::ParamStorerByMap::setParam (const std::string & key,
						 const std::vector<std::string>& value)
{
  return templateSetParam(key, value);
}


// returns whether a parameter already had this name
bool
spades::ParamReader::ParamStorerByMap::addParam (const std::string& key,
						 ParamStorage* p)
{
  MapParam::iterator iter = map_param.find(key);
  bool ret = false;
  if (iter != map_param.end())
    {
      warninglog(10) << "I am replacing the parameter value for '" << key << "'" << ende;
      delete iter->second;
    }
  map_param[key] = p;
  return ret;
}


void
spades::ParamReader::ParamStorerByMap::print(std::ostream& o) const
{
  for (MapParam::const_iterator iter = map_param.begin();
       iter != map_param.end();
       iter++)
    {
      o << iter->first << ": ";
      iter->second->print(o);
      o << endl;
    }
}

/******************************************************************************************/
/******************************************************************************************/
// only success return should be 0
int
spades::ParamReader::IntParamStorage::readArgsFrom(const std::string& key,
						   const char* fileid,
						   const char* path,
						   std::istream& is,
						   bool have_argument)
{
  if (!have_argument)
    return ParamStorer::RR_FormatErr;

  int i;
  is >> i;
  if (is.fail())
    return ParamStorer::RR_FormatErr;
  *pval = i;
  return 0; //success
}

// returns the number of args processed
int
spades::ParamReader::IntParamStorage::readCmdLineArgs(const std::string& key,
						      int argc, const char* const* argv)
{
  if (argc <= 1)
    {
      errorlog << "Error: Int option '" << key << "' has no int value specified" << ende;
      return ParamStorer::RR_FormatErr;
    }
  
  istrstream iline(*(argv + 1));
  int i;
  iline >> i;
  if (iline.fail())
    return ParamStorer::RR_FormatErr;
  *pval = i;
  DEBUG(cout << "For '" << key << "' "  << " int " << *pval << endl);
  return 1; //1 arg consumed
}

/******************************************************************************************/
/******************************************************************************************/
// only success return should be 0
int
spades::ParamReader::BoolParamStorage::readArgsFrom(const std::string& key,
						    const char* fileid,
						    const char* path,
						    std::istream& is,
						    bool have_argument)
{
  if (!have_argument)
    {
      *pval = true;
      return 0; //success
    }
  
  string value;
  is >> value;
  if (value == "on" || value == "true")
    *pval = true;
  else if (value == "off" || value == "false")
    *pval = false;
  else
    {
      errorlog << fileid 
	       << ": Expecting 'on', 'off', 'true' or 'false', not '" << value.c_str()
	       << "'" << ende;
      return ParamStorer::RR_FormatErr;
    }
  return 0; //success
}


// returns the number of args processed
int
spades::ParamReader::BoolParamStorage::readCmdLineArgs(const std::string& key,
						       int argc, const char* const* argv)
{
  if (argc <= 1 || (**(argv+1) == '-'))
    {
      *pval = true;
      DEBUG(cout << "For '" << key << "' "  << " bool default" << endl);
      return 0; //no args consumed
    }
  
  string value;
  istrstream ifile(*(argv + 1));
  ifile >> value;
  if (ifile.fail())
    {
      return ParamStorer::RR_FormatErr;
    }

  if (value == "on" || value == "true")
    {
      *pval = true;
      DEBUG(cout << "For '" << key << "' "  << " onoff on" << endl);
    }
  else if (value == "off" || value == "false")
    {
      *pval = false;
      DEBUG(cout << "For '" << key << "' "  << " onoff off" << endl);
    }
  else
    {
      errorlog << "Expecting 'on', 'off', 'true' or 'false', not '"
	       << value << "' for command line option '" << key << "'"
	       << ende;
      return ParamStorer::RR_FormatErr;
    }
  return 1; //1 arg consumed
}

/******************************************************************************************/
/******************************************************************************************/
// only success return should be 0
int
spades::ParamReader::DoubleParamStorage::readArgsFrom(const std::string& key,
						      const char* fileid,
						      const char* path,
						      std::istream& is,
						      bool have_argument)
{
  if (!have_argument)
    return ParamStorer::RR_FormatErr;
  double d;
  is >> d;
  if (is.fail())
    return ParamStorer::RR_FormatErr;
  *pval = d;
  return 0; //success
}


// returns the number of args processed
int
spades::ParamReader::DoubleParamStorage::readCmdLineArgs(const std::string& key,
							 int argc, const char* const* argv)
{
  if (argc <= 1)
    {
      errorlog << "Error: Double option '" << key << "' has no double value specified" << ende;
      return ParamStorer::RR_FormatErr;
    }
  
  istrstream iline(*(argv + 1));
  double d;
  iline >> d;
  if (iline.fail())
    return ParamStorer::RR_FormatErr;
  *pval = d;
  DEBUG(cout << "For '" << key << "' "  << " double " << *pval << endl);
  return 1; //1 arg consumed
}

/******************************************************************************************/
/******************************************************************************************/
// only success return should be 0
int
spades::ParamReader::StringParamStorage::readArgsFrom(const std::string& key,
						      const char* fileid,
						      const char* path,
						      std::istream& is,
						      bool have_argument)
{
  if (!have_argument)
    {      
      return set("", path) ? 0 : ParamStorer::RR_FormatErr;
    }

  skip_white_space(is);
  //Grab the rest of the line
  std::stringstream str_line;
  is.get(*str_line.rdbuf());
  return set(str_line.str(), path) ? 0 : ParamStorer::RR_FormatErr;
}


// returns the number of args processed
int
spades::ParamReader::StringParamStorage::readCmdLineArgs(const std::string& key,
							 int argc,
							 const char* const* argv)
{
  if (argc <= 1)
    {
      errorlog << "Error: Option '" << key << "' has no string value specified" << ende;
      return ParamStorer::RR_FormatErr;
    }
  
  return set(std::string(*(argv + 1)), "./") ? 1 : ParamStorer::RR_FormatErr;
}

/******************************************************************************************/
/******************************************************************************************/

bool
spades::ParamReader::FilePathParamStorage::set(const std::string& v,
					       const char* starting_dir)
{
  // if empty, just set it
  if (v.empty())
    *pval = v;
    
  std::string true_path;
  // if we start with %, then do no processing
  if (v[0] == '%')
    true_path = v;
  // & means that not to resolve this path, but do remove the ampersand
  else if (v[0] == '&')
    true_path.assign(v, 1, v.length() - 1);
  // do our normal resolve operation
  else
    true_path = createAbsPathGivenStartingFile(starting_dir, v.c_str());
  if (true_path.empty())
    {
      warninglog(10) << "Failed to resolve path '" << v << "' during parameter parsing" << ende;
      return false;
    }
  *pval = true_path;
  return true;
}

/******************************************************************************************/
/******************************************************************************************/
