
#include <ios>
#include "OnlineRunner.h"
#include "CoachCommand.h"
#include "Module.h"
#include "ModuleRegistry.h"
#include "Logger.h"
#include "CoachParam.h"
#include "utility.h"

using namespace spades;
using namespace std;

/*************************************************************************************/
/* misc utility stuff */

class OnlineCaller : std::unary_function<Module*,void>
{
public:
  OnlineCaller(OnlineRunner* p) : p(p) {}
  void operator() (Module* m) { m->stateUpdateOnlineNotify(*p, p->getWorldHistory()); }
private:
  OnlineRunner* p;
};

class OnlineShutdownCaller : std::unary_function<Module*,void>
{
public:
  OnlineShutdownCaller(OnlineRunner* p) : p(p) {}
  void operator() (Module* m) { m->shutdownOnlineNotify(*p, p->getWorldHistory()); }
private:
  OnlineRunner* p;
};


/*************************************************************************************/

OnlineRunner::OnlineRunner(ModuleRegistry* pReg)
  : Runner(pReg),
    history(CoachParam::instance()->getCyclesToStore()),
    thrd_parser(),
    cv_parser_done(),
    mutex_parser_done(),
    parser_done(false),
    server_addr( CoachParam::instance()->getOnlinePort(), CoachParam::instance()->getOnlineHost() ),
    socket(),
    send_log((CoachParam::instance()->getLogfileDir() + '/' + CoachParam::instance()->getSendLogFN()).c_str()),
    recv_log((CoachParam::instance()->getLogfileDir() + '/' + CoachParam::instance()->getRecvLogFN()).c_str()),
    write_strmbuf( send_log, socket, server_addr ),
    read_strmbuf( recv_log, socket, server_addr ),
    write_strm( &write_strmbuf ),
    read_strm ( &read_strmbuf ),
    parser( *this ),
    player_type_storage(),
    my_side(TS_None),
    run_state(RS_Startup)
{
  if (!send_log && !CoachParam::instance()->getSendLogFN().empty())
    errorlog << "Could not open send log '" << CoachParam::instance()->getSendLogFN().empty()
	     << "'" << ende;
  if (!recv_log && !CoachParam::instance()->getRecvLogFN().empty())
    errorlog << "Could not open recv log '" << CoachParam::instance()->getRecvLogFN().empty()
	     << "'" << ende;
  
  if (CoachParam::instance()->getCreateCommandLog())
    {
      std::string fn(CoachParam::instance()->getLogfileDir() + '/' +
		     CoachParam::instance()->getCommandLogFN());
      command_outfile.open(fn.c_str());
      if (!command_outfile)
	errorlog << "Could not open the command outfile '" << fn << "'" << ende;
    }

  if (pthread_cond_init(&cv_parser_done, NULL) != 0)
    errorlog << "Could not initialize cv_parser_done" << ende;
  if (pthread_mutex_init(&mutex_parser_done, NULL) != 0)
    errorlog << "Could not initialize mutex_parser_done" << ende;

  CoachTagFunction* pTag = dynamic_cast<CoachTagFunction*>(spades::Logger::instance()->getTagFunction());
  if (pTag == NULL)
    {
      errorlog << "OnlineRunner: how is the logger tag function not a CoachTagFunction?" << ende;
    }
  else
    {
      pTag->setTimeTagger(this);
      actionlog(100) << "OnlineRunner: set the logger time tag function" << ende;
    }

  /*
  std::cout << "Parser addr at construct: "
	    << &parser << "\t"
	    << std::endl;
  */
}

OnlineRunner::~OnlineRunner()
{
  int errcode;
  if ((errcode=pthread_mutex_destroy(&mutex_parser_done)) != 0)
    errorlog << "Error detroying mutex_parser_done: " << errcode << ende;
  if ((errcode=pthread_cond_destroy(&cv_parser_done)) != 0)
    errorlog << "Error detroying cv_parser_done:" << errcode << ende;
  command_outfile.close();

  CoachTagFunction* pTag = dynamic_cast<CoachTagFunction*>(spades::Logger::instance()->getTagFunction());
  if (pTag == NULL)
    {
      errorlog << "OnlineRunner: how is the logger tag function not a CoachTagFunction?" << ende;
    }
  else
    {
      pTag->resetTimeTagger();
      actionlog(100) << "OnlineRunner: reset the logger time tag function" << ende;
    }
}

bool
OnlineRunner::sendToServer(const CoachCommand* pCom)
{
  write_strm << *pCom << ends << flush;
  actionlog(150) << "Sending: " << *pCom << ende;
  if (CoachParam::instance()->getCreateCommandLog())
    {
      command_outfile << (history.getNumAvail() ? spades::toString(history.getWorldState().getTime()) : "I")
		      << ": " << *pCom << endl;
    }
  return write_strm.good();
}

TeamSide
OnlineRunner::getSideOf(const char* team_name)
{
  if (left_team_name.empty())
    left_team_name = team_name;
  if (left_team_name == team_name)
    return TS_Left;
  if (right_team_name.empty())
    right_team_name = team_name;
  if (right_team_name == team_name)
    return TS_Right;
  return TS_None;
}


void
OnlineRunner::prot_initiateShutdown()
{
  int errcode = pthread_cancel(thrd_parser);
  //if we are getting this before the thread started or after it's done, it's fine
  if (errcode != 0 && errcode != ESRCH)
    errorlog << "Error in pthread_cancel: " << errcode << ende;
  // we need to wait for the thread to finish and everything. This isn't really the best way to do
  // this, but I think it will be fine
  sleep(1);
}

// this method should call all of the the dependency check methods of the appropriate
// type
bool
OnlineRunner::dependencyCheck(ModuleRegistry* pReg)
{
  DependencyCaller<OnlineRunner> caller(pReg, *this);
  for_each(pReg->getStorage().begin(), pReg->getStorage().end(), caller);
  return getDependencySuccess();
}

// the method to enter the loop and do all the work
bool
OnlineRunner::run()
{
  int errcode;
  
  run_state = RS_Init;
  
  CoachCommandInit init_com(CoachParam::instance()->getTeamName(),
			    CoachParam::instance()->getCoachName(),
			    CoachParam::instance()->getServerProtocolVersion());
  if (!sendToServer(&init_com))
    {
      errorlog << "Error sending init command" << ende;
      return false;
    }

  errcode = pthread_create(&thrd_parser, NULL, &startParser, this);
  if (errcode != 0)
    {
      errorlog << "Error creating parser thread: " << errcode << ende;
      return false;
    }

  std::cout << "OnlineRunner: send init, waiting to connect" << std::endl;
  
  while (!isShutdownRequested())
    {
      struct timeval tv;
      struct timezone tz;
      struct timespec ts;
      
      errcode = pthread_mutex_lock(&mutex_parser_done);
      if (errcode != 0)
	errorlog << "Could not lock mutex_parser_done: " << errcode << ende;
      while (!parser_done) {
	do
	  {
	    errcode = gettimeofday(&tv, &tz);
	  }
	while (errcode == -1 && errno == EINTR);
	if (errcode != 0)
	  errorlog << "Error in gettimeofday: " << errno << " " << strerror(errno) << ende;
	ts.tv_sec = tv.tv_sec + CoachParam::instance()->getOnlineMaxTimeWOutMsg();
	ts.tv_nsec = tv.tv_usec * 1000;
	errcode = pthread_cond_timedwait(&cv_parser_done, &mutex_parser_done, &ts);
	if (errcode == ETIMEDOUT)
	  break;
      }

      if (errcode == ETIMEDOUT)
	{
          struct timeval tv_now;
          if (gettimeofday(&tv_now, &tz) != 0)
            errorlog << "In time out, gettimeofday failed" << ende;
          
	  warninglog(10) << "Time out waiting to hear from server! "
                         << tv << " -> " << ts.tv_sec << ":" << ts.tv_nsec
                         << " -> " << tv_now
                         << ende;
	  initiateShutdown();
	  errcode = pthread_mutex_unlock(&mutex_parser_done);
	  if (errcode != 0)
	    errorlog << "On shutdown, could not unlock mutex_parser_done: " << errcode << ende;
	  continue;
	}
      
      handleParserDone();
      
      parser_done = false;

      errcode = pthread_cond_signal(&cv_parser_done);
      if (errcode != 0)
	errorlog << "Error in cond signal: " << errcode << ende;

      errcode = pthread_mutex_unlock(&mutex_parser_done);
      if (errcode != 0)
	errorlog << "Could not unlock mutex_parser_done: " << errcode << ende;
    }
  
  //We'll try to send a bye command, but who cares if it works
  CoachCommandBye bye_com;
  if (!sendToServer(&bye_com))
    {
      actionlog(100) << "My bye command could not be sent" << ende;
    }

  for_each(pReg->getStorage().begin(), pReg->getStorage().end(), OnlineShutdownCaller(this));

  return true;
}

void
OnlineRunner::handleParserDone()
{
  switch (run_state)
    {
    case RS_Startup:
      errorlog << "Should not be in RS_Startup in handleParserDone" << ende;
      break;
      
    case RS_Init: {
      if (parser.getLastMessageType() != OnlineParser::MT_Init)
	{
	  errorlog << "Did not get init in response to init message: " << parser.getLastMessageType() << ende;
	  initiateShutdown();
	  break;
	}

      std::cout << "OnlineRunner: connected" << std::endl;
      actionlog(30) << "Initialization done, sending eye command" << ende;
      
      CoachCommandEye eye_com(true);
      if (!sendToServer(&eye_com))
	{
	  errorlog << "Error sending eye command" << ende;
	  initiateShutdown();
	  break;
	}
      run_state = RS_Running;
    } break;

    case RS_Running: {
      switch (parser.getLastMessageType())
	{
	case OnlineParser::MT_None: {
          struct timeval tv;
          gettimeofday(&tv, NULL);
	  actionlog(200) << "Got an MT_None message: " << tv << ende;
        } break;
	case OnlineParser::MT_ServerParam:
	  parser.storeParamsTo(ServerParam::instance());
	  //The slow_down_factor really isn't handled correctly. This compensates for that
	  ServerParam::instance()->fixSPHalfCyclesFromServer();
	  break;
	case OnlineParser::MT_PlayerParam:
	  parser.storeParamsTo(PlayerParam::instance());
	  //The slow_down_factor really isn't handled correctly. This compensates for that
	  // We have to do this here because they are all one object
	  ServerParam::instance()->fixSPHalfCyclesFromServer();
	  break;
	case OnlineParser::MT_PlayerType:
	  parser.storePlayerTypeTo(player_type_storage);
	  break;
	case OnlineParser::MT_Visual:
	  // test code for the player storage
	  //cout << player_type_storage << endl;
	  history.finishPending();
	  actionlog(30) << "Got a visual, calling modules" << ende;
	  for_each(pReg->getStorage().begin(), pReg->getStorage().end(), OnlineCaller(this));
	  break;
	case OnlineParser::MT_Other:
	  actionlog(200) << "Got an MT_Other message" << ende;
	  break;
	case OnlineParser::MT_Error:
	  // should have already come out as an errorlog
	  actionlog(200) << "Got an MT_Error message" << ende;
	  break;
	default:
	  errorlog << "What is the msg type? " << parser.getLastMessageType() << ende;
	}
    } break;

    default:
      errorlog << "What is the run state? " << run_state << ende;
      initiateShutdown();
      break;
    }

  actionlog(100) << "Done processing message: " << parser.getLastMessageType() << ende;
}


static
void
cleanupSignalParserDone(void* p)
{
  ((OnlineRunner*)p)->cleanupSignalParserDone();
}

void
OnlineRunner::cleanupSignalParserDone()
{
  /* I'm not really sure if this is safe because we don't have the mutex_parser_done */
  parser_done = true;

  int errcode = pthread_cond_signal(&cv_parser_done);
  if (errcode != 0)
    errorlog << "cleanupSignalParserDone error in cond signal: " << errcode << ende;
}

void
OnlineRunner::runParser()
{
  int errcode;

  pthread_cleanup_push(&::cleanupSignalParserDone, this);
  errcode = pthread_setcanceltype(PTHREAD_CANCEL_ASYNCHRONOUS, NULL);
  if (errcode != 0)
    errorlog << "runParser: could not set cancel type: " << errcode << ende;
  //This is actually a loop that keeps parsing messages
  if (!parser.parse(read_strm))
    {
      errorlog << "Error in parse loop" << ende;
    }
  pthread_cleanup_pop(true);
}

//the arg should be an OnlineRunner)
//static
void*
OnlineRunner::startParser(void* arg)
{
  ((OnlineRunner*)arg)->runParser();
  return NULL;
}

