/* -*- Mode: C++ -*- */

/* This is a parser class for commands received when connected online */

#include <pthread.h>
#include "OnlineParser.h"
#include "OnlineRunner.h"
#include "ServerParam.h"
#include "PlayerParam.h"
#include "Logger.h"

using namespace std;
using namespace spades;

/*************************************************************************************/

std::ostream&
operator<<(std::ostream& o, OnlineParser::MsgType t)
{
  const char* MSG_TYPE_STRINGS[] = 
    { "None",
      "Init",
      "ServerParam",
      "PlayerParam",
      "PlayerType",
      "Visual",
      "Error",
      "Warning",
      "Other" 
    };

  if (t < 0 || t > 8)
    o << "MsgTypeInvalid(" << t << ")";
  else
    o << MSG_TYPE_STRINGS[(int)t];
  return o;
}


/*************************************************************************************/

OnlineParser::OnlineParser(OnlineRunner& runner)
  : rcc::Parser(clang_parser), runner(runner), last_msg_type(MT_None),
    clang_builder(), clang_parser(clang_builder)
{
}

OnlineParser::~OnlineParser()
{
}

/***********************/
void
OnlineParser::doMsgParsed()
{
  int errcode;
  errcode = pthread_mutex_lock(&runner.mutex_parser_done);
  if (errcode != 0)
    errorlog << "OnlineParser: Could not lock mutex_parser_done: " << errcode << ende;

  runner.parser_done = true;
  
  errcode = pthread_cond_signal(&runner.cv_parser_done);
  if (errcode != 0)
    errorlog << "OnlineParser: error in cond signal: " << errcode << ende;

  //we don't time out here; we let the runner thread handle that
  while (runner.parser_done) {
    pthread_cond_wait(&runner.cv_parser_done, &runner.mutex_parser_done);
  }

  errcode = pthread_mutex_unlock(&runner.mutex_parser_done);
  if (errcode != 0)
    errorlog << "OnlineParser: Could not unlock mutex_parser_done: " << errcode << ende;

  last_msg_type = MT_None;
}

/***********************/
template<class T> class ParamStorer : std::unary_function<std::pair<std::string,T>,void>
{
public:
  ParamStorer(spades::ParamReader* p) : p(p) {}
  void operator() (const std::pair<std::string,T>& val)
  { p->setParam(val.first, val.second); }
private:
  spades::ParamReader* p;
};

void
OnlineParser::storeParamsTo(spades::ParamReader* paramreader)
{
  if (last_msg_type != MT_ServerParam && last_msg_type != MT_PlayerParam)
    errorlog << "storeParamsTo: what is the last_msg_type? " << last_msg_type << ende;
  for_each(last_int_param.begin(), last_int_param.end(), ParamStorer<int>(paramreader));
  for_each(last_double_param.begin(), last_double_param.end(), ParamStorer<double>(paramreader));
  for_each(last_string_param.begin(), last_string_param.end(), ParamStorer<std::string>(paramreader));
  paramreader->postReadProcessing();
  clearLastParam();
}

//doubles sometimes get parsed as ints, so we have to check both
double
OnlineParser::getDoubleParamVal(const char* key)
{
  if (last_double_param.count(key))
    return last_double_param[key];
  if (last_int_param.count(key))
    return (double)last_int_param[key];
  errorlog << "getDoubleParamVal: could not find param in either map" << ende;
  return 0.0;
}


void
OnlineParser::storePlayerTypeTo(PlayerTypeStorage& storage)
{
  if (last_msg_type != MT_PlayerType)
    errorlog << "addPlayerTypeTo: what is the last_msg_type? " << last_msg_type << ende;
  storage.addPlayerType(last_int_param["id"],
			HeteroPlayerType( getDoubleParamVal("player_speed_max"),
					  getDoubleParamVal("stamina_inc_max"),
					  getDoubleParamVal("player_decay"),
					  getDoubleParamVal("inertia_moment"),
					  getDoubleParamVal("dash_power_rate"),
					  getDoubleParamVal("player_size"),
					  getDoubleParamVal("kickable_margin"),
					  getDoubleParamVal("kick_rand"),
					  getDoubleParamVal("extra_stamina"),
					  getDoubleParamVal("effort_max"),
					  getDoubleParamVal("effort_min") ));
  clearLastParam();
}


void
OnlineParser::clearLastParam()
{
  last_int_param.clear();
  last_double_param.clear();
  last_string_param.clear();
}


//Parameter stuff
void
OnlineParser::doBuildServerParam()
{
  actionlog(150) << "OnlineParser: doBuildServerParam" << ende;
  last_msg_type = MT_ServerParam;
}

void
OnlineParser::doBuildPlayerParam()
{
  actionlog(150) << "OnlineParser: doBuildPlayerParam" << ende;
  last_msg_type = MT_PlayerParam;
}

void
OnlineParser::doBuildParam( const std::string& name, int value )
{
  actionlog(150) << "OnlineParser: Param(s): '" << name << "' = " << value << ende;
  last_int_param[name] = value;
}

void
OnlineParser::doBuildParam( const std::string& name, double value )
{
  actionlog(150) << "OnlineParser: Param(d): '" << name << "' = " << value << ende;
  last_double_param[name] = value;
}

void
OnlineParser::doBuildParam( const std::string& name, const std::string& value )
{
  actionlog(150) << "OnlineParser: Param(s): '" << name << "' = " << value << ende;
  last_string_param[name] = value;
}


void
OnlineParser::doBuildPlayerType()
{
  actionlog(150) << "OnlineParser: doBuildPlayerType" << ende;
  last_msg_type = MT_PlayerType;
}


/***********************/
//Init
void
OnlineParser::doBuildCoachInit()
{
  actionlog(150) << "OnlineParser: doBuildInit called " << last_side << ende;
  runner.setMySide(last_side);
  last_msg_type = MT_Init;
}


/***********************/
// Global viz
void
OnlineParser::doBuildGlobalVisual( int time )
{
  actionlog(150) << "OnlineParser: doBuildGlobalVisual called " << time << ende;
  runner.history.getPendingWorldState().setTime(time);
  last_msg_type = MT_Visual;
}

void
OnlineParser::doBuildGlobalGoal( double x, double y )
{
  //Do nothing; I don't know why goals are included at all
}

void
OnlineParser::doBuildGlobalBall( double x, double y, double delta_x, double delta_y )
{
  actionlog(210) << "OnlineParser: Adding ball: " << ende;
  BallInfo ball(VecPosition(x,y),
		VecPosition(getErrorFor(x, DIGITS_OF_PRECISION), getErrorFor(y, DIGITS_OF_PRECISION)),
		VecPosition(delta_x, delta_y),
		VecPosition(getErrorFor(delta_x, DIGITS_OF_PRECISION), getErrorFor(delta_y, DIGITS_OF_PRECISION)));
  runner.history.getPendingWorldState().setBall(ball);
}

//there are two other variants of doBuildGLobalPlayer that have more info, but I'm
//letting that be thrown away
void
OnlineParser::doBuildGlobalPlayer( double x, double y,
				   double delta_x, double delta_y,
				   double orientation, double head_orientation )
{
  actionlog(210) << "OnlineParser: Adding player: " << last_team_name << " " << last_unum << ende;

  TeamSide side = runner.getSideOf(last_team_name.c_str());
  if (side == TS_None)
    {
      errorlog << "I don't know this team name '" << last_team_name << "'" << ende;
      return;
    }
  
  PlayerInfo player( side, last_unum,
		     VecPosition(x,y),
		     VecPosition(getErrorFor(x, DIGITS_OF_PRECISION), getErrorFor(y, DIGITS_OF_PRECISION)),
		     VecPosition(delta_x, delta_y),
		     VecPosition(getErrorFor(delta_x, DIGITS_OF_PRECISION), getErrorFor(delta_y, DIGITS_OF_PRECISION)),
		     ErrBoundValue(orientation, getErrorFor(orientation, DIGITS_OF_PRECISION)),
		     ErrBoundValue(head_orientation, getErrorFor(head_orientation, DIGITS_OF_PRECISION)),
		     last_goalie );

  runner.history.getPendingWorldState().setPlayer(player);
		     
  last_goalie = false;
  last_team_name = "";
  last_unum = -1;
}

/*******************/
// Play Modes
void
OnlineParser::doBuildCatchBallLeftPlayMode()
{
  actionlog(150) << "OnlineParser: play mode catch ball left" << ende;
  runner.history.setLastGoalieCatchTime();
}
void
OnlineParser::doBuildCatchBallRightPlayMode()
{
  actionlog(150) << "OnlineParser: play mode catch ball right" << ende;
  runner.history.setLastGoalieCatchTime();
}
void
OnlineParser::doBuildBeforeKickOffPlayMode()
{
  actionlog(150) << "OnlineParser: play mode bko" << ende;
  runner.history.getPendingWorldState().setPlayMode(PM_BeforeKickOff);
  last_msg_type = MT_Other;
}
void
OnlineParser::doBuildTimeOverPlayMode()
{
  actionlog(150) << "OnlineParser: play mode time over" << ende;
  runner.history.getPendingWorldState().setPlayMode(PM_TimeOver);
  last_msg_type = MT_Other;
}
void
OnlineParser::doBuildPlayOnPlayMode()
{
  actionlog(150) << "OnlineParser: play mode play on" << ende;
  runner.history.getPendingWorldState().setPlayMode(PM_PlayOn);
  last_msg_type = MT_Other;
}
void
OnlineParser::doBuildKickOffLeftPlayMode()
{
  actionlog(150) << "OnlineParser: play mode ko left " << ende;
  runner.history.getPendingWorldState().setPlayMode(PM_KickOff_Left);
  last_msg_type = MT_Other;
}
void
OnlineParser::doBuildKickOffRightPlayMode()
{
  actionlog(150) << "OnlineParser: play mode ko right " << ende;
  runner.history.getPendingWorldState().setPlayMode(PM_KickOff_Right);
  last_msg_type = MT_Other;
}
void
OnlineParser::doBuildKickInLeftPlayMode()
{
  actionlog(150) << "OnlineParser: play mode ki left " << ende;
  runner.history.getPendingWorldState().setPlayMode(PM_KickIn_Left);
  last_msg_type = MT_Other;
}
void
OnlineParser::doBuildKickInRightPlayMode()
{
  actionlog(150) << "OnlineParser: play mode ki right" << ende;
  runner.history.getPendingWorldState().setPlayMode(PM_KickIn_Right);
  last_msg_type = MT_Other;
}
void
OnlineParser::doBuildFreeKickLeftPlayMode()
{
  actionlog(150) << "OnlineParser: play mode fk left " << ende;
  runner.history.getPendingWorldState().setPlayMode(PM_FreeKick_Left);
  last_msg_type = MT_Other;
}
void
OnlineParser::doBuildFreeKickRightPlayMode()
{
  actionlog(150) << "OnlineParser: play mode fk right " << ende;
  runner.history.getPendingWorldState().setPlayMode(PM_FreeKick_Right);
  last_msg_type = MT_Other;
}
void
OnlineParser::doBuildCornerKickLeftPlayMode()
{
  actionlog(150) << "OnlineParser: play mode ck left " << ende;
  runner.history.getPendingWorldState().setPlayMode(PM_CornerKick_Left);
  last_msg_type = MT_Other;
}
void
OnlineParser::doBuildCornerKickRightPlayMode()
{
  actionlog(150) << "OnlineParser: play mode ck right " << ende;
  runner.history.getPendingWorldState().setPlayMode(PM_CornerKick_Right);
  last_msg_type = MT_Other;
}
void
OnlineParser::doBuildGoalKickLeftPlayMode()
{
  actionlog(150) << "OnlineParser: play mode gk left " << ende;
  runner.history.getPendingWorldState().setPlayMode(PM_GoalKick_Left);
  last_msg_type = MT_Other;
}
void
OnlineParser::doBuildGoalKickRightPlayMode()
{
  actionlog(150) << "OnlineParser: play mode gk right " << ende;
  runner.history.getPendingWorldState().setPlayMode(PM_GoalKick_Right);
  last_msg_type = MT_Other;
}
void
OnlineParser::doBuildAfterGoalLeftPlayMode()
{
  actionlog(150) << "OnlineParser: play mode ag left " << ende;
  runner.history.getPendingWorldState().setPlayMode(PM_AfterGoal_Left);
  last_msg_type = MT_Other;
}
void
OnlineParser::doBuildAfterGoalRightPlayMode()
{
  actionlog(150) << "OnlineParser: play mode ag right " << ende;
  runner.history.getPendingWorldState().setPlayMode(PM_AfterGoal_Right);
  last_msg_type = MT_Other;
}
void
OnlineParser::doBuildDropBallPlayMode()
{
  actionlog(150) << "OnlineParser: play mode drop ball " << ende;
  runner.history.getPendingWorldState().setPlayMode(PM_Drop_Ball);
  last_msg_type = MT_Other;
}
void
OnlineParser::doBuildOffSideLeftPlayMode()
{
  actionlog(150) << "OnlineParser: play mode off left " << ende;
  runner.history.getPendingWorldState().setPlayMode(PM_OffSide_Left);
  last_msg_type = MT_Other;
}
void
OnlineParser::doBuildOffsideRightPlayMode()
{
  actionlog(150) << "OnlineParser: play mode off right " << ende;
  runner.history.getPendingWorldState().setPlayMode(PM_OffSide_Right);
  last_msg_type = MT_Other;
}
void
OnlineParser::doBuildPenaltyKickLeftPlayMode()
{
  runner.history.getPendingWorldState().setPlayMode(PM_PK_Left);
  warninglog(10) << "I think I got a small size play mode, which will probably break me" << ende;
  last_msg_type = MT_Other;
}
void
OnlineParser::doBuildPenaltyKickRightPlayMode()
{
  runner.history.getPendingWorldState().setPlayMode(PM_PK_Right);
  warninglog(10) << "I think I got a small size play mode, which will probably break me" << ende;
  last_msg_type = MT_Other;
}
void
OnlineParser::doBuildFirstHalfOverPlayMode()
{
  runner.history.getPendingWorldState().setPlayMode(PM_FirstHalfOver);
  // pfr: I think this is actually ok, but I'm not quite sure
  //warninglog(10) << "I think I got a small size play mode, which will probably break me" << ende;
  last_msg_type = MT_Other;
}
void
OnlineParser::doBuildPausePlayMode()
{
  runner.history.getPendingWorldState().setPlayMode(PM_Pause);
  warninglog(10) << "I think I got a small size play mode, which will probably break me" << ende;
  last_msg_type = MT_Other;
}
void
OnlineParser::doBuildHumanPlayMode()
{
  runner.history.getPendingWorldState().setPlayMode(PM_Human);
  warninglog(10) << "I think I got a small size play mode, which will probably break me" << ende;
  last_msg_type = MT_Other;
}
void
OnlineParser::doBuildFoulLeftPlayMode()
{
  actionlog(150) << "Got a foul_l play mode (I'm expecting a free_kick_r next)" << ende;
}
void
OnlineParser::doBuildFoulRightPlayMode()
{
  actionlog(150) << "Got a foul_r play mode (I'm expecting a free_kick_l next)" << ende;
}
void
OnlineParser::doBuildFoulChargeLeftPlayMode()
{
  runner.history.getPendingWorldState().setPlayMode(PM_Foul_Charge_Left);
  warninglog(10) << "I think I got a small size play mode, which will probably break me" << ende;
  last_msg_type = MT_Other;
}
void
OnlineParser::doBuildFoulChargeRightPlayMode()
{
  runner.history.getPendingWorldState().setPlayMode(PM_Foul_Charge_Right);
  warninglog(10) << "I think I got a small size play mode, which will probably break me" << ende;
  last_msg_type = MT_Other;
}
void
OnlineParser::doBuildFoulPushLeftPlayMode()
{
  runner.history.getPendingWorldState().setPlayMode(PM_Foul_Push_Left);
  warninglog(10) << "I think I got a small size play mode, which will probably break me" << ende;
  last_msg_type = MT_Other;
}
void
OnlineParser::doBuildFoulPushRightPlayMode()
{
  runner.history.getPendingWorldState().setPlayMode(PM_Foul_Push_Right);
  warninglog(10) << "I think I got a small size play mode, which will probably break me" << ende;
  last_msg_type = MT_Other;
}
void
OnlineParser::doBuildFoulMultipleAttackerLeftPlayMode()
{
  runner.history.getPendingWorldState().setPlayMode(PM_Foul_MultipleAttacker_Left);
  warninglog(10) << "I think I got a small size play mode, which will probably break me" << ende;
  last_msg_type = MT_Other;
}
void
OnlineParser::doBuildFoulMultipleAttackerRightPlayMode()
{
  runner.history.getPendingWorldState().setPlayMode(PM_Foul_MultipleAttacker_Right);
  warninglog(10) << "I think I got a small size play mode, which will probably break me" << ende;
  last_msg_type = MT_Other;
}
void
OnlineParser::doBuildFoulBallOutLeftPlayMode()
{
  runner.history.getPendingWorldState().setPlayMode(PM_Foul_BallOut_Left);
  warninglog(10) << "I think I got a small size play mode, which will probably break me" << ende;
  last_msg_type = MT_Other;
}
void
OnlineParser::doBuildFoulBallOutRightPlayMode()
{
  runner.history.getPendingWorldState().setPlayMode(PM_Foul_BallOut_Right);
  warninglog(10) << "I think I got a small size play mode, which will probably break me" << ende;
  last_msg_type = MT_Other;
}
void
OnlineParser::doBuildBackPassLeftPlayMode()
{
  runner.history.getPendingWorldState().setPlayMode(PM_Back_Pass_Left);
  last_msg_type = MT_Other;
}
void
OnlineParser::doBuildBackPassRightPlayMode()
{
  runner.history.getPendingWorldState().setPlayMode(PM_Back_Pass_Right);
  last_msg_type = MT_Other;
}
void
OnlineParser::doBuildFreeKickFaultLeftPlayMode()
{
  runner.history.getPendingWorldState().setPlayMode(PM_Free_Kick_Fault_Left);
  last_msg_type = MT_Other;
}
void
OnlineParser::doBuildFreeKickFaultRightPlayMode()
{
  runner.history.getPendingWorldState().setPlayMode(PM_Free_Kick_Fault_Right);
  last_msg_type = MT_Other;
}
void
OnlineParser::doBuildCatchFaultLeftPlayMode()
{
  runner.history.getPendingWorldState().setPlayMode(PM_CatchFault_Left);
  last_msg_type = MT_Other;
}
void
OnlineParser::doBuildCatchFaultRightPlayMode()
{
  runner.history.getPendingWorldState().setPlayMode(PM_CatchFault_Right);
  last_msg_type = MT_Other;
}
void
OnlineParser::doBuildIndirectFreeKickLeftPlayMode()
{
  runner.history.getPendingWorldState().setPlayMode(PM_IndFreeKick_Left);
  last_msg_type = MT_Other;
}
void
OnlineParser::doBuildIndirectFreeKickRightPlayMode()
{
  runner.history.getPendingWorldState().setPlayMode(PM_IndFreeKick_Right);
  last_msg_type = MT_Other;
}
void
OnlineParser::doBuildPenaltySetupLeftPlaymode()
{
  runner.history.getPendingWorldState().setPlayMode(PM_PenaltySetup_Left);
  last_msg_type = MT_Other;
}
void
OnlineParser::doBuildPenaltySetupRightPlaymode()
{
  runner.history.getPendingWorldState().setPlayMode(PM_PenaltySetup_Right);
  last_msg_type = MT_Other;
}
void
OnlineParser::doBuildPenaltyReadyLeftPlaymode()
{
  runner.history.getPendingWorldState().setPlayMode(PM_PenaltyReady_Left);
  last_msg_type = MT_Other;
}
void
OnlineParser::doBuildPenaltyReadyRightPlaymode()
{
  runner.history.getPendingWorldState().setPlayMode(PM_PenaltyReady_Right);
  last_msg_type = MT_Other;
}
void
OnlineParser::doBuildPenaltyTakenLeftPlaymode()
{
  runner.history.getPendingWorldState().setPlayMode(PM_PenaltyTaken_Left);
  last_msg_type = MT_Other;
}
void
OnlineParser::doBuildPenaltyTakenRightPlaymode()
{
  runner.history.getPendingWorldState().setPlayMode(PM_PenaltyTaken_Right);
  last_msg_type = MT_Other;
}
void
OnlineParser::doBuildPenaltyMissLeftPlaymode()
{
  runner.history.getPendingWorldState().setPlayMode(PM_PenaltyMiss_Left);
  last_msg_type = MT_Other;
}
void
OnlineParser::doBuildPenaltyMissRightPlaymode()
{
  runner.history.getPendingWorldState().setPlayMode(PM_PenaltyMiss_Right);
  last_msg_type = MT_Other;
}
void
OnlineParser::doBuildPenaltyScoreLeftPlaymode()
{
  runner.history.getPendingWorldState().setPlayMode(PM_PenaltyScore_Left);
  last_msg_type = MT_Other;
}
void
OnlineParser::doBuildPenaltyScoreRightPlaymode()
{
  runner.history.getPendingWorldState().setPlayMode(PM_PenaltyScore_Right);
  last_msg_type = MT_Other;
}
void
OnlineParser::doBuildPenaltyOnFieldLeftPlaymode()
{
  errorlog << "OnlineParser: I don't support this penalty mode" << ende;
}
void
OnlineParser::doBuildPenaltyOnFieldRightPlaymode()
{
  errorlog << "OnlineParser: I don't support this penalty mode" << ende;
}
void
OnlineParser::doBuildPenaltyFoulLeftPlaymode()
{
  errorlog << "OnlineParser: I don't support this penalty mode" << ende;
}
void
OnlineParser::doBuildPenaltyFoulRightPlaymode()
{
  errorlog << "OnlineParser: I don't support this penalty mode" << ende;
}
void
OnlineParser::doBuildPenaltyWinnerLeftPlaymode()
{
  errorlog << "OnlineParser: I don't support this penalty mode" << ende;
}
void
OnlineParser::doBuildPenaltyWinnerRightPlaymode()
{
  errorlog << "OnlineParser: I don't support this penalty mode" << ende;
}
void
OnlineParser::doBuildPenaltyDrawPlaymode()
{
  errorlog << "OnlineParser: I don't support penalty mode" << ende;
}

/*******************/
// Other info
void
OnlineParser::doBuildGoalRefAudio( int time, int score )
{
  runner.history.getPendingWorldState().setScore(last_side, score);
  last_msg_type = MT_Other;  
}

void
OnlineParser::doBuildUnknownRefAudio( int time, const std::string& message )
{
  if (message == "half_time")
    return;
  if (message == "time_up")
    return;
  errorlog << "OnlineParser: got a unknown ref audio: " << time
           << " '" << message << "'" << ende;
  last_msg_type = MT_Error;
}

void
OnlineParser::doBuildPlayerAudio( int time, const std::string& msg )
{
  actionlog(190) << "Got player audio from "
                 << last_team_name << ' ' << last_unum
                 << " at " << time << ": " << msg
                 << ende;
}


/*******************/
// Errors and warnings
void
OnlineParser::doBuildCantReconnect()
{
  errorlog << "OnlineParser: received error: can't reconnect" << ende;
  last_msg_type = MT_Error;
}

void
OnlineParser::doBuildInitError()
{
  errorlog << "OnlineParser: received error: init error" << ende;
  last_msg_type = MT_Error;
}

void
OnlineParser::doBuildNoMoreTeamOrPlayerOrGoalieError()
{
  errorlog << "OnlineParser: received error: no more team or player" << ende;
  last_msg_type = MT_Error;
}

void
OnlineParser::doBuildNoSuchTeamOrAlreadyHaveCoachError()
{
  errorlog << "OnlineParser: received error: no such team or already have coach" << ende;
  last_msg_type = MT_Error;
}

void
OnlineParser::doBuildTooManyMovesError()
{
  errorlog << "OnlineParser: received error: too many moves" << ende;
  last_msg_type = MT_Error;
}

void
OnlineParser::doBuildUnknownCommandError()
{
  errorlog << "OnlineParser: received error: unknown command" << ende;
  last_msg_type = MT_Error;
}

void
OnlineParser::doBuildIllegalCommandError()
{
  errorlog << "OnlineParser: received error: illegal command" << ende;
  last_msg_type = MT_Error;
}

void
OnlineParser::doBuildSayMessageTooLongError()
{
  errorlog << "OnlineParser: received error: say message too long" << ende;
  last_msg_type = MT_Error;
}

void
OnlineParser::doBuildIllegalModeError()
{
  errorlog << "OnlineParser: received error: illegal mode" << ende;
  last_msg_type = MT_Error;
}

void
OnlineParser::doBuildIllegalObjectFormError()
{
  errorlog << "OnlineParser: received error: illegal objcet form" << ende;
  last_msg_type = MT_Error;
}

void
OnlineParser::doBuildSaidTooManyFreeformMessagesError()
{
  errorlog << "OnlineParser: received error: too many freeform" << ende;
  last_msg_type = MT_Error;
}

void
OnlineParser::doBuildCannotSayFreeformWhilePlayonError()
{
  errorlog << "OnlineParser: received error: no freeform while playon" << ende;
  last_msg_type = MT_Error;
}

void
OnlineParser::doBuildSaidTooManyMetaMessagesError()
{
  errorlog << "OnlineParser: received error: too many meta" << ende;
  last_msg_type = MT_Error;
}

void
OnlineParser::doBuildSaidTooManyAdviceMessagesError()
{
  errorlog << "OnlineParser: received error: too many advice" << ende;
  last_msg_type = MT_Error;
}

void
OnlineParser::doBuildSaidTooManyInfoMessagesError()
{
  errorlog << "OnlineParser: received error: too many info" << ende;
  last_msg_type = MT_Error;
}

void
OnlineParser::doBuildSaidTooManyDefineMessagesError()
{
  errorlog << "OnlineParser: received error: too many define" << ende;
  last_msg_type = MT_Error;
}

void
OnlineParser::doBuildCouldNotParseSayError()
{
  errorlog << "OnlineParser: received error: could not parse say" << ende;
  last_msg_type = MT_Error;
}

void
OnlineParser::doBuildSaidTooManyMessagesError()
{
  errorlog << "OnlineParser: received error: said too many messages" << ende;
  last_msg_type = MT_Error;
}

void
OnlineParser::doBuildUnknownError( const std::string& error )
{
  errorlog << "OnlineParser: received error: unknown '" << error << "'" << ende;
  last_msg_type = MT_Error;
}

void
OnlineParser::doBuildNoTeamFoundWarning()
{
  errorlog << "OnlineParser: received warning: no team found" << ende;
  last_msg_type = MT_Warning;
}

void
OnlineParser::doBuildNoSuchPlayerWarning()
{
  errorlog << "OnlineParser: received warning: no such player" << ende;
  last_msg_type = MT_Warning;
}

void
OnlineParser::doBuildCannotSubWhilePlayOnWarning()
{
  errorlog << "OnlineParser: received warning: no subs in playon" << ende;
  last_msg_type = MT_Warning;
}

void
OnlineParser::doBuildNoSubsLeftWarning()
{
  errorlog << "OnlineParser: received warning: no subs left" << ende;
  last_msg_type = MT_Warning;
}

void
OnlineParser::doBuildMaxOfThatPlayerTypeOnFieldWarning()
{
  errorlog << "OnlineParser: received warning: max of player type on field" << ende;
  last_msg_type = MT_Warning;
}

void
OnlineParser::doBuildCannotChangeGoalieWarning()
{
  errorlog << "OnlineParser: received warning: cannot change goalie" << ende;
  last_msg_type = MT_Warning;
}

void
OnlineParser::doBuildCompressionWarning()
{
  errorlog << "OnlineParser: received warning: compression" << ende;
  last_msg_type = MT_Warning;
}

void
OnlineParser::doBuildUnknownWarning( const std::string& warning )
{
  errorlog << "OnlineParser: received warning: unknown '" << warning << "'" << ende;
  last_msg_type = MT_Warning;
}

