/* A module that captures and draws various game features */

#include <sstream>
#include <iomanip>
#include "ModStatImage.h"
#include "ModuleRegistry.h"
#include "CoachParam.h"
#include "ModFeatures.h"
#include "Logger.h"

using namespace spades;


void
ModStatImage::initialize(ModuleRegistry* pReg)
{
  if (!CoachParam::instance()->getUseModStatImage())
    return;
  
  pReg->addModule(new ModStatImage());
}


ModStatImage::ModStatImage()
  : Module("StatImage"),
    last_time_draw_ws(-1),
    last_time_draw_ball_path(-1),
    ball_path_img_idx(0),
    fi_ball_path(1, CoachParam::instance()->getModStatBallPathScale()),
    rec_for_misc_ball(),
    pmFeatures(NULL),
    sds_ball_x(),
    sds_ball_y(),
    sds_ball_on_left(),
    sds_left_ball_owner(),
    sds_left_control_length(),
    sds_right_control_length(),
    sds_left_on_attack(),
    sds_right_on_attack(),
    sds_ball_in_rect(), 
    this_control_start(-1)
{
  if (CoachParam::instance()->getModStatBallPathUseRec())
    {
      rec_for_ball_path = 
        Rectangle(VecPosition(CoachParam::instance()->getModStatBallPathRecCorners()[0],
                              CoachParam::instance()->getModStatBallPathRecCorners()[1]),
                  VecPosition(CoachParam::instance()->getModStatBallPathRecCorners()[2],
                              CoachParam::instance()->getModStatBallPathRecCorners()[3]));
    }

  rec_for_misc_ball =
    Rectangle(VecPosition(CoachParam::instance()->getModStatMiscBallRec()[0],
                          CoachParam::instance()->getModStatMiscBallRec()[1]),
              VecPosition(CoachParam::instance()->getModStatMiscBallRec()[2],
                          CoachParam::instance()->getModStatMiscBallRec()[3]));

  std::istringstream is(CoachParam::instance()->getModStatBallPathOwnerSide());
  is >> ball_path_owner_side;
  if (is.fail())
    errorlog << "ModStatImage: Did not understand getModStatBallPathOwnerSide '"
             << CoachParam::instance()->getModStatBallPathOwnerSide() << "'" << ende;
  if (!CoachParam::instance()->getModStatMiscBallInRecXFN().empty())
    {
      os_for_misc_ball_rec_x.open(CoachParam::instance()->getModStatMiscBallInRecXFN().c_str());
      if (!os_for_misc_ball_rec_x)
        errorlog << "ModStatImage: Could not open misc ball in rec X pos file '"
                 << CoachParam::instance()->getModStatMiscBallInRecXFN()
                 << ende;
      os_for_misc_ball_rec_x << "# This file was generated by ModStatImage module" << std::endl
                             << "# It represents the X positions when the ball was in play while in a rectangle" << std::endl
                             << "# The rectangle is: " << rec_for_misc_ball << std::endl;
    }
}

ModStatImage::~ModStatImage() 
{
  saveBallPathImage();
  doFinalMiscStats();
}

bool
ModStatImage::dependencyCheck(ModuleRegistry* pReg, OnlineRunner& runner)
{
  if (CoachParam::instance()->getModStatDoMisc() ||
      CoachParam::instance()->getModStatDrawBallPath())
    {
      pmFeatures = (ModFeatures*)pReg->lookup("Features");
      if (pmFeatures == NULL)
        {
          errorlog << "ModStatImage needs ModFeatures" << ende;
          return false;
        }
    }
  return true;
}

bool
ModStatImage::dependencyCheck(ModuleRegistry* pReg, LogfileRunner& runner)
{
  if (CoachParam::instance()->getModStatDoMisc() ||
      CoachParam::instance()->getModStatDrawBallPath())
    {
      pmFeatures = (ModFeatures*)pReg->lookup("Features");
      if (pmFeatures == NULL)
        {
          errorlog << "ModStatImage needs ModFeatures" << ende;
          return false;
        }
    }
  return true;
}


void
ModStatImage::protStateUpdateNotify(Runner& runner, const WorldHistory& history)
{
  handleDrawWorldState(history);
  handleDrawBallPath(history);
  handleMiscStats(history);
}

void
ModStatImage::protSingleCall(SingleCallRunner& runner)
{
  warninglog(10) << "ModStatImage: I don't handle singleCall" << ende;
}


void
ModStatImage::handleDrawWorldState(const WorldHistory& history)
{
  if (!CoachParam::instance()->getModStatDrawWorldState())
    return;

  if (history.getNumAvail() == 0)
    {
      actionlog(100) << "handleDrawWorldState: no world states avail" << ende;
      return;
    }
  
  const WorldState& ws = history.getWorldState();
  
  if (last_time_draw_ws != -1 && last_time_draw_ws == ws.getTime())
    {
      actionlog(100) << "handleDrawWorldState: time is the same " << last_time_draw_ws << ende;
      return;
    }

  if (ws.getTime() % CoachParam::instance()->getModStatWorldStateInterval() != 0)
    {
      actionlog(100) << "handleDrawWorldState: interval delay" << ende;
      return;
    }
  
  last_time_draw_ws = ws.getTime();

  FieldImage fi(2, CoachParam::instance()->getModStatWorldStateScale());
  fi.addFieldLines();
  
  ws.draw(&fi);

  std::ostringstream ostr_fn;
  ostr_fn << CoachParam::instance()->getModStatWorldStateFStem()
	  << std::setw(4) << std::setfill('0') << ws.getTime();
  fi.writeTo(ostr_fn.str().c_str());
}

void
ModStatImage::handleDrawBallPath(const WorldHistory& history)
{
  if (!CoachParam::instance()->getModStatDrawBallPath())
    return;
  
  if (history.getNumAvail() == 0)
    {
      actionlog(100) << "handleDrawWorldState: no world states avail" << ende;
      return;
    }
  
  if (last_time_draw_ball_path == history.getWorldState().getTime())
    return;

  last_time_draw_ball_path = history.getWorldState().getTime();

  if (history.getWorldState().getTime() % CoachParam::instance()->getModStatBallPathRefreshInterval() == 1)
    {
      if (history.getWorldState().getTime() != 1)
	{
	  saveBallPathImage();
	}
      fi_ball_path.erase();
      fi_ball_path.addFieldLines();
      
      if (CoachParam::instance()->getModStatBallPathUseRec())
        {
          fi_ball_path.addRectangle(rec_for_ball_path,
                                    FieldImage::COLOR_RED,
                                    FieldImage::COLOR_CLEAR);
        }
      
      std::ostringstream ostr;
      ostr << "Ball path from " << history.getWorldState().getTime()
	   << " to " << history.getWorldState().getTime() + CoachParam::instance()->getModStatBallPathRefreshInterval();
      fi_ball_path.addLegendLine(ostr.str().c_str());
    }

  if (history.getWorldState().getTime() %
      CoachParam::instance()->getModStatBallPathSampleInterval() == 0)
    {
      if (ball_path_owner_side == TS_Both ||
          ball_path_owner_side == pmFeatures->getBallOwner().side)
        {
          FieldImage::Color c(FieldImage::COLOR_BLUE);
          if (CoachParam::instance()->getModStatBallPathUseRec() &&
              CoachParam::instance()->getModStatBallPathRecUseColors())
            {
              c = (rec_for_ball_path.isInside(history.getWorldState().getBall().getPos()))
                ? FieldImage::COLOR_BLUE : FieldImage::COLOR_GREY;
            }
          
          fi_ball_path.addPoint(history.getWorldState().getBall().getPos(), c);
        }
    }
  
}

void
ModStatImage::saveBallPathImage()
{
  // save previous image
  std::ostringstream ostr_fn;
  ostr_fn << CoachParam::instance()->getModStatBallPathFStem()
	  << ball_path_img_idx;
  fi_ball_path.writeTo(ostr_fn.str().c_str());
  ball_path_img_idx++;
}

	  
void
ModStatImage::handleMiscStats(const WorldHistory& history)
{
  if (!CoachParam::instance()->getModStatDoMisc())
    return;
      
  if (history.getWorldState().getPlayMode() == PM_PlayOn)
    {
      sds_ball_x.addPoint(history.getWorldState().getBall().getPos().getX());
      sds_ball_y.addPoint(history.getWorldState().getBall().getPos().getY());
      sds_ball_on_left.addPoint(history.getWorldState().getBall().getPos().getX() <= 0 ? 1.0 : 0.0);

      sds_ball_in_rect.addPoint(rec_for_misc_ball.isInside(history.getWorldState().getBall().getPos()));

      if (!CoachParam::instance()->getModStatMiscBallInRecXFN().empty())
        {
          if (rec_for_misc_ball.isInside(history.getWorldState().getBall().getPos()))
            os_for_misc_ball_rec_x << history.getWorldState().getBall().getPos().getX() << std::endl;
        }
  
    }

  if (history.getNumAvail() > 1 &&
      history.getWorldState().getTime() != history.getWorldState(1).getTime())
    {
      if (pmFeatures->getBallOwner().side == TS_Left)
        sds_left_ball_owner.addPoint(1.0);
      else if (pmFeatures->getBallOwner().side == TS_Right)
        sds_left_ball_owner.addPoint(0.0);
    }

  if (history.getNumAvail() > 1)
    {
      if (this_control_start != -1)
        {
          //see if this is a loss of control
          if (history.getWorldState().getPlayMode() != PM_PlayOn ||
              pmFeatures->getBallOwner().side != pmFeatures->getPrevBallOwner().side)
            {
              int control_time = history.getWorldState().getTime() - this_control_start;
              if (control_time >= CoachParam::instance()->getModStatMinControlLength())
                {
                  if (pmFeatures->getPrevBallOwner().side == TS_Left)
                    sds_left_control_length.addPoint(control_time);
                  else
                    sds_right_control_length.addPoint(control_time);
                }
              this_control_start = -1;
            }
        }
      // have to check again because we can end one control and start another in the same
      // cycle
      if (this_control_start == -1)
        {
          // see if this is a gain in control
          if (history.getWorldState().getPlayMode() == PM_PlayOn &&
              (pmFeatures->getBallOwner().side == TS_Left ||
               pmFeatures->getBallOwner().side == TS_Right))
            {
              this_control_start = history.getWorldState().getTime();
            }
        }
    }

  if (history.getWorldState().getPlayMode() == PM_PlayOn)
    {
      if (history.getWorldState().getBall().getPos().getX() > 0 &&
          pmFeatures->getBallOwner().side == TS_Left)
        sds_left_on_attack.addPoint(1.0);
      else
        sds_left_on_attack.addPoint(0.0);
        
      if (history.getWorldState().getBall().getPos().getX() < 0 &&
          pmFeatures->getBallOwner().side == TS_Right)
        sds_right_on_attack.addPoint(1.0);
      else
        sds_right_on_attack.addPoint(0.0);
    }
}

void
ModStatImage::doFinalMiscStats()
{
  if (!CoachParam::instance()->getModStatDoMisc())
    return;

  std::ofstream out(CoachParam::instance()->getModStatMiscOutFN().c_str());
  if (!out)
    {
      errorlog << "Couldn't out out file for misc stats '"
               << CoachParam::instance()->getModStatMiscOutFN() << "'"
               << ende;
      return;
    }

  out << "# This file was generated by the misc stat of the CMOwl coach" << std::endl;
  out << "# Format: <desc>: <num points> <mean> <variance> <stdev>" << std::endl;

  out << "BallX: ";
  sds_ball_x.writeCompactInfoToFile(out, true);
  out << std::endl;
  out << "BallY: ";
  sds_ball_y.writeCompactInfoToFile(out, true);
  out << std::endl;
  out << "BallOnLeft: ";
  sds_ball_on_left.writeCompactInfoToFile(out, true);
  out << std::endl;
  out << "LeftBallOwner: ";
  sds_left_ball_owner.writeCompactInfoToFile(out, true);
  out << std::endl;
  out << "LeftControlLength: ";
  sds_left_control_length.writeCompactInfoToFile(out, true);
  out << std::endl;
  out << "RightControlLength: ";
  sds_right_control_length.writeCompactInfoToFile(out, true);
  out << std::endl;
  out << "LeftOnAttack: ";
  sds_left_on_attack.writeCompactInfoToFile(out, true);
  out << std::endl;
  out << "RightOnAttack: ";
  sds_right_on_attack.writeCompactInfoToFile(out, true);
  out << std::endl;
  out << "BallInRec: ";
  sds_ball_in_rect.writeCompactInfoToFile(out, true);
  out << std::endl;
}

