/* -*- Mode: C++ -*- */

#ifndef _MODGLOBALADAPT_H_
#define _MODGLOBALADAPT_H_

#include <string>
#include <iostream>
#include <fstream>
#include "Module.h"
#include "data.h"

class ModFeatures;

class FixedAdviceAdaptationUnit;
class GAPredicateAdaptor;
class GAStrategySelector;

enum AdaptStyle {
  AS_Invalid,
  AS_Defensive,
  AS_Normal,
  AS_Offensive,
  AS_MAX
};

std::ostream& operator<<(std::ostream& os, const AdaptStyle& s);
std::istream& operator>>(std::istream& is, AdaptStyle& s);
AdaptStyle getStyleFor(const char* str);

/***************************************************************************/
class StrategySpec
{
public:
  StrategySpec() : style(AS_Invalid), style_idx(-1) {}
  StrategySpec(AdaptStyle style, int style_idx)
    : style(style), style_idx(style_idx) {}

  AdaptStyle getAdaptStyle() const { return style; }
  int getStyleIdx() const { return style_idx; }
  
  friend std::ostream& operator<<(std::ostream& os, const StrategySpec& s)
  { os << s.style << ' ' << s.style_idx; return os; }
  friend std::istream& operator>>(std::istream& is, StrategySpec& s)
  { is >> s.style >> s.style_idx; return is; }

  bool operator<(const StrategySpec& s) const;
  bool operator<=(const StrategySpec& s) const { return (*this < s) || (*this == s); }
  bool operator>(const StrategySpec& s) const { return s < *this; }
  bool operator>=(const StrategySpec& s) const { return (*this > s) || (*this == s); }
  bool operator==(const StrategySpec& s) const
  { return style == s.style && style_idx == s.style_idx; }
  bool operator!=(const StrategySpec& s) const
  { return !(*this == s); }
  
private:
  AdaptStyle style;
  int style_idx;
};

class StrategySpecStyleEq
{
public:
  StrategySpecStyleEq(AdaptStyle style) : style(style) {}
  bool operator()(const StrategySpec& spec) const
  { return spec.getAdaptStyle() == style; }
  template <class Key> bool operator()(const std::pair<Key, const StrategySpec>& pair) const
  { return pair.second.getAdaptStyle() == style; }
  template <class Value> bool operator()(const std::pair<const StrategySpec, Value>& pair) const
  { return pair.first.getAdaptStyle() == style; }
  AdaptStyle style;
};

/***************************************************************************/

typedef std::vector<StrategySpec> FlatStrategySpecSpace;
typedef std::map<RelativeTeamSide, int> PossCountStorage;

/***************************************************************************/

// This is an interface that anyone who wants to get notified on strategy changes needs
// to implement
class AdaptationUnit
{
public:
  AdaptationUnit() {}
  virtual ~AdaptationUnit() {}

  virtual const char* getAdaptationUnitName() = 0;

  virtual int getNumStrategiesForStyle(AdaptStyle style) = 0;

  virtual void changeStrategy(const StrategySpec& oldspec, const StrategySpec& newspec) = 0;
};

/***************************************************************************/

class LoggerAdaptationUnit
  : public AdaptationUnit
{
public:
  LoggerAdaptationUnit() : AdaptationUnit() {}
  ~LoggerAdaptationUnit() {}

  const char* getAdaptationUnitName() { return "Logger"; }

  int getNumStrategiesForStyle(AdaptStyle style) { return 1; }

  void changeStrategy(const StrategySpec& oldspec, const StrategySpec& newspec);
};

/***************************************************************************/

class ModGlobalAdapt
  : public Module
{
public:
  class CurrentStrategyInfo
  {
  public:
    CurrentStrategyInfo() { reset(); }
    ~CurrentStrategyInfo() {}

    void reset();
    void update(const WorldHistory& history, TeamSide my_side,  ModFeatures* pmFeatures);
    void initFrom(const StrategySpec& newspec, const WorldState& state, TeamSide my_side);
    
    const StrategySpec& getSpec() const { return spec; }
    int getStartCycle() const { return start_cycle; }
    int getStartMyGoals() const { return start_my_goals; }
    int getStartTheirGoals() const { return start_their_goals; }
    const SingleDataSummary& getSDSBallX() const { return sds_ball_x; }
    const PossCountStorage& getPossCounts() const { return poss_counts; }
    
    void setStrategy(const StrategySpec& s) { spec = s; }

    friend std::ostream& operator<<(std::ostream& os, const CurrentStrategyInfo& s);

  private:
    
    StrategySpec spec;
    int start_cycle;
    int start_my_goals;
    int start_their_goals;
    SingleDataSummary sds_ball_x;
    PossCountStorage poss_counts;
  };

  // Goals are floating point numbers so that we can easily scale data down
  class EpisodeInfo
  {
  public:
    EpisodeInfo();
    EpisodeInfo(const CurrentStrategyInfo& curr_strat, const WorldState& end_state, TeamSide my_side);
    ~EpisodeInfo() {}

    void setNull();
    
    std::string getDateStr() const { return date_str; }
    int getCycles() const { return cycles; }
    const StrategySpec& getSpec() const { return spec; }
    float getMyGoals() const { return my_goals; }
    float getTheirGoals() const { return their_goals; }
    const SingleDataSummary& getSDSBallX() const { return sds_ball_x; }
    const PossCountStorage& getPossCounts() const { return poss_counts; }
    
    void setDateFromCurrent();

    EpisodeInfo operator+(const EpisodeInfo& arg) const
     { EpisodeInfo ret(*this); return (ret += arg); }

    const EpisodeInfo& operator+=(const EpisodeInfo& arg);

    //Now a bunch of interesting predicates
    bool isWinning() const { return my_goals > their_goals; }
    bool isLosing() const { return their_goals > my_goals; }
    bool isTied() const { return their_goals == my_goals; }
    bool hasGoals() const { return my_goals > 0 || their_goals > 0; }

    float meanScoreDiffPerCycle() const { return (my_goals - their_goals) / cycles; }
    float meanMyGoalsPerCycle() const { return (my_goals) / cycles; }
    float meanTheirGoalsPerCycle() const { return (their_goals) / cycles; }

    float getPossPercentage(RelativeTeamSide rts) const;
    
    bool hasGoodDataAmt() const;
    
    friend std::ostream& operator<<(std::ostream& os, const EpisodeInfo& epinf);
    friend std::istream& operator>>(std::istream& is, EpisodeInfo& epinf);
    
  private:
    void addPossCount(const EpisodeInfo& arg, RelativeTeamSide s);
    void writePossCount(std::ostream& os, RelativeTeamSide s) const;
    
    std::string  date_str;
    int          cycles;
    StrategySpec spec;
    float        my_goals;
    float        their_goals;
    // only for PlayOn
    SingleDataSummary sds_ball_x;
    PossCountStorage poss_counts;
  };

  // THis class is intended to amalgamate date about past episodes into their styles
  class EpisodeSummaries
  {
  public:
    EpisodeSummaries() : results(), summary_ep() { summary_ep.setNull(); }
    ~EpisodeSummaries() {}

    void addEpisode(const EpisodeInfo& epinfo);

    bool haveDataFor(const StrategySpec& spec) const;
    const EpisodeInfo& getDataFor(const StrategySpec& spec) const;

    const EpisodeInfo& getGlobalSummaryEpisode() const { return summary_ep; }
    
    friend std::ostream& operator<<(std::ostream& os, const EpisodeSummaries& eps);

  private:
    typedef std::map<StrategySpec, EpisodeInfo> ResultsStorage;

    ResultsStorage results;
    EpisodeInfo summary_ep;
  };
  
  
public:
  static void initialize(ModuleRegistry* pReg);

public:
  ModGlobalAdapt();
  ~ModGlobalAdapt();

  void protStateUpdateNotify(Runner& runner, const WorldHistory& history);
  // these two variants just worry about setting my_side before called protStateUpdateNotify
  void protStateUpdateOnlineNotify(OnlineRunner& online_runner, const WorldHistory& history);
  void protStateUpdateLogfileNotify(LogfileRunner& logfile_runner, const WorldHistory& history);
  void protSingleCall(SingleCallRunner& runner);

  void shutdownOnlineNotify(OnlineRunner& online_runner, const WorldHistory& history);
  void shutdownLogfileNotify(LogfileRunner& logfile_runner, const WorldHistory& history);

  //Checks that all inter-module dependencies are satisfied; Called after all
  // modules initialized
  bool dependencyCheck(ModuleRegistry* pReg, OnlineRunner& runner);
  bool dependencyCheck(ModuleRegistry* pReg, SingleCallRunner& runner);
  bool dependencyCheck(ModuleRegistry* pReg, LogfileRunner& runner);

  // The AdaptationUnit need to be prepared to have changeStrategy called when it is added
  void addAdaptationUnit(AdaptationUnit* p);

protected:
  friend class GAPredicateAdaptor;
  
  bool shouldConsiderChangeThisCycle(const WorldHistory& history);
  void changeStrategy(const WorldState& state, const StrategySpec& spec);
  // state is the state at the end of the episode; reads current_strategy
  void storeCurrentEpisode(const WorldState& state);
  void readPastEpisodes();

  void doInitialStrategySelect(const WorldHistory& history);
  
  int findMaxIndexForStyle(AdaptStyle style) const;
  void setSpecSpace();
  int getFlatIdxOf(const StrategySpec& spec) const;

  //also looks in posat episodes becasue we still could have been using the same
  // strategy for a while
  int getTimeForCurrentStrategySpec(const WorldState& state) const;
  
  TeamSide my_side;
  
  LoggerAdaptationUnit au_logger;

  FixedAdviceAdaptationUnit* pau_fixed_advice;
  
  typedef std::vector<AdaptationUnit*> UnitStorage;
  UnitStorage vUnits;

  FlatStrategySpecSpace spec_space;
  
  CurrentStrategyInfo current_strategy;
  typedef std::vector<EpisodeInfo> PastEpisodeStorage;
  PastEpisodeStorage past_episodes;
  EpisodeSummaries episode_summaries;
  
  std::ofstream os_episodes;

  GAStrategySelector* pgass;

  ModFeatures* pmFeatures;

  bool done_initial_selection;
  

} ;


#endif
