/* -*- Mode: C++ -*- */

#include <sstream>
#include <errno.h>
#include "ModGlobalAdapt.h"
#include "ModuleRegistry.h"
#include "OnlineRunner.h"
#include "LogfileRunner.h"
#include "misc.h"
#include "ModFeatures.h"
#include "CoachParam.h"
#include "FixedAdviceAU.h"
#include "GAPredicateAdaptor.h"
#include "GAStrategySelector.h"
#include "Logger.h"

using namespace spades;

/**********************************************************************************/
const char* ADAPT_STYLE_STRINGS[] = {"Invalid", "Defensive", "Normal", "Offensive"};

std::ostream&
operator<<(std::ostream& os, const AdaptStyle& s)
{
  int s_idx = static_cast<int>(s);
  if (s < 0 || s >= AS_MAX)
    os << "INVALID_ADAPT_STYLE(" << s_idx << ")";
  else
    os << ADAPT_STYLE_STRINGS[s_idx];
  return os;
}

std::istream&
operator>>(std::istream& is, AdaptStyle& s)
{
  std::string str;
  s = AS_Invalid;
  is >> str;
  if (is.fail())
    return is;
  for (int i = 0; i < AS_MAX; i++)
    {
      if (strcasecmp(str.c_str(), ADAPT_STYLE_STRINGS[i]) == 0)
        {
          s = static_cast<AdaptStyle>(i);
          break;
        }
    }
  return is;
}

AdaptStyle
getStyleFor(const char* str)
{
  std::istringstream is(str);
  AdaptStyle s;
  is >> s;
  if (is.fail())
    return AS_Invalid;
  return s;
}

/**********************************************************************************/
bool
StrategySpec::operator<(const StrategySpec& s) const
{
  if (style != s.style)
    return static_cast<int>(style) < static_cast<int>(s.style);
  return style_idx < s.style_idx;
}

/**********************************************************************************/

void
ModGlobalAdapt::CurrentStrategyInfo::reset()
{
  spec = StrategySpec(AS_Invalid, 0);
  start_cycle = 0;
  start_my_goals = 0;
  start_their_goals = 0;
  sds_ball_x.reset();
  poss_counts.clear();
}

void
ModGlobalAdapt::CurrentStrategyInfo::update(const WorldHistory& history, TeamSide my_side,
                                            ModFeatures* pmFeatures)
{
  //handle ball position
  if (history.getNumAvail() >= 2 &&
      history.getWorldState().getPlayMode() == PM_PlayOn &&
      history.getWorldState().getTime() != history.getWorldState(1).getTime())
    {
      if (my_side != TS_Left && my_side != TS_Right)
        errorlog << "ModGlobalAdapt::CurrentStrategyInfo::update: what is my_side? "
                 << my_side << ende;
      float factor = (my_side == TS_Left) ? 1.0 : -1.0;
      sds_ball_x.addPoint(factor * history.getWorldState().getBall().getPos().getX());

    }

  // we count both play on and not play on modes
  if (history.getNumAvail() >= 2 &&
      history.getWorldState().getTime() != history.getWorldState(1).getTime())
    {
      //handle possession
      // This is a little bit  yucky because RelativeTeamSide does not have
      // a both value -- which it really should, but it's too much to change that now
      TeamSide abs_side = pmFeatures->getBallOwner().side;
      RelativeTeamSide side =
        (abs_side == TS_Both) ? RTS_None : absTeamSideToRelative(abs_side, my_side);
      // this transparently creates an entry for that side if it does not exist
      ++(poss_counts[side]);
    }
}

void
ModGlobalAdapt::CurrentStrategyInfo::initFrom(const StrategySpec& newspec,
                                              const WorldState& state,
                                              TeamSide my_side)
{
  spec = newspec;
  start_cycle = state.getTime();
  start_my_goals = state.getScore(relativeTeamSideToAbs(RTS_Mine, my_side));
  start_their_goals = state.getScore(relativeTeamSideToAbs(RTS_Theirs, my_side));
  sds_ball_x.reset();
  poss_counts.clear();
}

//friend
std::ostream&
operator<<(std::ostream& os, const ModGlobalAdapt::CurrentStrategyInfo& s)
{
  os << "("
     << s.spec << ", "
     << s.start_cycle << ", "
     << s.start_my_goals << ", "
     << s.start_their_goals << ", ";
  s.sds_ball_x.writeCompactInfoToFile(os, true);
  os << ", (";
  for (PossCountStorage::const_iterator iter = s.poss_counts.begin();
       iter != s.poss_counts.end();
       ++iter)
    os << iter->first << "=" << iter->second << " ";
  os << ") ";
  os << ")";
  return os;
}


/**********************************************************************************/

ModGlobalAdapt::EpisodeInfo::EpisodeInfo()
  : date_str(), cycles(-1), spec(), my_goals(-1), their_goals(-1)
{
  setDateFromCurrent();
}

ModGlobalAdapt::EpisodeInfo::EpisodeInfo(const CurrentStrategyInfo& curr_strat,
                                         const WorldState& end_state,
                                         TeamSide my_side)
  : cycles(end_state.getTime() - curr_strat.getStartCycle()),
    spec(curr_strat.getSpec()),
    my_goals(end_state.getScore(relativeTeamSideToAbs(RTS_Mine, my_side)) - curr_strat.getStartMyGoals()),
    their_goals(end_state.getScore(relativeTeamSideToAbs(RTS_Theirs, my_side)) - curr_strat.getStartTheirGoals()),
    sds_ball_x(curr_strat.getSDSBallX()),
    poss_counts(curr_strat.getPossCounts())
{
  setDateFromCurrent();
}

void
ModGlobalAdapt::EpisodeInfo::setNull()
{
  date_str = "NULL";
  cycles = 0;
  spec = StrategySpec(AS_Invalid, 0);
  my_goals = 0;
  their_goals = 0;
  sds_ball_x.reset();
  poss_counts.clear();
}


void
ModGlobalAdapt::EpisodeInfo::setDateFromCurrent()
{
  struct timeval tv;
  if (gettimeofday(&tv, NULL) != 0)
    errorlog << "EpisodeInfo::setDateFromCurrent: error in gettimeofday "
	     << errno << ": " << strerror(errno) << ende;
  date_str = ctime(&tv.tv_sec);
  std::string::size_type idx;
  while ( (idx = date_str.find(' ', 0)) != std::string::npos)
    {
      date_str.replace(idx, 1, "_", 1);
    }
  // remove the trailing \n
  date_str.erase(date_str.end() - 1);
}

/* If we add episodes, they should have the same spec
   The expection is that an episode with AS_Invalid can be added to anything
   and the left hand side determines the spec */
const ModGlobalAdapt::EpisodeInfo&
ModGlobalAdapt::EpisodeInfo::operator+=(const ModGlobalAdapt::EpisodeInfo& arg)
{
  if (spec != arg.spec &&
      spec.getAdaptStyle() != AS_Invalid &&
      arg.spec.getAdaptStyle() != AS_Invalid)
    errorlog << "ModGlobalAdapt::EpisodeInfo::operator+=: adding together different styles: "
             << *this << "; " << arg << ende;

  date_str += std::string("+") + arg.date_str;
  cycles += arg.cycles;
  my_goals += arg.my_goals;
  their_goals += arg.their_goals;
  sds_ball_x.combineWith(&arg.sds_ball_x);
  addPossCount(arg, RTS_None);
  addPossCount(arg, RTS_Mine);
  addPossCount(arg, RTS_Theirs);
  return *this;
}

float
ModGlobalAdapt::EpisodeInfo::getPossPercentage(RelativeTeamSide rts) const
{
  float sum = 0;
  sum += safeConstMapAccess(poss_counts, RTS_None);
  sum += safeConstMapAccess(poss_counts, RTS_Mine);
  sum += safeConstMapAccess(poss_counts, RTS_Theirs);
  
  return safeConstMapAccess(poss_counts, rts) / sum;
}


bool
ModGlobalAdapt::EpisodeInfo::hasGoodDataAmt() const
{
  return cycles >= CoachParam::instance()->getGlobalAdaptGoodDataAmtThreshold();
}


//friend
std::ostream&
operator<<(std::ostream& os, const ModGlobalAdapt::EpisodeInfo& epinf)
{
  os << epinf.date_str << ' '
     << epinf.cycles << ' '
     << epinf.spec << ' '
     << epinf.my_goals << ' '
     << epinf.their_goals << ' ';
  epinf.sds_ball_x.writeCompactInfoToFile(os, true);
  os << ' ';
  epinf.writePossCount(os, RTS_None);
  os << ' ';
  epinf.writePossCount(os, RTS_Mine);
  os << ' ';
  epinf.writePossCount(os, RTS_Theirs);
  os << ' ';
  
  os << std::endl;
  return os;
}

//friend
std::istream&
operator>>(std::istream& is, ModGlobalAdapt::EpisodeInfo& epinf)
{
  is >> epinf.date_str
     >> epinf.cycles
     >> epinf.spec
     >> epinf.my_goals
     >> epinf.their_goals;

    if (!epinf.sds_ball_x.readCompactInfoFromFile(is))
    is.setstate(is.rdstate() | std::ios::failbit);

  is >> epinf.poss_counts[RTS_None]
     >> epinf.poss_counts[RTS_Mine]
     >> epinf.poss_counts[RTS_Theirs];

  return is;
}

void
ModGlobalAdapt::EpisodeInfo::addPossCount(const EpisodeInfo& arg, RelativeTeamSide s)
{
  PossCountStorage::const_iterator iter = arg.poss_counts.find(s);
  if (iter == arg.poss_counts.end())
    return;
  poss_counts[s] += iter->second;
}

void
ModGlobalAdapt::EpisodeInfo::writePossCount(std::ostream& os, RelativeTeamSide s) const
{
  PossCountStorage::const_iterator iter = poss_counts.find(s);
  if (iter == poss_counts.end())
    os << "0";
  else
    os << iter->second;
}


/**********************************************************************************/
/**********************************************************************************/

void
ModGlobalAdapt::EpisodeSummaries::addEpisode(const EpisodeInfo& epinfo)
{
  ResultsStorage::iterator iter = results.find(epinfo.getSpec());
  if (iter == results.end())
    {
      results[epinfo.getSpec()] = epinfo;
    }
  else
    {
      iter->second += epinfo;
    }
  summary_ep += epinfo;
}

bool
ModGlobalAdapt::EpisodeSummaries::haveDataFor(const StrategySpec& spec) const
{
  return results.count(spec) != 0;
}

const ModGlobalAdapt::EpisodeInfo&
ModGlobalAdapt::EpisodeSummaries::getDataFor(const StrategySpec& spec) const
{
  static EpisodeInfo null_epinfo;

  ResultsStorage::const_iterator iter = results.find(spec);
  if (iter == results.end())
    {
      errorlog << "EpisodeSummaries::getDataFor: tried to get data for non-existent "
               << spec << ende;
      return null_epinfo;
    }
  return iter->second;
}

    

//friend
std::ostream&
operator<<(std::ostream& os, const ModGlobalAdapt::EpisodeSummaries& eps)
{
  for (ModGlobalAdapt::EpisodeSummaries::ResultsStorage::const_iterator iter =
         eps.results.begin();
       iter != eps.results.end();
       ++iter)
    {
      os << iter->first << "\t" << iter->second << std::endl;
    }
  return os;
}


/**********************************************************************************/
/**********************************************************************************/

void
ModGlobalAdapt::initialize(ModuleRegistry* pReg)
{
  if (!CoachParam::instance()->getUseModGlobalAdapt())
    return;
  
  pReg->addModule(new ModGlobalAdapt());
}

/**********************************************************************************/

ModGlobalAdapt::ModGlobalAdapt()
  : Module("GlobalAdapt"),
    my_side(TS_None), au_logger(), pau_fixed_advice(NULL),
    vUnits(), current_strategy(),
    past_episodes(), os_episodes(),
    pgass(NULL),
    pmFeatures(NULL),
    done_initial_selection(false)
{
  current_strategy.setStrategy(StrategySpec(getStyleFor(CoachParam::instance()->getGlobalAdaptInitialStrategyStyle().c_str()),
                                            CoachParam::instance()->getGlobalAdaptInitialStrategyIndex()));

  readPastEpisodes();

  if (CoachParam::instance()->getGlobalAdaptWriteResults())
    {
      os_episodes.open(CoachParam::instance()->getGlobalAdaptOutResultsFN().c_str(),
                       std::ios::out |
                       (CoachParam::instance()->getGlobalAdaptAppend() ? std::ios::app : static_cast<std::ios_base::openmode>(0)));
      if (!os_episodes)
        errorlog << "ModGlobalAdapt: could not open out file '"
                 << CoachParam::instance()->getGlobalAdaptOutResultsFN()
                 << "'" << ende;
    }

  //pgass = new GASS_Fixed(StrategySpec(AS_Defensive, 1));
  //pgass = new GASS_FixedAdvance();
  pgass = new GASS_SimpleHandCoded();

  addAdaptationUnit(&au_logger);

}

ModGlobalAdapt::~ModGlobalAdapt()
{
  //Note that we do NOT delete the adaptation units in vUnits
  delete pau_fixed_advice;
  delete pgass;
}


void
ModGlobalAdapt::protStateUpdateNotify(Runner& runner, const WorldHistory& history)
{
  if (history.getNumAvail() == 0)
    return;

  if (!done_initial_selection)
    {
      doInitialStrategySelect(history);
      done_initial_selection = true;
    }
  
  current_strategy.update(history, my_side, pmFeatures);
    
  if (shouldConsiderChangeThisCycle(history))
    {
      storeCurrentEpisode(history.getWorldState());

      //std::cout << episode_summaries;

      GAPredicateAdaptor gapred(history, this);

      //gapred.printDebugTo(std::cout);
      
      StrategySpec newspec = pgass->selectNewStrategy(gapred);

      std::cout << history.getWorldState().getTime()
                << ": ModGlobalAdapt: " << *pgass << " selected " << newspec << std::endl;
      actionlog(30) << "ModGlobalAdapt: " << *pgass << " selected " << newspec << ende;
      
      changeStrategy(history.getWorldState(), newspec);
    }
}

void
ModGlobalAdapt::protStateUpdateOnlineNotify(OnlineRunner& online_runner, const WorldHistory& history)
{
  if (my_side == TS_None)
    my_side = online_runner.getMySide();
  
  protStateUpdateNotify(online_runner, history);
}

void
ModGlobalAdapt::protStateUpdateLogfileNotify(LogfileRunner& logfile_runner, const WorldHistory& history)
{
  if (my_side == TS_None)
    {
      my_side = logfile_runner.getSideToAnalyze();
      if (my_side == TS_Both)
        {
          errorlog << "ModGlobalAdapt: I can't analyze both sides in a logfile!" << ende;
          my_side = TS_Left;
        }
    }

  protStateUpdateNotify(logfile_runner, history);
}


void
ModGlobalAdapt::protSingleCall(SingleCallRunner& runner)
{
  warninglog(10) << "ModGlobalAdapt: I do not do anything with singleCall" << ende;
}

void
ModGlobalAdapt::shutdownOnlineNotify(OnlineRunner& online_runner,
                                     const WorldHistory& history)
{
  if (history.getNumAvail() == 0)
    return;
  storeCurrentEpisode(history.getWorldState());
}

void
ModGlobalAdapt::shutdownLogfileNotify(LogfileRunner& logfile_runner,
                                      const WorldHistory& history)
{
  if (history.getNumAvail() == 0)
    return;
  storeCurrentEpisode(history.getWorldState());
}


/*****************************************************************************/

bool
ModGlobalAdapt::dependencyCheck(ModuleRegistry* pReg, OnlineRunner& runner)
{
  pmFeatures = (ModFeatures*)pReg->lookup("Features");
  if (pmFeatures == NULL)
    {
      errorlog << "ModGlobalAdapt needs ModFeatures" << ende;
      return false;
    }
  if (CoachParam::instance()->getGlobalAdaptUseFixedAdvice())
    {
      std::cout << "Loading FixedAdviceAdaptationUnit..." << std::endl;
      pau_fixed_advice =
        new FixedAdviceAdaptationUnit(pReg,
                                      CoachParam::instance()->getGlobalAdaptFixedAdviceMapFN().c_str());

      addAdaptationUnit(pau_fixed_advice);
      std::cout << "Done" << std::endl;
    }
  return true;
}

bool
ModGlobalAdapt::dependencyCheck(ModuleRegistry* pReg, SingleCallRunner& runner)
{
  return true;
}

bool
ModGlobalAdapt::dependencyCheck(ModuleRegistry* pReg, LogfileRunner& runner)
{
  pmFeatures = (ModFeatures*)pReg->lookup("Features");
  if (pmFeatures == NULL)
    {
      errorlog << "ModGlobalAdapt needs ModFeatures" << ende;
      return false;
    }
  return true;
}

/**********************************************************************************/

void
ModGlobalAdapt::doInitialStrategySelect(const WorldHistory& history)
{
  GAPredicateAdaptor gapred(history, this);
  
  //gapred.printDebugTo(std::cout);

  StrategySpec newspec = pgass->selectNewStrategy(gapred);

  std::cout << "ModGlobalAdapt: initial selection by " << *pgass << ": " << newspec
            << std::endl;
  actionlog(30) << "ModGlobalAdapt: initial selection by " << *pgass << ": " << newspec
                << ende;

  changeStrategy(history.getWorldState(), newspec);
}


// The AdaptationUnit need to be prepared to have changeStrategy called when it is added
void
ModGlobalAdapt::addAdaptationUnit(AdaptationUnit* p)
{
  actionlog(50) << "GlobalAdapt: adding unit " << p->getAdaptationUnitName() << ende;
  
  vUnits.push_back(p);
  setSpecSpace();
  
  p->changeStrategy(StrategySpec(AS_Invalid, -1), current_strategy.getSpec());
}

bool
ModGlobalAdapt::shouldConsiderChangeThisCycle(const WorldHistory& history)
{
  // standard adjust interval
  if (CoachParam::instance()->getGlobalAdaptAdjustInterval() > 0 &&
      history.getNumAvail() >= 2 &&
      history.getWorldState().getTime() != history.getWorldState(1).getTime() &&
      ((history.getWorldState().getTime() - current_strategy.getStartCycle()) %
       CoachParam::instance()->getGlobalAdaptAdjustInterval() == 0))
    return true;
  // adjust after goals
  if (CoachParam::instance()->getGlobalAdaptAdjustAfterGoals() &&
      history.getNumAvail() >= 2 &&
      (history.getWorldState(1).getLeftScore() != history.getWorldState().getLeftScore() ||
       history.getWorldState(1).getRightScore() != history.getWorldState().getRightScore()))
    return true;
  // adjust at play stoppage
  if (CoachParam::instance()->getGlobalAdaptAdjustAtStoppage() &&
      history.getNumAvail() >= 2 &&
      history.getWorldState(1).getPlayMode() == PM_PlayOn &&
       history.getWorldState().getPlayMode() != PM_PlayOn)
    return true;
  
  return false;
}



void
ModGlobalAdapt::changeStrategy(const WorldState& state, const StrategySpec& newspec)
{
  actionlog(50) << "GlobalAdapt: Changing from "
                << current_strategy.getSpec()
                << " to "
                << newspec
                << ende;
  
  CurrentStrategyInfo next_strategy;

  next_strategy.initFrom(newspec, state, my_side);
  
  for (UnitStorage::iterator iter = vUnits.begin();
       iter != vUnits.end();
       ++iter)
    {
      (*iter)->changeStrategy(current_strategy.getSpec(), newspec);
    }

  current_strategy = next_strategy;

  actionlog(60) << "GlobalAdapt: Strategy start point is " << current_strategy << ende;
  
}

  
// state is the state at the end of the episode; reads current_strategy
void
ModGlobalAdapt::storeCurrentEpisode(const WorldState& state)
{
  if (my_side == TS_None)
    return;
  
  EpisodeInfo epinfo(current_strategy, state, my_side);

  if (epinfo.getCycles() == 0)
    return;
  
  past_episodes.push_back(epinfo);
  episode_summaries.addEpisode(epinfo);
  
  if (CoachParam::instance()->getGlobalAdaptWriteResults())
    os_episodes << epinfo;
}

void
ModGlobalAdapt::readPastEpisodes()
{
  if (!CoachParam::instance()->getGlobalAdaptReadResults())
    return;

  std::ifstream is(CoachParam::instance()->getGlobalAdaptInResultsFN().c_str());
  if (!is)
    {
      errorlog << "ModGlobalAdapt::readPastEpisodes: failed to open '"
               << CoachParam::instance()->getGlobalAdaptInResultsFN() << "'"
               << ende;
      return;
    }
  
  for (;;)
    {
      if (!skip_to_non_comment(is))
        break;
      EpisodeInfo epinfo;
      is >> epinfo;
      if (is.fail())
        {
          errorlog << "ModGlobalAdapt::readPastEpisodes: reading failed" << ende;
          break;
        }
      past_episodes.push_back(epinfo);
      episode_summaries.addEpisode(epinfo);
    }

  std::cout << "ModGlobalAdapt: I read " << past_episodes.size()
            << " past episodes" << std::endl;
}

int
ModGlobalAdapt::findMaxIndexForStyle(AdaptStyle style) const
{
  int max = 0;
  for (UnitStorage::const_iterator iter = vUnits.begin();
       iter != vUnits.end();
       ++iter)
    {
      int cnt = (*iter)->getNumStrategiesForStyle(style);
      if (cnt > max)
        max = cnt;
    }
  return max;
}

void
ModGlobalAdapt::setSpecSpace()
{
  spec_space.clear();
  for (int istyle = 0; istyle < static_cast<int>(AS_MAX); istyle++)
    {
      AdaptStyle style = static_cast<AdaptStyle>(istyle);
      if (style == AS_Invalid)
        continue;
      int count = findMaxIndexForStyle(style);
      for (int i=0; i<count; i++)
        spec_space.push_back(StrategySpec(style, i));
    }
}

int
ModGlobalAdapt::getFlatIdxOf(const StrategySpec& spec) const
{
  FlatStrategySpecSpace::const_iterator iter =
    std::find(spec_space.begin(), spec_space.end(), spec);
  if (iter == spec_space.end())
    return -1;
  return (iter - spec_space.begin());
}

int
ModGlobalAdapt::getTimeForCurrentStrategySpec(const WorldState& state) const
{
  int time = state.getTime() - current_strategy.getStartCycle();
  for (PastEpisodeStorage::const_reverse_iterator iter = past_episodes.rbegin();
       iter != past_episodes.rend();
       ++iter)
    {
      if (iter->getSpec() != current_strategy.getSpec())
        break;
      // Don't go past the beginning of a game
      if (time + iter->getCycles() > state.getTime())
        break;
      time += iter->getCycles();
    }
  return time;
}




/**********************************************************************************/
/**********************************************************************************/

void
LoggerAdaptationUnit::changeStrategy(const StrategySpec& oldspec, const StrategySpec& newspec)
{
  actionlog(50) << "LoggerAdaptationUnit: change from "
                << oldspec
                << " to "
                << newspec
                << ende;
}

