/* -*- Mode: C++ -*- */

#include <iomanip>
#include <iterator>
#include <algorithm>
#include "ModFormation.h"
#include "ModCMRep.h"
#include "soccer_utils.h"
#include "Logger.h"
#include "CoachParam.h"
#include "ModuleRegistry.h"
#include "OnlineRunner.h"
#include "LogfileRunner.h"
#include "version.h"

using namespace spades;

/**********************************************************************************/  

class FormationSender
{
public:
  FormationSender(CoachMessageQueue* q, const std::vector<FormationFormatter*>& vformatter)
    : q(q), vformatter(vformatter) {}
  void operator() (const Formation& f) { f.createRule(q, vformatter); }
  CoachMessageQueue* q;
  const std::vector<FormationFormatter*>& vformatter;
};

class FormationPermute
{
public:
  FormationPermute(int goalie_num) : goalie_num(goalie_num) {}
  void operator() (Formation& f) { f.permuteForGoalie(goalie_num); }
  int goalie_num;
};

/**********************************************************************************/  

/* static method */
void
ModFormation::initialize(ModuleRegistry* pReg)
{
  if (!CoachParam::instance()->getUseModFormation())
    return;

  pReg->addModule(new ModFormation());
}

/**********************************************************************************/  
/*   things left to do:
 D* picking a formation
 * creating mark advice
 D* gather data mode
 D* setting side to learn
 D* handling of random formations
 D* reading in formations
 D* access function to get current form
        parameters unused
  * formation_send_marks
*/
ModFormation::ModFormation()
  : Module("Formation"),
    pCMRep(NULL),
    learner(),
    list_form(),
    iter_curr_form(list_form.end()),
    mqueue("Formation"),
    out_gather_data(),
    form_opponent(),
    form_opponent_valid(false),
    sent_formations(false),
    need_to_assign_marks(false)
{

  if (CoachParam::instance()->getFormationGatherData())
    {
      out_gather_data.open(CoachParam::instance()->getFormationLearnDataFN().c_str(),
			   std::ios::app);
      if (!out_gather_data)
	errorlog << "Formation: Could not open data gathering file: "
		 << CoachParam::instance()->getFormationLearnDataFN()
		 << ende;
      learner.writeRecordHeader(out_gather_data);
    }

  if (!CoachParam::instance()->getFormationOppFormFN().empty())
    {
      std::ifstream infile(CoachParam::instance()->getFormationOppFormFN().c_str());
      if (!infile)
	{
	  errorlog << "Could not open opponent formation file '"
		   << CoachParam::instance()->getFormationOppFormFN()
		   << "'" << ende;
	}
      else
	{
	  infile >> form_opponent;
	  if (infile.fail())
	    {
	      errorlog << "Error reading opponent formation " << ende;
	    }
	  else
	    {
	      form_opponent_valid = true;
	    }
	}
    }

  readFormations();

  if (CoachParam::instance()->getFormationUseRandom())
    {
      Formation fRandom("MyRandForm");
      fRandom.generateRandom(30, 15, 40, 15);
      list_form.push_front(fRandom);
      std::ofstream out("Formations/MyRandForm.form");
      out << fRandom << std::endl;
      out.close();
    }

  initializeFormationFormatters();
}

ModFormation::~ModFormation()
{
  if (CoachParam::instance()->getFormationLearn())
    {
      std::cout << "Learning formation data" << std::endl;
      Formation f;
      if (!learner.learnFormation(f))
	errorlog << "Error in learning formation" << ende;

      f.setNameFromFilename(CoachParam::instance()->getFormationLearnOutputFN().c_str());
      std::ofstream out(CoachParam::instance()->getFormationLearnOutputFN().c_str());
      if (!out)
	errorlog << "Could not open learned formation output file '"
		 << CoachParam::instance()->getFormationLearnOutputFN()
		 << ende;
      out << f;
      out.close();

      list_form.push_back(f);
      if (CoachParam::instance()->getFormationDraw())
	{
	  drawFormations(CoachParam::instance()->getFormationDrawOutputDir());
	}
    }

  std::for_each(vformatter.begin(), vformatter.end(), deleteptr<FormationFormatter>());

  if (pCMRep)
    pCMRep->removeCMQ(&mqueue);
}


void
ModFormation::protStateUpdateNotify(Runner& runner, const WorldHistory& history)
{
  //we'll use this as a callee of the other stateUpdate functions

  // This does data gather which we can later combine
  if (CoachParam::instance()->getFormationGatherData())
    {
      if (history.getWorldState().getTime() % CoachParam::instance()->getFormationLearnSampleStep() == 0 &&
	  (history.getNumAvail() == 1 ||
	   history.getWorldState().getTime() != history.getWorldState(1).getTime()))
	{
	  actionlog(100) << "ModFormation: gathering date for later learning" << ende;
	  if (!learner.recordCurrDataToFile(out_gather_data, history.getWorldState()))
	    errorlog << "Could not record current formation data to file" << ende;
	}
    }	

  // Now see if we are trying to learn a formation with the date we get here,
  // If so, we better store it to the learner
  if (CoachParam::instance()->getFormationLearn())
    {
      actionlog(100) << "ModFormation: adding current date to learner" << ende;
      if (!learner.addCurrentToSummary(history.getWorldState()))
	errorlog << "Could not add current data to formation learner" << ende;
    }
}

void
ModFormation::protStateUpdateOnlineNotify(OnlineRunner& online_runner, const WorldHistory& history)
{
  if (CoachParam::instance()->getFormationGatherData() ||
      CoachParam::instance()->getFormationLearn())
    {
      // check on the side to learn
      if (learner.getSideToLearn() == TS_None)
	{
	  TeamSide ts =
	    online_runner.determineSide(CoachParam::instance()->getFormationSideToLearn().c_str(),
					CoachParam::instance()->getFormationTeamToLearn().c_str());
	  if (ts == TS_None)
	    {
	      errorlog << "ModFormation: I could not determine which side to learn" << ende;
	      online_runner.initiateShutdown();
	      return;
	    }
	  if (ts != TS_Both)
	    learner.setSideToLearn(ts);
	  else
	    return;  // <-----------------
	}
    }

  if (CoachParam::instance()->getFormationSendForm())
    {
      if (!sent_formations)
	{
	  Unum my_goalie = online_runner.getWorldHistory().getWorldState().getGoalieNum(online_runner.getMySide());
	  if (my_goalie < 0)
	    {
	      if (online_runner.getWorldHistory().getWorldState().getNumPlayers(online_runner.getMySide()) >=
		  ServerParam::instance()->getSPTeamSize())
		{
		  warninglog(10) << "I don't seem to have a goalie on my side" << ende;
		  sendAllFormations();
		}
	      else
		{
		  actionlog(60) << "ModFormation: Waiting for the goalie on my side" << ende;
		}
	    }
	  else
	    {
	      std::for_each(list_form.begin(), list_form.end(), FormationPermute(my_goalie));
	      sendAllFormations();
	    }

          pGlobalAdapt->addAdaptationUnit(this);
	}
    }
  else
    {
      sent_formations = true;
    }

  if (sent_formations &&
      iter_curr_form == list_form.end() &&
      !list_form.empty())
    {
      pickFormation(CoachParam::instance()->getFormationSendForm());
    }
  
  protStateUpdateNotify(online_runner, history);
}

void
ModFormation::protStateUpdateLogfileNotify(LogfileRunner& logfile_runner,
					   const WorldHistory& history)
{
  if (CoachParam::instance()->getFormationGatherData() ||
      CoachParam::instance()->getFormationLearn())
    {
      // check on the side to learn
      if (learner.getSideToLearn() == TS_None)
	{
	  TeamSide ts =
	    logfile_runner.determineSide(CoachParam::instance()->getFormationSideToLearn().c_str(),
					 CoachParam::instance()->getFormationTeamToLearn().c_str());
	  if (ts == TS_None)
	    {
	      errorlog << "ModFormation: I could not determine which side to learn" << ende;
	      logfile_runner.initiateShutdown();
	      return;
	    }
	  if (ts != TS_Both)
	    learner.setSideToLearn(ts);
	  else
	    return;  // <-----------------
	}
    }

  protStateUpdateNotify(logfile_runner, history);
}


void
ModFormation::protSingleCall(SingleCallRunner& runner)
{
  if (CoachParam::instance()->getFormationCombineData())
    {
      std::cout << "Combining learned formation data" << std::endl;
      std::ifstream infile(CoachParam::instance()->getFormationLearnDataFN().c_str());
      if (!infile)
	errorlog << "Could not open combine data file: "
		 << CoachParam::instance()->getFormationLearnDataFN().c_str()
		 << ende;
      learner.loadDataFrom(infile);
    }
  if (CoachParam::instance()->getFormationDraw())
    {
      drawFormations(CoachParam::instance()->getFormationDrawOutputDir());
    }
}


//Checks that all inter-module dependencies are satisfied; Called after all
// modules initialized
bool
ModFormation::dependencyCheck(ModuleRegistry* pReg, OnlineRunner& runner)
{
  //we'll wait to add ourselves until we have actually sent the formations out
  pGlobalAdapt = (ModGlobalAdapt*)pReg->lookup("GlobalAdapt");
  if (pGlobalAdapt == NULL)
    {
      errorlog << "ModFormation needs ModGlobalAdapt" << ende;
      return false;
    }

  pCMRep = (ModCMRep*)pReg->lookup("CMRep");
  if (pCMRep == NULL)
    {
      errorlog << "ModFormation needs ModCMRep" << ende;
      return false;
    }

  pCMRep->addCMQ(&mqueue);
  return true;
}
  
bool
ModFormation::dependencyCheck(ModuleRegistry* pReg, SingleCallRunner& runner)
{
  return true;
}
  
bool
ModFormation::dependencyCheck(ModuleRegistry* pReg, LogfileRunner& runner)
{
  return true;
}

const Formation*
ModFormation::getCurrentFormation() const
{
  if (iter_curr_form == list_form.end())
    {
      errorlog << "I have no current formation" << ende;
      return NULL;
    }
  return &(*iter_curr_form);
}

void
ModFormation::initializeFormationFormatters()
{
  vformatter.clear();

  for (std::vector<std::string>::const_iterator iter = CoachParam::instance()->getFormationFormatters().begin();
       iter != CoachParam::instance()->getFormationFormatters().end();
       ++iter)
    {
      actionlog(200) << "ModFormation: handling formatter desc '"
                     << *iter << "' for player "
                     << (iter - CoachParam::instance()->getFormationFormatters().begin() + 1)
                     << ende;
      FormationFormatter* p = FormationFormatter::createFrom((*iter).c_str());
      if (p == NULL)
          errorlog << "ModFormation::initializeFormationFormatters: Failed to create formatter for " << *iter << ende;
      vformatter.push_back(p);
    }
}

void
ModFormation::readFormations()
{
  FormationFileListReader reader(list_form);
  reader.readFile(CoachParam::instance()->getFormationListFN().c_str(), g_Version);
  actionlog(50) << "ModFormation: read " << reader.getCount() << " formations" << ende;
  //TEST_CODE
  //printFormations(std::cout);
}

void
ModFormation::printFormations(std::ostream& o)
{
  o << "# Formations from ModFormation: " << std::endl;
  std::copy(list_form.begin(), list_form.end(), std::ostream_iterator<Formation>(o, "\n"));
}

void
ModFormation::sendAllFormations()
{
  actionlog(50) << "ModFormation: Sending all formation rules" << ende;
  std::for_each(list_form.begin(), list_form.end(), FormationSender(&mqueue, vformatter));
  sent_formations = true;
}

void
ModFormation::pickFormation(bool send)
{
  actionlog(50) << "ModFormation: Picking a new formation" << ende;
  //all we are doing is picking the first one -- how silly
  FormationStorage::iterator iter_new_form = list_form.begin();
  changeToFormation(iter_new_form, send);
}

void
ModFormation::changeToFormation(FormationStorage::iterator iter_new_form, bool send)
{
  if (iter_new_form != iter_curr_form)
    {
      actionlog(50) << "ModFormation: Changing from "
		    << ((iter_curr_form == list_form.end()) ? "NONE" : iter_curr_form->getName())
		    << " to "
		    << iter_new_form->getName() << ende;
      if (iter_curr_form != list_form.end() &&
          iter_new_form != list_form.end() &&
          iter_curr_form->getRuleName() == iter_new_form->getRuleName())
        {
          actionlog(60) << "ModFormation: no need to send rule messages, it's the same form"
                        << ende;
        }
      else
        {
          if (send && iter_curr_form != list_form.end())
            iter_curr_form->setEnableState(false, &mqueue);
          if (send && iter_new_form != list_form.end())
            iter_new_form->setEnableState(true, &mqueue);
          iter_curr_form = iter_new_form;
          need_to_assign_marks = true;
        }
    }
}


void
ModFormation::drawFormations(const std::string& out_dir)
{
  FieldImage fi(6);

  std::cout << "Drawing formations..." << std::endl;
  for (FormationStorage::iterator iter = list_form.begin();
       iter != list_form.end();
       ++iter)
    {
      fi.erase();
      fi.addFieldLines();
      iter->draw(&fi);
      std::string fn = out_dir + "/formation-" + iter->getName();
      fi.writeTo(fn.c_str());
    }
}

/**********************************************************************/  

int
ModFormation::getNumStrategiesForStyle(AdaptStyle style)
{
  if (style == AS_Defensive || style == AS_Normal || style == AS_Offensive)
    return CoachParam::instance()->getFormationNumPerAdaptStyle()[static_cast<int>(style)-1];
  return -1;
}

void
ModFormation::changeStrategy(const StrategySpec& oldspec, const StrategySpec& newspec)
{
  int base_idx = 0;
  switch (newspec.getAdaptStyle())
    {
    case AS_Defensive:
      base_idx = 0;
      break;
    case AS_Normal:
      base_idx = CoachParam::instance()->getFormationNumPerAdaptStyle()[0];
      break;
    case AS_Offensive:
      // def and norm
      base_idx =  CoachParam::instance()->getFormationNumPerAdaptStyle()[0] + 
       CoachParam::instance()->getFormationNumPerAdaptStyle()[1];
      break;
    default:
      errorlog << "ModFormation::changeStrategy: I did not understand style "
               << newspec.getAdaptStyle() << ende;
    }

  int final_idx = base_idx +
    (newspec.getStyleIdx() % CoachParam::instance()->getFormationNumPerAdaptStyle()[static_cast<int>(newspec.getAdaptStyle())-1]);
  FormationStorage::iterator iter_new_form = list_form.begin();
  std::advance(iter_new_form, final_idx);

  if (iter_new_form == list_form.end())
    errorlog << "ModFormation::changeStrategy: I don't have a form for "
             << newspec
             << ", num forms is " << list_form.size()
             << ende;
  
  changeToFormation(iter_new_form, CoachParam::instance()->getFormationSendForm());
}


/**********************************************************************/  
// The fileid is to be printed in front of error messages and stuff
// The path is the full path of the file
// the version is the version of this file
bool
FormationFileListReader::processLine(std::istrstream& line,
				     const char* fileid,
				     const char* path,
				     float version)
{
  std::string form_fn;
  
  skip_white_space(line);
  line >> form_fn;

  form_fn = resolvePathGivenStartingFile( path, form_fn.c_str() );

  std::ifstream ffile(form_fn.c_str());

  if (!ffile)
    {
      errorlog << "Could not open formation file '" << form_fn.c_str() << "'" << ende;
      return false;
    }

  for (;;)
    {
      if (!ffile)
	break;
      Formation f;
      ffile >> f;
      if (ffile.fail())
	{
	  if (!ffile.eof())
	    warninglog(10) << "Error in formation read " << ende;
	  break;
	}
      actionlog(50) << "Read in formation '" << f.getName() << "'" << ende;
      storage.push_back(f);
      count++;
    }
    
  return true;
}

#ifdef OLD_CODE
  if (need_to_assign_marks &&
      Mem->NumConnectedPlayers(Mem->MySide) == Mem->SP_team_size &&
      Mem->NumConnectedPlayers(Mem->TheirSide) == Mem->SP_team_size) {

    if (!Mem->CP_analyze_log)
      formOpponent.permuteForGoalie(Mem->TheirSide);
    
    //we need to do this every time we switch formations
    if (Mem->CP_formation_create_marks) {
      LogAction2(30, "ModFormation: assigning marks based on the formation");
      assignMarks(formOpponent);
    }

    need_to_assign_marks = false;
  }

/* we'll use a define method so that we can redefine later */
void ModFormation::assignMarks(Formation& fOpp)
{
  /* we'll do the stupid thing and just mark the closest unmarked forward for each
     defender we find */
  //1-based arrays
  bool *player_marked = new bool[Mem->SP_team_size+1];
  Unum *mark_target = new Unum[Mem->SP_team_size+1];

  //we only want to mark forwards, so we'll consider everyone else marked
  for (int num=1; num<=Mem->SP_team_size; num++) {
    if (!fOpp.getHomeRegion(num)) {
      LogAction3(50, "Can't assign marks to opponent %d because it has no home region", num);
      player_marked[num] = true;
    } else {
      player_marked[num] = (fOpp.getRole(num) != PT_Forward);
    }
  }
    
  //Now, we'll go through our players and assign marks
  for (int num=1; num<=Mem->SP_team_size; num++) {
    if (getCurrentFormation()->getRole(num) != PT_Defender) {
      mark_target[num] = 0;
      continue;
    }

    RecordBestN best(1);
    for (int marknum=1; marknum<=Mem->SP_team_size; marknum++) {
      if (player_marked[marknum])
        continue;
      float dist_to_reg;
      getCurrentFormation()->getHomeRegion(num)->getClosestPointTo(fOpp.getHomeRegion(marknum)->getSomePointIn(), &dist_to_reg);
      best.AddPoint((void*)marknum, dist_to_reg);
    }

    if (best.GetNumStored() > 0) {
      Unum n = (int)best.GetBest();
      player_marked[n] = true;
      mark_target[num] = n;
    } else {
      mark_target[num] = 0;
    }
  }

  createMarkAdvice(mark_target);
  
  delete [] player_marked;
  delete [] mark_target;
}

/**************************************************************************/

bool ModFormation::createMarkAdvice(Unum aMarks[])
{
  //if a teammate is this close, the opp is considered ot already be marked
  const float max_mark_dist = 5.0;
  const char* mark_cond_base_name = "FMarkCond";
  const char* mark_dir_base_name = "FMarkDir";

  for (int num=1; num<=Mem->SP_team_size; num++) {
    if (aMarks[num] == 0)
      continue;

    char cname[max_name_len];
    char dname[max_name_len];
    UnumSet uset;

    sprintf(cname, "%s%d", mark_cond_base_name, num);
    sprintf(dname, "%s%d", mark_dir_base_name, num);
    
    CM_CondPlayerPosition* pCond;
    uset.addNum(0);
    uset.removeNum(num);
    pCond = new CM_CondPlayerPosition(true, uset, 1, 11);
    pCond->setRegion(RegionPtr(new RegArc(new RegPointPlayer(false, aMarks[num]),
                                          0, max_mark_dist, 0, 360)));
    mqueue.pushDefine(new CM_DefTokCondition(cname, new CM_CondNot(pCond)));

    uset.clear();
    uset.addNum(num);
    CM_DirCommand* pDir = new CM_DirCommand(true, true, uset);
    uset.clear();
    uset.addNum(aMarks[num]);
    pDir->setAction(new CM_ActMark(uset));
    mqueue.pushDefine(new CM_DefTokDirective(dname, pDir));
    
    if (!aBaseMarkAdviceSent[num-1]) {
      aBaseMarkAdviceSent[num-1] = true;

      CM_TokRule* pTok = new CM_TokRule(99999);
      CM_CondAnd* pCond = new CM_CondAnd;
      pCond->addCondition(new CM_CondPlayMode(CMPM_PlayOn));
      pCond->addCondition(new CM_CondNamed(cname));
      pTok->setCondition(pCond);
      pTok->addDirective(new CM_DirNamed(dname));

      mqueue.pushAdvice(pTok);
    }
    
  }
  return true;
}

#endif

