/* -*- Mode: C++ -*- */

#include <algorithm>
#include <iterator>
#include "ModFeatures.h"
#include "ModuleRegistry.h"
#include "CoachParam.h"
#include "Logger.h"

using namespace spades;

void
ModFeatures::initialize(ModuleRegistry* pReg)
{
  if (!CoachParam::instance()->getUseModFeatures())
    return;
  
  pReg->addModule(new ModFeatures());
}

/**********************************************************************************/

ModFeatures::ModFeatures()
  : Module("Features")
{
  curr_idx_ball_owner = 0;
  curr_idx_ball_poss = 0;
  aBallOwner = new PlayerID[CoachParam::instance()->getFeaturesCyclesToStore()];
  avBallPossess = new vector<PlayerID>[CoachParam::instance()->getFeaturesCyclesToStore()];
  for (int i = 0; i < CoachParam::instance()->getFeaturesCyclesToStore(); i++)
    {
      aBallOwner[i].side = TS_None;
      aBallOwner[i].num = 0;
    }
  if (CoachParam::instance()->getFeaturesCreateBallOwnerLog())
    {
      os_ball_owner_log.open(CoachParam::instance()->getFeaturesBallOwnerLogFN().c_str());
      if (!os_ball_owner_log)
	{
	  errorlog << "ModFeatures: Could not open ball owner log '"
		   << CoachParam::instance()->getFeaturesBallOwnerLogFN()
		   << "'"
		   << ende;
	}
      else
	{
	  os_ball_owner_log << "# This file was generated by ModFeatures of the OWL coach" << std::endl
			    << "# It gives the ball owner at each time" << std::endl
			    << "# Format: <time> <pid> (where <pid>=[side=<side>,num=<num>] (where <side>=None|Left|Right|Both and <num> is 1-based))" << std::endl;
	}
    }
  if (CoachParam::instance()->getFeaturesCreateBallPossessLog())
    {
      os_ball_possess_log.open(CoachParam::instance()->getFeaturesBallPossessLogFN().c_str());
      if (!os_ball_possess_log)
	{
	  errorlog << "ModFeatures: Could not open ball possess log '"
		   << CoachParam::instance()->getFeaturesBallPossessLogFN()
		   << "'"
		   << ende;
	}
      else
	{
	  os_ball_possess_log << "# This file was generated by ModFeatures of the OWL coach" << std::endl
			      << "# It gives the ball possessors (actually touching the ball) at each time" << std::endl
			      << "# Format: <time> <list of <pid>> (where <pid>=[side=<side>,num=<num>] (where <side>=None|Left|Right|Both and <num> is 1-based))" << std::endl;
	}
    }
}

ModFeatures::~ModFeatures()
{
  delete [] aBallOwner;
  delete [] avBallPossess;
}


void
ModFeatures::protStateUpdateNotify(Runner& runner, const WorldHistory& history)
{
  setBallPossessors(history.getWorldState());
  setBallOwner(history.getWorldState());
  logBallPossessors(history.getWorldState());
  logBallOwner(history.getWorldState());
}

void
ModFeatures::protSingleCall(SingleCallRunner& runner)
{
  warninglog(10) << "ModFeatures: I do not do anything with singleCall" << ende;
}


void
ModFeatures::setBallPossessors(const WorldState& ws)
{
  curr_idx_ball_poss = (curr_idx_ball_poss+1) % CoachParam::instance()->getFeaturesCyclesToStore();
  avBallPossess[curr_idx_ball_poss].clear();

  for (int num=0; num<ServerParam::instance()->getSPTeamSize()*2; num++)
    {

      const PlayerInfo* pPI = ws.getPlayer(num);
      if (pPI == NULL)
	continue;

      float karea;
      PlayerID player((num < ServerParam::instance()->getSPTeamSize()) ? TS_Left : TS_Right,
			(num % ServerParam::instance()->getSPTeamSize())+1);
      karea = ServerParam::instance()->getSPKickableArea();
      // SMURF: We're not doing anything with player types here! We should do that some time
#ifdef OLD_CODE
      if (Mem->CP_analyze_log)
	{
	  karea = Mem->getType(Mem->getPlayerTypeLog(player)).getKickableArea();
	}
      else
	{
	  if ((Mem->MySide == TS_Left && num < ServerParam::instance()->getSPTeamSize()) ||
	      (Mem->MySide == TS_Right && num >= ServerParam::instance()->getSPTeamSize()))
	    karea = Mem->getType(Mem->getTeammateType(player.num)).getKickableArea();
	  else
	    karea = Mem->pmObservePType->getMaxPossible(player.num).getKickableArea();
	}
#endif
    
      actionlog(220) << "ModFeatures::setBallPossessors " << num << " pos " << pPI->getPos()
		     << " ballpos " << ws.getBall().getPos() << ende;
      actionlog(220) << "ModFeatures::setBallPossessors " << num
		     << ", dist: " << pPI->getPos().getDistanceTo(ws.getBall().getPos())
		     << " karea: " << karea
		     << ende;
      if (pPI->getPos().getDistanceTo(ws.getBall().getPos()) <= karea)
	{
	  avBallPossess[curr_idx_ball_poss].push_back(player);
	}
    }

  actionlog(70) << "ModFeatures: saw " << avBallPossess[curr_idx_ball_poss].size()
	       << " ball possessors" << ende;
}

void
ModFeatures::setBallOwner(const WorldState& ws)
{
  //setBallPossessors must be called before this!
  int prev_idx_bo = curr_idx_ball_owner;
  curr_idx_ball_owner = (curr_idx_ball_owner+1) % CoachParam::instance()->getFeaturesCyclesToStore();

  switch (ws.getPlayMode())
    {
      //SMURF BUG: SPHalfCycles can be wrong when slow_down_factor is greater than 1
      // and we receive parameters from the server
    case PM_BeforeKickOff:
      if (ws.getTime() == 0)
	{
	  aBallOwner[curr_idx_ball_owner].side = TS_Left;
	  aBallOwner[curr_idx_ball_owner].num  = 0;
	}
      else if (ws.getTime() == ServerParam::instance()->getSPHalfCycles())
	{
	  aBallOwner[curr_idx_ball_owner].side = TS_Right;
	  aBallOwner[curr_idx_ball_owner].num  = 0;
	}
      else
	{
	  errorlog << "setBallOwner: I don't think we should be in BeforeKickOff here "
		   << ServerParam::instance()->getSPHalfCycles()
		   << ende;
	}
      break;
    case PM_TimeOver:
    case PM_PenaltySetup_Left:
    case PM_PenaltySetup_Right:
    case PM_PenaltyReady_Left:
    case PM_PenaltyReady_Right:
    case PM_PenaltyTaken_Left:
    case PM_PenaltyTaken_Right:
    case PM_PenaltyMiss_Left:
    case PM_PenaltyMiss_Right:
    case PM_PenaltyScore_Left:
    case PM_PenaltyScore_Right:
      aBallOwner[curr_idx_ball_owner].num  = 0;
      aBallOwner[curr_idx_ball_owner].side = TS_None;
      break;
    case PM_KickOff_Left:
    case PM_KickIn_Left:
    case PM_FreeKick_Left:
    case PM_CornerKick_Left:
    case PM_GoalKick_Left:
    case PM_AfterGoal_Left:
    case PM_IndFreeKick_Left:
      aBallOwner[curr_idx_ball_owner].side = TS_Left;
      aBallOwner[curr_idx_ball_owner].num  = 0;
      break;
    case PM_KickOff_Right:
    case PM_KickIn_Right:
    case PM_FreeKick_Right:
    case PM_CornerKick_Right:
    case PM_GoalKick_Right:
    case PM_AfterGoal_Right:
    case PM_IndFreeKick_Right:
      aBallOwner[curr_idx_ball_owner].side = TS_Right;
      aBallOwner[curr_idx_ball_owner].num  = 0;
      break;
    case PM_OffSide_Left:
    case PM_OffSide_Right:
    case PM_Back_Pass_Left:
    case PM_Back_Pass_Right:
    case PM_Free_Kick_Fault_Left:
    case PM_Free_Kick_Fault_Right:
    case PM_CatchFault_Left:
    case PM_CatchFault_Right:
      //these are temporary play modes -- all will shift to free kicks in a moment, so we'll just
      // call them all none
      aBallOwner[curr_idx_ball_owner].side = TS_None;
      aBallOwner[curr_idx_ball_owner].num  = 0;
      break;
    case PM_Drop_Ball:
      aBallOwner[curr_idx_ball_owner].side = TS_None;
      aBallOwner[curr_idx_ball_owner].num  = 0;
      break;
    case PM_PlayOn: {
      int s = avBallPossess[curr_idx_ball_poss].size();
      if (s == 0)
	{
	  aBallOwner[curr_idx_ball_owner] = aBallOwner[prev_idx_bo];
	  if (aBallOwner[curr_idx_ball_owner].side == TS_Both)
	    aBallOwner[curr_idx_ball_owner].side = TS_None;
	  return;
	}
      else if (s == 1)
	{
	  aBallOwner[curr_idx_ball_owner] = avBallPossess[curr_idx_ball_poss][0];
	}
      else
	{
	  vector<PlayerID>::iterator iter;
	  aBallOwner[curr_idx_ball_owner].num  = 0;
	  //Look to see what side should be
	  aBallOwner[curr_idx_ball_owner].side = TS_None;
	  bool done = false;
	  for (iter = avBallPossess[curr_idx_ball_poss].begin();
	       !done && iter != avBallPossess[curr_idx_ball_poss].end();
	       ++iter)
	    {
	      switch (aBallOwner[curr_idx_ball_owner].side)
		{
		case TS_Left:
		case TS_Right:
		  if (aBallOwner[curr_idx_ball_owner].side != iter->side)
		    {
		      aBallOwner[curr_idx_ball_owner].side = TS_Both;
		      done = true;
		    }
		  break;
		case TS_None:
		  aBallOwner[curr_idx_ball_owner].side = iter->side;
		  break;
		case TS_Both:
		  errorlog << "Once I see TS_Both, the loop should exit!";
		  break;
		default:
		  errorlog << "how did ball owner get this value? " << aBallOwner[curr_idx_ball_owner].side
			   << ende;
		}
	    }    
	}
    } break;
    default:
      errorlog << "setBallOwner: What is play mode? " << ws.getPlayMode() << ende;
    }

  actionlog(70) << "ModFeatures: ball owner is " << aBallOwner[curr_idx_ball_owner].side << ende;
}

const vector<PlayerID>&
ModFeatures::getPrevBallPossessors(int cycles_back)
{
  if (cycles_back >= CoachParam::instance()->getFeaturesCyclesToStore())
    {
      errorlog << "getPrevBallPossessors: can't look back that far "
	       << cycles_back << " >= " << CoachParam::instance()->getFeaturesCyclesToStore() << ende;
      return avBallPossess[curr_idx_ball_poss];
    }

  return avBallPossess[(curr_idx_ball_poss - cycles_back + CoachParam::instance()->getFeaturesCyclesToStore()) %
		       CoachParam::instance()->getFeaturesCyclesToStore()];
}

PlayerID
ModFeatures::getPrevBallOwner(int cycles_back) 
{
  if (cycles_back >= CoachParam::instance()->getFeaturesCyclesToStore())
    {
      errorlog << "getPrevBallOwner: can't look back that far "
	       << cycles_back << " >= " << CoachParam::instance()->getFeaturesCyclesToStore() << ende;
      return aBallOwner[curr_idx_ball_owner];
    }

  return aBallOwner[(curr_idx_ball_owner - cycles_back + CoachParam::instance()->getFeaturesCyclesToStore()) %
		    CoachParam::instance()->getFeaturesCyclesToStore()];
}

void
ModFeatures::logBallPossessors(const WorldState& ws)
{
  if (!CoachParam::instance()->getFeaturesCreateBallPossessLog())
    return;
  os_ball_possess_log << ws.getTime() << ' ';
  std::copy(getBallPossessors().begin(), getBallPossessors().end(),
	    std::ostream_iterator<PlayerID>(os_ball_possess_log, " "));
  os_ball_possess_log << std::endl;
  
}

void
ModFeatures::logBallOwner(const WorldState& ws)
{
  if (!CoachParam::instance()->getFeaturesCreateBallOwnerLog())
    return;
  os_ball_owner_log << ws.getTime() << ' '
		    << getBallOwner()
		    << std::endl;
}

  
