/* A Module for a repository of coach messages. Manages all in and out messages */

#include <iostream>
#include "ModCMRep.h"
#include "ModuleRegistry.h"
#include "CoachParam.h"
#include "Logger.h"
#include "clangparser.h"
#include "clangmsgbuilder.h"
#include "clangmsg.h"
#include "SorterMsgBuilder.h"
#include "constants.h"
#include "utility.h"
#include "OnlineRunner.h"

using namespace spades;
using namespace std;

void
ModCMRep::initialize(ModuleRegistry* pReg)
{
  if (!CoachParam::instance()->getUseModCMRep())
    return;

  pReg->addModule(new ModCMRep());
}

ModCMRep::ModCMRep()
  : Module("CMRep"),
    define_messages_left(-1),
    rule_messages_left(-1),
    del_messages_left(-1),
    msg_left_update_time(-1),
    freeform_messages_left(-1),
    last_ff_mess_update_time(-1),
    init_delay(CoachParam::instance()->getInitMessagesDelay()),
    messages_sent(0),
    did_say_last_cycle(false),
    init_messages_queue("InitMessages")
{
  addCMQ(&init_messages_queue);
  queueInitMessages();
}

ModCMRep::~ModCMRep()
{
}

void
ModCMRep::protStateUpdateNotify(Runner& runner, const WorldHistory& history)
{
  errorlog << "ModCMRep: how am I in normal stateUpdate?" << ende;
}

void
ModCMRep::protStateUpdateOnlineNotify(OnlineRunner& online_runner, const WorldHistory& history)
{
  updateMessagesLeft(history.getWorldState().getTime());
  handleMessageQueues(online_runner);
}

bool
ModCMRep::dependencyCheck(ModuleRegistry* pReg, LogfileRunner& runner)
{
  errorlog << "ModCMRep: Logfile not supported yet" << ende;
  return false;
}


void
ModCMRep::protSingleCall(SingleCallRunner& runner)
{
  errorlog << "ModCMRep: I don't handle single call" << ende;
}

//all modules that want to send coach messages should register here
void
ModCMRep::addCMQ(CoachMessageQueue* p)
{
  actionlog(20) << "Adding coach message generator '" << p->getName() << "'" << ende;
  vCMQ.push_back(p);
}

bool
ModCMRep::removeCMQ(CoachMessageQueue* p)
{
  vector<CoachMessageQueue*>::iterator i = vCMQ.begin();
  vector<CoachMessageQueue*>::iterator nexti = i;
  nexti++;
  for (;
       i != vCMQ.end();
       i=nexti, nexti++) {
    if (*i == p) {
      actionlog(20) << "Removed coach message generator '" << (*i)->getName() << "'" << ende;
      nexti = vCMQ.erase(i);
      return true;
    } 
  }
  errorlog << "Could not find coach message generator '" << p->getName() << "' to remove" << ende;
  return false;
}

bool
ModCMRep::updateMessagesLeft(int time)
{
  if ((time % (2*ServerParam::instance()->getSPHalfCycles())) == 0 &&
      ServerParam::instance()->getSPSayCoachCntMax() >= 0 &&
      time > last_ff_mess_update_time) {
    actionlog(150) << "Coach gets another set of freeform messages to say "
		   << time << ' '
		   << ServerParam::instance()->getSPHalfCycles() << ende;
    freeform_messages_left += ServerParam::instance()->getSPSayCoachCntMax();
    last_ff_mess_update_time = time;
  }

  /* The update here is a little funny. The server does it's update at the beginning
     of a cycle, but BEFORE the time is incremented. Therefore, we need to wait to
     increment for an extra cycle, but then say we incremented at the last cycle */
  if (msg_left_update_time >= 0 &&
      time - msg_left_update_time < ServerParam::instance()->getSPClangWinSize() + 1)
    return false;

  actionlog(100) << "Time for more coach messages to be allowed!" << ende;
  define_messages_left = ServerParam::instance()->getSPClangDefineWin();
  rule_messages_left   = ServerParam::instance()->getSPClangRuleWin();
  del_messages_left    = ServerParam::instance()->getSPClangDelWin();
  msg_left_update_time = time - 1;
  
  return true;
}

bool
ModCMRep::canISayFreeform(const WorldHistory& history)
{
  //SMURF: this is not right yet! We can say freeform in play on sometimes
  return history.getWorldState().getPlayMode() != PM_PlayOn;
}

bool
ModCMRep::canISayClang(const WorldHistory& history, int msg_window_cnt)
{
  if (history.getWorldState().getPlayMode() != PM_PlayOn ||
      msg_window_cnt != 0)
    return true;
  return false;
}

void
ModCMRep::decrClangWinCnt(const WorldHistory& history, int* p_msg_window_cnt)
{
  if (history.getWorldState().getPlayMode() != PM_PlayOn)
    return;
  if (*p_msg_window_cnt < 0)
    return;
  if (*p_msg_window_cnt == 0)
    {
      errorlog << "I was told to decrement a window count, but it's already 0!" << ende;
      return;
    }
  (*p_msg_window_cnt)--;
}


void
ModCMRep::queueInitMessages() 
{
  if (CoachParam::instance()->getInitialMessagesFN().empty())
    return;
  
  //Load the initial messages file
  ifstream infile_init_messages(CoachParam::instance()->getInitialMessagesFN().c_str());
  if (!infile_init_messages) {
    errorlog << "Could not open initial messages file '"
	     << CoachParam::instance()->getInitialMessagesFN()
	     << "'" << ende;
    return;
  }

  actionlog(50) << "Loading initial coach messages from file '"
		<< CoachParam::instance()->getInitialMessagesFN()
		<< "'" << ende;
  
  
  SorterMsgBuilder sorter_msg_builder(&init_messages_queue);
  rcss::clang::Parser clang_parser(sorter_msg_builder);

  /* We do our own line stuff so that we can skip comments in the file */
  for (;;)
    {
      if (!skip_to_non_comment(infile_init_messages))
	{
	  //eof
	  infile_init_messages.close();
	  break;
	}

      char line[MAX_SERVER_MSG_LEN];
      infile_init_messages.getline(line, MAX_SERVER_MSG_LEN);
      try {
	if (clang_parser.parse(line) != 0)
	  {
	    errorlog << "Could not parse line from initial_messages_fn: " << line << ende;
	  }
	else
	  {
	    //TEST CODE:
	    //std::cout << "Parsed line: " << line << endl;
	  }
      }
      catch( rcss::clang::BuilderErr e)
	{ errorlog << "Error parsing init messages file: " << e.what() << ende; }
      catch( rcss::util::NullErr e)
	{ errorlog << "Error parsing init messages file: " << e.what() << ende; }
    }

  actionlog(50) << "ModCMRep: Read initial messages from file: "
		<< sorter_msg_builder.getDefineElementCnt() << " define elems, "
		<< sorter_msg_builder.getFreeformElementCnt() << " freeform elems, "
		<< sorter_msg_builder.getRuleElementCnt() << " rule elems, "
		<< sorter_msg_builder.getDelElementCnt() << " del elems"
		<< ende;
}

template <class T>
class CopyDerefAndDelete 
  : public std::unary_function<T*, void>
{
public:
  CopyDerefAndDelete(rcss::util::HasMany<T>& l) : l(l) {}

  void operator() (T* p) { l.push_back(*p); delete p; }
private:
  rcss::util::HasMany<T>& l;
};

template <class T>
class SpliceListAndDelete
{
public:
  SpliceListAndDelete(std::list<T>& l) : l(l) {}
  void operator() (std::list<T>*p) { l.splice(l.end(), *p); delete p; }
private:
  std::list<T>& l;
};

void
ModCMRep::handleMessageQueues(OnlineRunner& online_runner)
{
  int num_players =
    online_runner.getWorldHistory().getWorldState().getNumPlayers(online_runner.getMySide());
  if (num_players < CoachParam::instance()->getMinPlayersBeforeSay())
    {
      actionlog(100) << "ModCMRep: Not enough player on the field to talk "
		     << num_players
		     << " < "
		     << CoachParam::instance()->getMinPlayersBeforeSay()
		     << ende;
      return;
    }
  
  if (init_delay > 0)
    {
      init_delay--;
      actionlog(100) << "ModCMRep: delaying the start of send: " << init_delay << " cycles left" << ende;
      return;
    }

  actionlog(100) << "ModCMRep: checking the message queues: "
		 << freeform_messages_left << " "
		 << define_messages_left << " "
		 << rule_messages_left << " "
		 << del_messages_left << ende;
  
  int messages_left_this_cycle = CoachParam::instance()->getMaxSaysPerCycle();
  if (online_runner.getWorldHistory().getWorldState().getTime() %
      CoachParam::instance()->getSayFrequency() != 0)
    messages_left_this_cycle = 0;
  if (messages_left_this_cycle < 0)
    messages_left_this_cycle = std::numeric_limits<int>::max();
  bool define_items_left = true;
  bool items_left = true;

  //do freeform queue
  CoachMessageQueueStorage::iterator cmqiter;

  if (canISayFreeform(online_runner.getWorldHistory()))
    {
      for(cmqiter = vCMQ.begin();
	  cmqiter != vCMQ.end();
	  ++cmqiter)
	{
	  while (messages_left_this_cycle > 0 && (*cmqiter)->getFreeformContainer().peek())
	    {
	      if (freeform_messages_left <= 0)
		{
		  errorlog << "I have a freeform message to say from '"
			   << (*cmqiter)->getName() << "', but I'm out of messages"
			   << ende;
		  break;
		}
	      rcss::clang::FreeformMsg* m = (*cmqiter)->getFreeformContainer().pop();
	      CoachCommandSay cmd(m);
	      online_runner.sendToServer(&cmd);
	      freeform_messages_left--;
	      messages_left_this_cycle--;
	      messages_sent++;
	    }

	}
    }

  int max_coach_mesg =
    Min(MAX_SERVER_COACH_SAY_LEN, 
	(CoachParam::instance()->getMaxSayLength() > 0) ? CoachParam::instance()->getMaxSayLength() : 999999);
  int max_define_size = max_coach_mesg - 9; // '(define' ')'
  int max_rule_size = max_coach_mesg - 6; // '(rule' ')'
  int max_del_size = max_coach_mesg - 9; // '(delete' ')'
  bool did_say_this_cycle = false;
  
  /* this loop looks different that the others because we want to check if there are define tokens
     left even if we have no define messages left */
  while (messages_left_this_cycle > 0)
    {
      // we use this so that define_items_left gets set correctly even though we can't say anything
      if (!canISayClang(online_runner.getWorldHistory(), define_messages_left))
	max_define_size = 0;
      std::list<rcss::clang::Def*> lDef =
	getElements<rcss::clang::Def> (max_define_size, CMQDefineExtractor(), &define_items_left);
      if (lDef.empty())
	break;
      rcss::clang::DefineMsg* msg = new rcss::clang::DefineMsg(lDef);
      CoachCommandSay cmd(msg);
      online_runner.sendToServer(&cmd);
      decrClangWinCnt(online_runner.getWorldHistory(), &define_messages_left);
      messages_left_this_cycle--;
      messages_sent++;
      did_say_this_cycle = true;
      
      if (!items_left)
	break;
    }
  
  // we should only go to the rule and delete queues if the define queues are empty 
  if (messages_left_this_cycle > 0  && !define_items_left)
    {
      //do rule queue
      while (messages_left_this_cycle > 0 &&
	     canISayClang(online_runner.getWorldHistory(), rule_messages_left))
	{
	  std::list<rcss::clang::ActivateRules*> lRule =
	    getElements<rcss::clang::ActivateRules> (max_rule_size, CMQRuleExtractor(), &items_left);
	  if (lRule.empty())
	    break;
	  rcss::clang::RuleMsg* msg = new rcss::clang::RuleMsg();
	  for_each(lRule.begin(), lRule.end(), 
		   CopyDerefAndDelete<rcss::clang::ActivateRules>(*msg));
	  CoachCommandSay cmd(msg);
	  online_runner.sendToServer(&cmd);
	  decrClangWinCnt(online_runner.getWorldHistory(), &rule_messages_left);
	  messages_left_this_cycle--;
	  messages_sent++;
	  did_say_this_cycle = true;

	  if (!items_left)
	    break;
	}

      //do delete queue
      while (messages_left_this_cycle > 0 &&
	     canISayClang(online_runner.getWorldHistory(), del_messages_left))
	{
	  std::list<rcss::clang::RuleIDList*> lDel =
	    getElements<rcss::clang::RuleIDList> (max_del_size, CMQDelExtractor(), &items_left);
	  if (lDel.empty())
	    break;
	  rcss::clang::RuleIDList output_list;
	  for_each(lDel.begin(), lDel.end(), SpliceListAndDelete<rcss::clang::RuleID>(output_list));
	  rcss::clang::DelMsg* msg = new rcss::clang::DelMsg(output_list);
	  CoachCommandSay cmd(msg);
	  online_runner.sendToServer(&cmd);
	  decrClangWinCnt(online_runner.getWorldHistory(), &del_messages_left);
	  messages_left_this_cycle--;
	  messages_sent++;
	  did_say_this_cycle = true;

	  if (!items_left)
	    break;
	}

    }

  actionlog(100) << "ModCMRep: done with message queues: "
		 << freeform_messages_left << " "
		 << define_messages_left << " "
		 << rule_messages_left << " "
		 << del_messages_left << ende;
  
  if (did_say_last_cycle && !did_say_this_cycle)
    {
      actionlog(100) << "Said last cycle but not this cycle" << ende;
      std::cout << "ModCMRep: I talked last cycle but not this cycle" << std::endl;
    }

  did_say_last_cycle = did_say_this_cycle;
}

