/* -*- Mode: C++ -*- */

#include <sstream>
#include "ModAbstract.h"
#include "CoachParam.h"
#include "ModuleRegistry.h"
#include "OnlineRunner.h"
#include "LogfileRunner.h"
#include "ModFeatures.h"
#include "ModCMRep.h"
#include "FixedRWMI.h"
#include "AbstractStateElements.h"
#include "AbstractStateDifference.h"
#include "MarkovChain.h"
#include "LibSeaGraphWriter.h"
#include "MDPConversion.h"
#include "QTableFlex.h"
#include "AdviceTree.h"
#include "Logger.h"
#include "SingleCallRunner.h"

using namespace spades;

const char* ModAbstract::ADVICE_TREE_RULE_NAME = "AbsAdviceTree";
const char* ModAbstract::ADVICE_TREE_MULTI_RULE_PREFIX = "ABMDP_";

void
ModAbstract::initialize(ModuleRegistry* pReg)
{
  if (!CoachParam::instance()->getUseModAbstract())
    return;

  pReg->addModule(new ModAbstract());
}

/**************************************************************************/

ModAbstract::ModAbstract()
  : Module("Abstract"),
    AdaptationUnit(),
    ppotracker(NULL),
    pmFeatures(NULL),
    pmCMRep(NULL),
    mqueue("Abstract"),
    state_description(),
    curr_abs_state(&state_description),
    abstract_state_tracker_init(false),
    abstract_state_trackers(),
    advice_by_mdp_init(false),
    advice_by_mdp_init_filters(false),
    soccer_model(),
    p_advice_tree(NULL),
    out_abmdp_track(),
    abmdp_lfilters_state(),
    abmdp_lfilters_act(),
    v_multi_abmdp()
{
  if (CoachParam::instance()->getAbstractTrackPO())
    {
      PlayerOccupancySet poset;
      POSetFileReader reader(&poset);
      reader.readFile(CoachParam::instance()->getAbstractTrackPOSetFN().c_str());
      // While set the team side later if it's not both
      ppotracker =
	new PlayerOccupancyTracker(CoachParam::instance()->getAbstractTrackPOBothTeams() ?
				   TS_Both : TS_None,
				   poset,
				   CoachParam::instance()->getAbstractTrackPOWritePerCycle() ?
				   CoachParam::instance()->getAbstractTrackPOPerCycleFN().c_str() :
				   NULL);
    }

  initializeMultiABMDP();
  
  initializeSoccerModel();
  
  if (CoachParam::instance()->getAbstractABMDPTrack())
    {
      out_abmdp_track.open(CoachParam::instance()->getAbstractABMDPTrackFN().c_str());
      if (!out_abmdp_track)
	errorlog << "AsbstractAdviceByMDPTrack: Could not open file '"
		 << CoachParam::instance()->getAbstractABMDPTrackFN()
		 << "'" << ende;
      out_abmdp_track << "# Generated by ModAbstract: advice by mdp tracking mode" << std::endl;
      out_abmdp_track << "# Format: <time> <state_idx> <reward> <list of rec actions>" << std::endl;
    }  
}


ModAbstract::~ModAbstract()
{
  if (ppotracker)
    {
      if (CoachParam::instance()->getAbstractTrackPOWriteSumm())
	{
	  std::ofstream os(CoachParam::instance()->getAbstractTrackPOSummFN().c_str());
	  if (!os)
	    errorlog << "ModAbstract: Could not open po tracker summary file '"
		     << CoachParam::instance()->getAbstractTrackPOSummFN()
		     << "'" << ende;
	  else
	    ppotracker->writeSummaryInfo(os);
	}
      delete ppotracker;
    }

  if (CoachParam::instance()->getAbstractStateTrackWriteSumm())
    {
      for (TrackerStorage::iterator iter = abstract_state_trackers.begin();
	   iter != abstract_state_trackers.end();
	   iter++)
	{
	  std::string fn = CoachParam::instance()->getAbstractStateTrackSummFN();
	  fn = spades::findReplaceInStr(fn, "%S",
					(*iter)->getMySide() == TS_Left ? "left" : "right");
	  std::ofstream os(fn.c_str());
	  if (!os)
	    errorlog << "ModAbstract: Could not open state tracker summary file '"
		     << fn
		     << "'" << ende;
	  else
	    (*iter)->writeSummaryInfo(os);
	}
    }

  std::for_each(abstract_state_trackers.begin(), abstract_state_trackers.end(),
		deleteptr<AbstractStateTracker>());
  abstract_state_trackers.clear();

  soccer_model.forgetStateDescription();

  clearABMDPFilters();

  std::for_each(v_multi_abmdp.begin(), v_multi_abmdp.end(), deleteptr<ABMDP>());
  v_multi_abmdp.clear();
  
  if (pmCMRep)
    pmCMRep->removeCMQ(&mqueue);
}

void
ModAbstract::initializeSoccerModel()
{
  if (CoachParam::instance()->getAbstractABMDPMultiCount() > 0)
    {
      if (CoachParam::instance()->getRunMode() != RM_Online)
        return;

      std::cout << "ModAbstract: loading multi mdp and qtable; this could take a while...." << std::endl;

      for (int idx = 0;
           idx < CoachParam::instance()->getAbstractABMDPMultiCount();
           ++idx)
        {
          if (!v_multi_abmdp[idx]->readMDPAndQTable(CoachParam::instance()->getAbstractABMDPMultiMDPFN()[idx].c_str(),
                                                    CoachParam::instance()->getAbstractABMDPMultiQTableFN()[idx].c_str()))
            errorlog << "ModAbstract: read for multi abdmp at idx " << idx << " failed: "
                     << CoachParam::instance()->getAbstractABMDPMultiMDPFN()[idx] << " "
                     << CoachParam::instance()->getAbstractABMDPMultiQTableFN()[idx]
                     << ende;
        }
      
      std::cout << "ModAbstract: Done loading." << std::endl;
    }
  else
    {
      if (CoachParam::instance()->getAbstractAdviceByMDP() ||
          CoachParam::instance()->getAbstractCompareStateTraceToModel() ||
          CoachParam::instance()->getAbstractComputeAvgReward() ||
          CoachParam::instance()->getAbstractGenerateMDPTraces() ||
          CoachParam::instance()->getAbstractABMDPTrack() ||
          CoachParam::instance()->getAbstractAnalyzePercentOptimal())
        {
          std::cout << "ModAbstract: loading mdp and qtable; this could take a while...." << std::endl;
          if (!soccer_model.readMDPFrom(CoachParam::instance()->getAbstractABMDPFN().c_str()))
            errorlog << "ModAbstract: could not read mdp for AdviceByMDP" << ende;
          if (!soccer_model.readQTableFrom(CoachParam::instance()->getAbstractABMDPQTableFN().c_str()))
            errorlog << "ModAbstract: could not read qtable for AdviceByMDP" << ende;
          std::cout << "ModAbstract: Done loading." << std::endl;
        }
    }
}

AbstractStateFactor*
ModAbstract::createAbstractStateTree(int num_xbins, int num_ybins, const char* poset_fn)
{
  AbstractStateFactorOr* pOr = new AbstractStateFactorOr();
  pOr->addFactor(new GoalFactor());

  AbstractStateFactorAnd* pDeadBall = new AbstractStateFactorAnd();
  pDeadBall->addFactor(new DeadBallFactor());
  BallGridFactor* pBGFac =
    new BallGridFactor(num_xbins, num_ybins);
  //#define DRAW_WHOLE_BALL_GRID
#ifdef DRAW_WHOLE_BALL_GRID
  std::cout << "Drawing entire ball grid" << std::endl;
  FieldImage* pBGImage = new FieldImage(0, 8);
  pBGImage->addFieldLines(FieldImage::COLOR_GREY);
  pBGFac->drawEntireGrid(pBGImage, FieldImage::COLOR_BLACK);
  //big hack for circle pass stuff
  //pBGImage->addPoint(VecPosition(10, -17), FieldImage::COLOR_BLUE, 2.0);
  //pBGImage->addPoint(VecPosition(20, 0), FieldImage::COLOR_BLUE, 2.0);
//   pBGImage->addPoint(VecPosition(10, 17), FieldImage::COLOR_BLUE, 2.0);
//   pBGImage->addPoint(VecPosition(-10, 17), FieldImage::COLOR_BLUE, 2.0);
//   pBGImage->addPoint(VecPosition(-20, 0), FieldImage::COLOR_BLUE, 2.0);
//   pBGImage->addPoint(VecPosition(-19, -21), FieldImage::COLOR_BLUE, 2.0);
  if (!pBGImage->writeTo("ballgrid"))
    errorlog << "Error writing ball grid image" << ende;
  delete pBGImage;
#endif	
  pDeadBall->addFactor(pBGFac);
  pOr->addFactor(pDeadBall);
      
  AbstractStateFactorAnd* pAnd = new AbstractStateFactorAnd();
  pAnd->addFilter(new PlayModeFilter(PM_PlayOn));
  pAnd->addFilter(new BallKickableFilter(pmFeatures));
  PlayerOccupancySet poset;
  POSetFileReader reader(&poset);
  reader.readFile(poset_fn);
  //This line creates a single factor for the whole POSet
  //pAnd->addFactor(new POSetFactor(poset));
  // while this line creates an AND factor with each element underneath
  // This latter way is better so that we can do more uniform operations
  // on abstract state factors
  pAnd->addFactor(poset.createAbstractStateFactor());
  pAnd->addFactor(new BallGridFactor(num_xbins, num_ybins));
  pAnd->addFactor(new BallOwnerFactor(pmFeatures));

  pOr->addFactor(pAnd);

  return pOr;
}


void
ModAbstract::initializeStateDescription()
{
  actionlog(100) << "ModAbstract: Initializing state description" << ende;

  if (CoachParam::instance()->getAbstractABMDPMultiCount() > 0)
    {
      if (CoachParam::instance()->getRunMode() != RM_Online)
        {
          errorlog << "ModAbstract::initializeStateDescription: I probably shouldn't be in muti in non online mode" << ende;
          return;
        }

      for (int idx = 0;
           idx < CoachParam::instance()->getAbstractABMDPMultiCount();
           ++idx)
        {
          AbstractStateFactor* pfac =
            createAbstractStateTree(CoachParam::instance()->getAbstractStateBallNumXBins(),
                                    CoachParam::instance()->getAbstractStateBallNumYBins(),
                                    CoachParam::instance()->getAbstractABMDPMultiPOSetFN()[idx].c_str());
          v_multi_abmdp[idx]->setStateDescription(new AbstractStateDescription(pfac));
        }
      
    }
  else
    {

      state_description.setFactor(createAbstractStateTree(CoachParam::instance()->getAbstractStateBallNumXBins(),
                                                          CoachParam::instance()->getAbstractStateBallNumYBins(),
                                                          CoachParam::instance()->getAbstractStatePOSetFN().c_str()));
  
      // Since we added factors, we need to let our state know
      curr_abs_state.updateStorageSize();

      if (CoachParam::instance()->getAbstractAdviceByMDP() ||
          CoachParam::instance()->getAbstractCompareStateTraceToModel() ||
          CoachParam::instance()->getAbstractComputeAvgReward() ||
          CoachParam::instance()->getAbstractAnalyzePercentOptimal())
        {
          cout << "ModAbstract: Processing state description; This could take a while..." << endl;
          soccer_model.setStateDescription(&state_description);
          cout << "ModAbstract: Done processing" << endl;
        }
    }

}

void
ModAbstract::initializeMultiABMDP()
{
  if (CoachParam::instance()->getAbstractABMDPMultiCount() <= 0)
    return;

  if (CoachParam::instance()->getRunMode() != RM_Online)
    return;

  if ((signed)CoachParam::instance()->getAbstractABMDPMultiPOSetFN().size() !=
      CoachParam::instance()->getAbstractABMDPMultiCount())
    {
      errorlog << "ModAbstract: multi ABMDP bad size for poset "
               << CoachParam::instance()->getAbstractABMDPMultiPOSetFN().size()
               << " " << CoachParam::instance()->getAbstractABMDPMultiCount()
               << ende;
      return;
    }
  if ((signed)CoachParam::instance()->getAbstractABMDPMultiMDPFN().size() !=
      CoachParam::instance()->getAbstractABMDPMultiCount())
    {
      errorlog << "ModAbstract: multi ABMDP bad size for mdp "
               << CoachParam::instance()->getAbstractABMDPMultiMDPFN().size()
               << " " << CoachParam::instance()->getAbstractABMDPMultiCount()
               << ende;
      return;
    }
  if ((signed)CoachParam::instance()->getAbstractABMDPMultiQTableFN().size() !=
      CoachParam::instance()->getAbstractABMDPMultiCount())
    {
      errorlog << "ModAbstract: multi ABMDP bad size for qtable "
               << CoachParam::instance()->getAbstractABMDPMultiQTableFN().size()
               << " " << CoachParam::instance()->getAbstractABMDPMultiCount()
               << ende;
      return;
    }

  if (CoachParam::instance()->getAbstractABMDPMultiStratToMDPIdx().size() != AS_MAX - 1)
    {
      warninglog(10) << "ModAbstract: multi-abmdp: I'm expecting the same number of mdps as adapt styles: "
                     << CoachParam::instance()->getAbstractABMDPMultiStratToMDPIdx().size()
                     << " " << (AS_MAX - 1)
                     << ende;
      
    }

  char leading_char = 'A';
  for (int idx = 0;
       idx < CoachParam::instance()->getAbstractABMDPMultiCount();
       ++idx)
    {
      ABMDP* pabmdp = new ABMDP;
      pabmdp->setLeadingChar(leading_char);
      pabmdp->setTopRuleName(std::string(ADVICE_TREE_MULTI_RULE_PREFIX) + leading_char);
      pabmdp->setTreeAssocFN(CoachParam::instance()->getAbstractABMDPMultiTreeAssocFPat());

      v_multi_abmdp.push_back(pabmdp);

      ++leading_char;
    }
}

void
ModAbstract::clearABMDPFilters()
{
  std::for_each(abmdp_lfilters_state.begin(), abmdp_lfilters_state.end(),
		deleteptr<SoccerStateFilter>());
  std::for_each(abmdp_lfilters_act.begin(), abmdp_lfilters_act.end(),
		deleteptr<SoccerActionFilter>());
  abmdp_lfilters_state.clear();
  abmdp_lfilters_act.clear();

  advice_by_mdp_init_filters = false;
}



void
ModAbstract::protStateUpdateNotify(Runner& runner, const WorldHistory& history)
{
  errorlog << "ModAbstract::stateUpdateNotify should never be called" << ende;
}

void
ModAbstract::protStateUpdateOnlineNotify(OnlineRunner& online_runner,
					 const WorldHistory& history)
{
  handlePOTracker(online_runner.getMySide(), history.getWorldState());
  initializeAbstractStateTrackers(online_runner.getMySide());
  handleAbstractStateTracker(online_runner.getMySide(), history.getWorldState());
  handleAdviceByMDP(online_runner.getMySide(), history);
  handleTrackAdvisedActions(online_runner.getMySide(), history.getWorldState());
}

void
ModAbstract::protStateUpdateLogfileNotify(LogfileRunner& logfile_runner,
					  const WorldHistory& history)
{
  handlePOTracker(logfile_runner.getSideToAnalyze(), history.getWorldState());
  initializeAbstractStateTrackers(logfile_runner.getSideToAnalyze());
  handleAbstractStateTracker(logfile_runner.getSideToAnalyze(), history.getWorldState());
  handleTrackAdvisedActions(logfile_runner.getSideToAnalyze(), history.getWorldState());
}

void
ModAbstract::protSingleCall(SingleCallRunner& runner)
{
  if (CoachParam::instance()->getAbstractDrawPOSet())
    {
      drawPOSet();
    }

  if (CoachParam::instance()->getAbstractLearnMarkovChain())
    {
      learnMarkovChain();
    }

  if (CoachParam::instance()->getAbstractLearnMCV())
    {
      learnMarkovChainValue(runner);
    }

  if (CoachParam::instance()->getAbstractPruneMC())
    {
      pruneMarkovChain();
    }

  if (CoachParam::instance()->getAbstractMCCreateGraph())
    {
      createMarkovChainGraph();
    }

  if (CoachParam::instance()->getAbstractMCCalcDist())
    {
      calcMarkovChainDist();
    }

  if (CoachParam::instance()->getAbstractDrawStates())
    {
      drawAbstractStates(runner);
    }

  if (CoachParam::instance()->getAbstractClassifyMC())
    {
      classifyMC();
    }
  if (CoachParam::instance()->getAbstractConvertMDP())
    {
      convertMCToMDP();
    }
  if (CoachParam::instance()->getAbstractSolveMDP())
    {
      solveMDP(runner);
    }
  if (CoachParam::instance()->getAbstractShowMDPState())
    {
      showMDPState();
    }
  if (CoachParam::instance()->getAbstractAnalyzeQTable())
    {
      analyzeQTable();
    }
  if (CoachParam::instance()->getAbstractCompareStateTraceToModel())
    {
      compareStateTraceToModel();
    }
  if (CoachParam::instance()->getAbstractComputeAvgReward())
    {
      computeAverageReward();
    }
  if (CoachParam::instance()->getAbstractAnalyzeMDP())
    {
      analyzeMDP();
    }
  if (CoachParam::instance()->getAbstractGenerateMDPTraces())
    {
      generateMDPStateTraces();
    }
  if (CoachParam::instance()->getAbstractConvertToBinary())
    {
      convertToBinary();
    }
  if (CoachParam::instance()->getAbstractAnalyzePercentOptimal())
    {
      analyzePercentOpt();
    }
}


//Checks that all inter-module dependencies are satisfied; Called after all
// modules initialized
bool
ModAbstract::dependencyCheck(ModuleRegistry* pReg, OnlineRunner& runner)
{
  pmFeatures = (ModFeatures*)pReg->lookup("Features");
  if (pmFeatures == NULL)
    {
      errorlog << "ModAbstract needs ModFeatures" << ende;
      return false;
    }
  if (CoachParam::instance()->getAbstractAdviceByMDP() ||
      CoachParam::instance()->getAbstractABMDPMultiCount() > 0)
    {
      pmCMRep = (ModCMRep*)pReg->lookup("CMRep");
      if (pmCMRep == NULL)
	{
	  errorlog << "ModAbstract needs ModCMRep" << ende;
	  return false;
	}
      pmCMRep->addCMQ(&mqueue);
    }

  if (CoachParam::instance()->getAbstractABMDPMultiCount() > 0 ||
      CoachParam::instance()->getAbstractStateTrack() ||
      CoachParam::instance()->getAbstractAdviceByMDP() ||
      CoachParam::instance()->getAbstractABMDPTrack())
    initializeStateDescription();

  initializeAdviceByMDP();

  if (CoachParam::instance()->getAbstractABMDPMultiCount() > 0)
    {
      pmGlobalAdapt = (ModGlobalAdapt*)pReg->lookup("GlobalAdapt");
      if (pmGlobalAdapt == NULL)
	{
	  errorlog << "ModAbstract needs ModGlobalAdapt when in multi abmdp" << ende;
	  return false;
	}
      pmGlobalAdapt->addAdaptationUnit(this);
    }
  
  return true;
}

bool
ModAbstract::dependencyCheck(ModuleRegistry* pReg, SingleCallRunner& runner)
{
  return true;
}

bool
ModAbstract::dependencyCheck(ModuleRegistry* pReg, LogfileRunner& runner)
{
  pmFeatures = (ModFeatures*)pReg->lookup("Features");
  if (pmFeatures == NULL)
    {
      errorlog << "ModAbstract needs ModFeatures" << ende;
      return false;
    }
  initializeStateDescription();
  return true;
}

/***********************************************************************/

void
ModAbstract::handlePOTracker(TeamSide my_side, const WorldState& ws)
{
  if (!ppotracker)
    return;

  if (ppotracker->getMySide() == TS_None)
    {
      if (my_side == TS_None)
	{
	  actionlog(150) << "handlePOTracker: no side available" << ende;
	  return;
	}
      ppotracker->setMySide(my_side);
    }

  if (CoachParam::instance()->getAbstractTrackPOOnlyPlayOn() && ws.getPlayMode() != PM_PlayOn)
    {
      actionlog(50) << "handlePOTracker: not play on, doing nothing" << ende;
      return;
    }

  if (CoachParam::instance()->getAbstractTrackPoOnlyKickable() &&
      pmFeatures->getBallPossessors().empty())
    {
      actionlog(50) << "handlePOTracker: no ball possessors, doing nothing" << ende;
      return;
    }
    

  ppotracker->processWorldState(ws);
}

/***********************************************************************/

void
ModAbstract::initializeAbstractStateTrackers(TeamSide my_side)
{
  if (abstract_state_tracker_init)
    return;

  abstract_state_tracker_init = true;

  if (!CoachParam::instance()->getAbstractStateTrack())
    return;

  if (my_side == TS_Left ||
      CoachParam::instance()->getAbstractStateTrackBothTeams())
    {
      std::string fn("");
      if (CoachParam::instance()->getAbstractStateTrackWritePerCycle())
	{
	  fn = CoachParam::instance()->getAbstractStateTrackPerCycleFN();
	  fn = spades::findReplaceInStr(fn, "%S", "left");
	}
      abstract_state_trackers.push_back(new AbstractStateTracker(TS_Left,
								 &state_description,
								 fn.c_str()));
    }
  
  if (my_side == TS_Right ||
      CoachParam::instance()->getAbstractStateTrackBothTeams())
    {
      std::string fn("");
      if (CoachParam::instance()->getAbstractStateTrackWritePerCycle())
	{
	  fn = CoachParam::instance()->getAbstractStateTrackPerCycleFN();
	  fn = spades::findReplaceInStr(fn, "%S", "right");
	}
      abstract_state_trackers.push_back(new AbstractStateTracker(TS_Right,
								 &state_description,
								 fn.c_str()));
    }
}


void
ModAbstract::handleAbstractStateTracker(TeamSide my_side, const WorldState& ws)
{
  for (TrackerStorage::iterator iter = abstract_state_trackers.begin();
       iter != abstract_state_trackers.end();
       iter++)
    {
      (*iter)->processWorldState(ws);
      actionlog(100) << "Abstract state: tracker "
		     << (iter-abstract_state_trackers.begin())
		     << " finished"
		     << ende;
    }
  
}

void
ModAbstract::handleTrackAdvisedActions(TeamSide my_side, const WorldState& ws)
{
  if (!CoachParam::instance()->getAbstractABMDPTrack())
    return;
  
  initAdviceByMDPFilters(CoachParam::instance()->getAbstractABMDPPercOpt(),
                         CoachParam::instance()->getAbstractABMDPChooseWorstActions());

  AbstractState state(&state_description);
  if (!state_description.getStateForWorldState(ws,
					       my_side,
					       &state))
    errorlog << "AdviceByMDP: tracking could not get abstract state" << ende;
  
  if (state.getStateIdx() == -1)
    {
      actionlog(50) << "AdviceByMDP: tracking, state is -1" << ende;
    }
  else
    {
      out_abmdp_track << ws.getTime() << ' ';
      SoccerModelAdviserOutput adviser(soccer_model, out_abmdp_track);
      soccer_model.adviseForState(&state, abmdp_lfilters_act, &adviser);
      actionlog(50) << "AdviceByMDP: tracking, state is "
		    << state.getStateIdx()
		    << ": " << state
		    << ende;
    }
#ifdef SMURF_OLD_CODE
  AbstractState state(&state_description);
  if (!state_description.getStateForWorldState(ws, my_side, &state))
    {
      actionlog(100) << "handleTrackAdvisedActions: the state description rejected this world state, doing nothing" << ende;
      return;
    }
  
  os_track_advised << ws.getTime() << " ";
  SoccerModelAdviserOutput adviser(soccer_model, os_track_advised);
  
  soccer_model.adviseForState(&state, abmdp_lfilters_act, &adviser);
#endif
}



/******************************************************************************/
void
ModAbstract::initializeAdviceByMDP()
{
  if (advice_by_mdp_init)
    return;

  // mark ourself as initialized
  advice_by_mdp_init = true;

  if (CoachParam::instance()->getAbstractABMDPMultiCount() > 0)
    {
      if (CoachParam::instance()->getRunMode() != RM_Online)
        errorlog << "Mutli abmdp: I probably shouldn't be in initializeAdviceByMDP when not online" << ende;
      if (!CoachParam::instance()->getAbstractABMDPUseAdviceTree())
        errorlog << "Multi abmdp: Can't use non tree" << ende;

      queueAdviceByMDPTree();
    }
  else
    {
      if (!CoachParam::instance()->getAbstractAdviceByMDP())
        return;
  
      state_description.getFactor()->createInitialAdvice(mqueue);
      
      if (CoachParam::instance()->getAbstractABMDPUseAdviceTree())
        queueAdviceByMDPTree();
      else
        queueAdviceByMDPFlat();
    }

}

void
ModAbstract::initAdviceByMDPFilters(float percent_optimal, bool choose_worst)
{
  if (advice_by_mdp_init_filters)
    return;
  advice_by_mdp_init_filters = true;

  if (CoachParam::instance()->getAbstractABMDPMultiCount() > 0)
    {
      // Make sure we are in play on mode
      ASDiffPatternElement* ppat_dead_ball = new ASDiffPatternSimple(ASF_DeadBall);
      ASDiffPatternElement* ppat =
        ASDiffPatternChildren::makePattern(ASF_Or, ASF_And, ppat_dead_ball);
      for (MultiABMDPStorage::iterator iter = v_multi_abmdp.begin();
           iter != v_multi_abmdp.end();
           ++iter)
        {
          (*iter)->addStateFilter(new SSFilterNumActions(1, 1000));
          (*iter)->addStateFilter(new SSFilterHasCLangAct());
          if (!ppat->matchTo((*iter)->getStateDescription()))
            errorlog << "ModAbstract::initializeAdviceByMDPFilters: How did my pattern not match?" << ende;
          (*iter)->addStateFilter(new SSFilterFactorHasValue(ppat_dead_ball->getMatchingFactorIdx(), -1));
          (*iter)->addActionFilter(new SAFilterNearOpt(percent_optimal, choose_worst));
        }
      delete ppat; // this deletes ppat_dead_ball for us
    }
  else
    {
  
      abmdp_lfilters_state.push_back(new SSFilterNumActions(1, 1000));
      abmdp_lfilters_state.push_back(new SSFilterHasCLangAct());
      // Make sure we are in play on mode
      ASDiffPatternElement* ppat_dead_ball = new ASDiffPatternSimple(ASF_DeadBall);
      ASDiffPatternElement* ppat =
        ASDiffPatternChildren::makePattern(ASF_Or, ASF_And, ppat_dead_ball);
      if (!ppat->matchTo(&state_description))
        errorlog << "ModAbstract::initializeAdviceByMDPFilters: How did my pattern not match?" << ende;
      abmdp_lfilters_state.push_back(new SSFilterFactorHasValue(ppat_dead_ball->getMatchingFactorIdx(), -1));
      delete ppat; // this deletes ppat_dead_ball for us

      //Now check for our rank and value filters
      StateValue<float>*  psvrank;
      if (!CoachParam::instance()->getAbstractABMDPRankFilterFN().empty())
        {
          std::ifstream in_rank(CoachParam::instance()->getAbstractABMDPRankFilterFN().c_str());
          if (!in_rank)
            errorlog << "AdviceByMDP: Could not open rank StateValue file '"
                     << CoachParam::instance()->getAbstractABMDPRankFilterFN() << "'"
                     << ende;
          psvrank = new StateValue<float>;
          in_rank >> *psvrank;
          if (in_rank.fail())
            errorlog << "AdviceByMDP: Could not read StateValue from rank StateValue file '"
                     << CoachParam::instance()->getAbstractABMDPRankFilterFN() << "'"
                     << ende;
          abmdp_lfilters_state.push_back(new SSFilterByRank<float>(psvrank,
                                                                   CoachParam::instance()->getAbstractABMDPRankFilterCount(),
                                                                   CoachParam::instance()->getAbstractABMDPRankFilterFromTop()));
        }

      // this is hear so that it doesn't get destoryed prematurely
      StateValue<float>* psvvalue = NULL;
      if (!CoachParam::instance()->getAbstractABMDPValueFilterFN().empty())
        {
          std::ifstream in_value(CoachParam::instance()->getAbstractABMDPValueFilterFN().c_str());
          if (!in_value)
            errorlog << "AdviceByMDP: Could not open value StateValue file '"
                     << CoachParam::instance()->getAbstractABMDPValueFilterFN() << "'"
                     << ende;
          in_value >> *psvvalue;
          if (in_value.fail())
            errorlog << "AdviceByMDP: Could not read StateValue from value StateValue file '"
                     << CoachParam::instance()->getAbstractABMDPValueFilterFN() << "'"
                     << ende;
          abmdp_lfilters_state.push_back(new SSFilterByValue<float>(psvvalue,
                                                                    CoachParam::instance()->getAbstractABMDPValueFilterMin(),
                                                                    CoachParam::instance()->getAbstractABMDPValueFilterMax()));
        }
  
  
      abmdp_lfilters_act.push_back(new SAFilterNearOpt(percent_optimal, choose_worst));
    }
  
}

void
ModAbstract::queueAdviceByMDPFlat()
{
  // Now we get to say lots of stuff
  initAdviceByMDPFilters(CoachParam::instance()->getAbstractABMDPPercOpt(),
                         CoachParam::instance()->getAbstractABMDPChooseWorstActions());

  if (CoachParam::instance()->getAbstractABMDPMultiCount() > 0)
    {
      errorlog << "ModAbstract::queueAdviceByMDPFlat: shouldn't be here with multi abmdp"
               << ende;
      return;
    }
  
  SoccerModelAdviser* padviser = NULL;
  if (CoachParam::instance()->getAbstractABMDPSendAdvice())
    // A is for Abstract
    padviser = new SoccerModelAdviserFlat("A", &mqueue);
  else
    padviser = new SoccerModelAdviserNull();
    
  int no_act_count;
  int cnt = soccer_model.adviseFor(abmdp_lfilters_state, abmdp_lfilters_act, padviser, &no_act_count);
				   
  actionlog(50) << "AdviceByMDP: advised about " << cnt << " states " << ende;
  std::cout << "AdviceByMDP: Queued up advice for " << cnt << " states " << std::endl;
  warninglog(10) << "AdviceByMDP: no actions in advice for " << no_act_count << " states" << ende;

  delete padviser;
}

void
ModAbstract::queueAdviceByMDPTree()
{
  initAdviceByMDPTree();
  initAdviceByMDPFilters(CoachParam::instance()->getAbstractABMDPPercOpt(),
                         CoachParam::instance()->getAbstractABMDPChooseWorstActions());

  if (CoachParam::instance()->getAbstractABMDPMultiCount() > 0)
    {
      if (CoachParam::instance()->getRunMode() != RM_Online)
        return;

      for (MultiABMDPStorage::iterator iter = v_multi_abmdp.begin();
           iter != v_multi_abmdp.end();
           ++iter)
        {
          (*iter)->generateAdvice(mqueue);
          (*iter)->setState(mqueue, false);
        }
    }
  else
    {

      if (p_advice_tree == NULL)
        {
          errorlog << "queueAdviceByMDPTree: tree did not init!" << ende;
          return;
        }
  
      // Now we get to say lots of stuff
      SoccerModelAdviser* padviser = NULL;
      if (CoachParam::instance()->getAbstractABMDPSendAdvice())
        padviser = new SoccerModelAdviserTree(p_advice_tree);
      else
        padviser = new SoccerModelAdviserNull();

      int cnt = soccer_model.adviseFor(abmdp_lfilters_state, abmdp_lfilters_act, padviser);
				   
      actionlog(50) << "AdviceByMDP: advised about " << cnt << " states " << ende;
      std::cout << "AdviceByMDP: Queued up advice for " << cnt << " states " << std::endl;

      if (CoachParam::instance()->getAbstractABMDPAdviceTreeShuffle())
        {
          actionlog(50) << "Shuffling the actions in the advice tree" << ende;
          std::cout << "Shuffling the actions in the advice tree" << std::endl;
          p_advice_tree->shuffleActions();
        }

      std::ostream* p_os_tree_assoc = NULL;
      if (CoachParam::instance()->getAbstractABMDPCreateTreeAssoc())
        {
          p_os_tree_assoc = new std::ofstream(CoachParam::instance()->getAbstractABMDPTreeAssocFN().c_str());
          if (!p_os_tree_assoc)
            {
              errorlog << "Could not open file '"
                       << CoachParam::instance()->getAbstractABMDPTreeAssocFN().c_str()
                       << "' for tree association file"
                       << ende;
              delete p_os_tree_assoc;
              p_os_tree_assoc = NULL;
            }
        }
  
      // Now we actually have to send the advice
      // A is for Abstract
      std::string top_rule_name = p_advice_tree->createAdvice(&mqueue, "A", p_os_tree_assoc);
      if (p_os_tree_assoc)
        {
          delete p_os_tree_assoc;
          p_os_tree_assoc = NULL;
        }

      //Now we'll create a nested rule for the advice tree top rul
      if (CoachParam::instance()->getAbstractABMDPSendAdvice() &&
          !CoachParam::instance()->getAbstractABMDPUseBackChannel() &&
          !top_rule_name.empty())
        {
          rcss::clang::CondAnd* pAnd = new rcss::clang::CondAnd;
          pAnd->push_back(std::auto_ptr<rcss::clang::Cond>(new rcss::clang::CondPlayMode(rcss::clang::PlayOn)));
          pAnd->push_back(std::auto_ptr<rcss::clang::Cond>(new rcss::clang::CondBallOwner(true, rcss::clang::UNumSet(rcss::clang::UNum(rcss::clang::UNum::uAll)))));
          rcss::clang::NestedRule* pRule =
            new rcss::clang::NestedRule( std::auto_ptr<rcss::clang::Cond>(pAnd) );
          rcss::clang::RuleIDList idlist;
          idlist.push_back(top_rule_name);
          pRule->getRules().push_back( new rcss::clang::IDListRule(idlist) );

          rcss::clang::DefRule* pDef =
            new rcss::clang::DefRule(ADVICE_TREE_RULE_NAME,
                                     std::auto_ptr<rcss::clang::Rule>(pRule),
                                     false);
          mqueue.getDefineContainer().push(pDef);

          idlist.clear();
          idlist.push_back(ADVICE_TREE_RULE_NAME);
          mqueue.getRuleContainer().push(new rcss::clang::ActivateRules(true, idlist));
        }

      delete padviser;
    }
  
}

void
ModAbstract::initAdviceByMDPTree()
{
  //Don't have to do anything for multi
  if (CoachParam::instance()->getAbstractABMDPMultiCount() <= 0)
    {
      // We'll make a pattern to find the factor we want, then create the tree
      // This is a bit of a hack! we saw that the factor we wnat is the one
      // that has a ball owner element under it
      ASDiffPatternChildren* pOr = new ASDiffPatternChildren(ASF_Or);
      ASDiffPatternChildren* pAnd = new ASDiffPatternChildren(ASF_And);
      pAnd->addChild(new ASDiffPatternSimple(ASF_BallOwner));
      pOr->addChild(pAnd);
      if (!pOr->matchTo(&state_description))
        {
          errorlog << "ModAbstract::initAdvideByMDPTree: How did the pattern not match?"
                   << ende;
          return;
        }

      AbstractStateFactor* p_root_factor = pAnd->getMatchingFactor();

      p_advice_tree = p_root_factor->createAdviceTree(pAnd->getMatchingFactorIdx());
	
      delete pOr;
    }
  
}


void
ModAbstract::handleAdviceByMDP(TeamSide my_side, const WorldHistory& history)
{
  if (!CoachParam::instance()->getAbstractAdviceByMDP())
    return;
  
  //#define TEST_AS_COND
#ifdef TEST_AS_COND
  int sidx = int_random(state_description.getNumStates());
  AbstractState state(&state_description);
  state.setStateIdx(sidx);
  rcss::clang::Cond* pCond = state.createCondition();
  if (pCond)
    {
      std::string rule_name =
	std::string("TEST_") + toString(sidx);
      actionlog(50) << "AdviceByMDP: creating test advice " <<  rule_name
		    << " for: " << state << ende;
      rcss::clang::SimpleRule* pRule =
	new rcss::clang::SimpleRule(std::auto_ptr<rcss::clang::Cond>(pCond));
      rcss::clang::UNumSet uset;
      uset.add(rcss::clang::UNum::uAll);
      rcss::clang::DirComm* pDir = new rcss::clang::DirComm(true, true, uset);
      pDir->add(std::auto_ptr<rcss::clang::Action>(new rcss::clang::ActHold));
      //pRule->getDirs().push_back(std::auto_ptr<rcss::clang::Dir>(pDir));
      pRule->getDirs().push_back(pDir);
      rcss::clang::DefRule* pDef =
	new rcss::clang::DefRule( rule_name,
				  std::auto_ptr<rcss::clang::Rule>(pRule),
				  false );
      mqueue.getDefineContainer().push(pDef);
    }
#endif	

  //#define TEST_AS_COND2
#ifdef TEST_AS_COND2
  std::list<SoccerStateFilter*> lfilters_state;
  std::list<SoccerActionFilter*> lfilters_act;
  lfilters_act.push_back(new SAFilterNearOpt(CoachParam::instance()->getAbstractABMDPPercOpt()));
  lfilters_state.push_back(new SSFilterNumActions(1, 1000));
  lfilters_state.push_back(new SSFilterHasCLangAct());
  lfilters_state.push_back(new SSFilterRand(.00001));
  soccer_model.adviseFor(mqueue, lfilters_state, lfilters_act);
  std::for_each(lfilters_state.begin(), lfilters_state.end(),
		deleteptr<SoccerStateFilter>());
  std::for_each(lfilters_act.begin(), lfilters_act.end(),
		deleteptr<SoccerActionFilter>());
  lfilters_state.clear();
  lfilters_act.clear();
#endif	

  if (CoachParam::instance()->getAbstractABMDPUseBackChannel() &&
      CoachParam::instance()->getAbstractABMDPMultiCount() > 0)
    errorlog << "ModAbstract: I don't support back channel and multi-abmdp" << ende;
    
  if (CoachParam::instance()->getRunMode() != RM_Online)
    return;

  if (CoachParam::instance()->getAbstractABMDPUseBackChannel() &&
      p_advice_tree != NULL)
    {
      AbstractState state(&state_description);
      if (!state_description.getStateForWorldState(history.getWorldState(),
						   my_side,
						   &state))
	errorlog << "AdviceByMDP: back channel could not get abstract state" << ende;

      BinaryFileWriter writer(CoachParam::instance()->getAbstractABMDPBackChannelFN().c_str());
      if (!writer.writeInt(history.getWorldState().getTime()))
	errorlog << "ABMDP: failed to write to back channel time" << ende;
      if (!writer.writeString(p_advice_tree->getRuleForState(state.getStateIdx()).c_str()))
	errorlog << "ABMDP: failed to write to back channel rule" << ende;
    }
}


/******************************************************************************/
int
ModAbstract::getNumStrategiesForStyle(AdaptStyle style)
{
  // Right now, we're just going allow one MDP per style
  return 1;
}

void
ModAbstract::changeStrategy(const StrategySpec& oldspec, const StrategySpec& newspec)
{
  if (oldspec.getAdaptStyle() == newspec.getAdaptStyle())
    return;

  int old_mdp_idx = (oldspec.getAdaptStyle() == AS_Invalid) ? -1 :
    CoachParam::instance()->getAbstractABMDPMultiStratToMDPIdx()[static_cast<int>(oldspec.getAdaptStyle()) - 1];
  int new_mdp_idx = (newspec.getAdaptStyle() == AS_Invalid) ? -1 :
    CoachParam::instance()->getAbstractABMDPMultiStratToMDPIdx()[static_cast<int>(newspec.getAdaptStyle()) - 1];

  // It's the same MDP
  if (old_mdp_idx == new_mdp_idx)
    return;
  
  if (old_mdp_idx != -1)
    v_multi_abmdp[old_mdp_idx]->setState(mqueue, false);
  if (new_mdp_idx != -1)
    v_multi_abmdp[new_mdp_idx]->setState(mqueue, true);
}

/******************************************************************************/

void
ModAbstract::drawPOSet()
{
  PlayerOccupancySet poset;
  POSetFileReader reader(&poset);
  reader.readFile(CoachParam::instance()->getAbstractDrawPOSetInFN().c_str());
  FieldImage fi(1, 8);
  fi.addFieldLines();
  VarBindings bindings;
  FixedRWMI wmi;
  std::ostringstream fn;
  fn << CoachParam::instance()->getAbstractDrawPOSetInFN();
  if (CoachParam::instance()->getAbstractDrawPOSetStateNum() >= 0)
    {
      PlayerOccupancyState state(&poset);
      state.setOverallIdx(CoachParam::instance()->getAbstractDrawPOSetStateNum());
      //state.draw(&fi, true, &wmi, bindings);
      state.draw(&fi, false, &wmi, bindings);
      fn << '.' << CoachParam::instance()->getAbstractDrawPOSetStateNum();
    }
  else
    {
      poset.draw(&fi, true, &wmi, bindings);
      // we won't add anything to the file name; FieldImage added .png
    }
  //remember this appends a .png
  fi.writeTo(fn.str().c_str());
}

void
ModAbstract::learnMarkovChain()
{
  cout << "Learning Markov Chain..." << endl;
  std::ofstream out(CoachParam::instance()->getAbstractLearnMarkovChainOutFN().c_str());
  if (!out)
    errorlog << "Could not open out file '"
	     << CoachParam::instance()->getAbstractLearnMarkovChainOutFN() << "'" << ende;
  MarkovChain chain(CoachParam::instance()->getAbstractLearnMarkovChainStates());
  chain.learnFromTransitions(CoachParam::instance()->getAbstractLearnMarkovChainInFN().c_str(),
			     CoachParam::instance()->getAbstractLearnMarkovChainNewStatesFN().c_str(),
			     CoachParam::instance()->getAbstractLearnMarkovChainNewTranFN().c_str());
  cout << "...Finishing up" << endl;
  out << chain;
  out.close();
}

void
ModAbstract::learnMarkovChainValue(SingleCallRunner& runner)
{
  MarkovChain chain;
  MarkovChainValue value(&chain, CoachParam::instance()->getAbstractLearnMCVDiscountFactor());
      
  std::ifstream is_chain(CoachParam::instance()->getAbstractLearnMCVChainFN().c_str());
  if (!is_chain)
    errorlog << "Could not open chain file '"
	     << CoachParam::instance()->getAbstractLearnMCVChainFN() << "'" << ende;

  is_chain >> chain;
  if (is_chain.fail())
    errorlog << "Failed reading the markov chain" << ende;
  is_chain.close();

  if (!CoachParam::instance()->getAbstractLearnMCVInValueFN().empty())
    {
      std::ifstream is_value(CoachParam::instance()->getAbstractLearnMCVInValueFN().c_str());
      if (!is_value)
	errorlog << "Could not open value file '"
		 << CoachParam::instance()->getAbstractLearnMCVInValueFN() << "'"
		 << ende;
      is_value >> value;
      if (is_value.fail())
	errorlog << "Could not read value" << ende;
      is_value.close();
    }

  value.setChain(&chain);

  std::ofstream of_value(CoachParam::instance()->getAbstractLearnMCVOutValueFN().c_str());
  if (!of_value)
    errorlog << "Could not open out file '"
	     << CoachParam::instance()->getAbstractLearnMCVOutValueFN().c_str()
	     << "'" << ende;

  if (!value.learnValues(CoachParam::instance()->getAbstractLearnMCVProgressInterval(),
			 CoachParam::instance()->getAbstractLearnMCVTotalChangeLim(),
			 CoachParam::instance()->getAbstractLearnMCVPerCapChangeLim(),
			 CoachParam::instance()->getAbstractLearnMCVIterLim(),
			 &runner.shutdown))
    errorlog << "Error in learning values" << ende;

  of_value << value;
}

void
ModAbstract::pruneMarkovChain()
{
  cout << "Pruning a MarkovChain" << endl;
  MarkovChain chain;
      
  std::ifstream is_chain(CoachParam::instance()->getAbstractPruneMCInFN().c_str());
  if (!is_chain)
    errorlog << "Could not open chain file '"
	     << CoachParam::instance()->getAbstractLearnMCVChainFN() << "'" << ende;

  is_chain >> chain;
  if (is_chain.fail())
    errorlog << "Failed reading the markov chain" << ende;
  is_chain.close();

  ofstream of_chain(CoachParam::instance()->getAbstractPruneMCOutFN().c_str());
  if (!of_chain)
    errorlog << "Could not open out file '"
	     << CoachParam::instance()->getAbstractPruneMCOutFN() << "'" << ende;

  // prune some specific transitions
  class TranFileReader
    : public FileReader
  {
  public:
    TranFileReader(MarkovChain& mc) : FileReader(), mc(mc) {}
    bool processLine(std::istrstream& line,
		     const char* fileid,
		     const char* path,
		     float version)
    {
      int state1, state2;
      line >> state1 >> state2;
      if (line.fail())
	return false;
      mc.pruneSpecificTransition(state1, state2);
      return true;
    }

  private:
    MarkovChain& mc;
  };

  if (!CoachParam::instance()->getAbstractPruneMCTranFN().empty())
    {
      cout << "Nuking transitions from file '" << CoachParam::instance()->getAbstractPruneMCTranFN() << "'" << endl;
      TranFileReader fr_prune_tran(chain);
      fr_prune_tran.readFile(CoachParam::instance()->getAbstractPruneMCTranFN().c_str(),
			     std::numeric_limits<float>::max());
    }
  
  // This removes self transistions for a certain state range
  if (CoachParam::instance()->getAbstractPruneMCNukeSelfMin() >= 0 &&
      CoachParam::instance()->getAbstractPruneMCNukeSelfMax() >
      CoachParam::instance()->getAbstractPruneMCNukeSelfMin())
    {
      cout << "Nuking self transitions in range " 
	   << CoachParam::instance()->getAbstractPruneMCNukeSelfMin()
	   << " - " 
	   << CoachParam::instance()->getAbstractPruneMCNukeSelfMax()
	   << endl;
      for (int state = CoachParam::instance()->getAbstractPruneMCNukeSelfMin();
	   state < CoachParam::instance()->getAbstractPruneMCNukeSelfMax();
	   state++)
	{
	  double old_weight = chain.removeTransition(state, state);
	  actionlog(50) << "Pruning self transition on " << state << "; old weight " << old_weight << ende;
	}
    }
  
  // These have to be in this order because pruning states can create dead transitions
  if (CoachParam::instance()->getAbstractPruneMCTargetState() >= 0)
    {
      cout << "Pruning for reachable..." << endl;
      int count =
	chain.pruneNonReverseReachable(CoachParam::instance()->getAbstractPruneMCTargetState());
      cout << "Removed " << count << " states (not reachable)" << endl;
    }
  cout << "Pruning dead transitions..." << endl;
  int count = chain.pruneDeadTransitions();
  cout << "Removed " << count << " transitions (dead transitions)" << endl;
  cout << "Normalizing..." << endl;
  chain.normalize();
      
  of_chain << chain;
}

void
ModAbstract::createMarkovChainGraph()
{
  cout << "Creating graph from MarkovChain" << endl;
  MarkovChain chain;
  MarkovChainValue value(&chain, 0.0);
  MarkovChainValue tree_value(&chain, 0.0);
      
  std::ifstream is_chain(CoachParam::instance()->getAbstractMCGraphChainFN().c_str());
  if (!is_chain)
    errorlog << "Could not open chain file '"
	     << CoachParam::instance()->getAbstractMCGraphChainFN() << "'" << ende;

  is_chain >> chain;
  if (is_chain.fail())
    errorlog << "Failed reading the markov chain" << ende;
  is_chain.close();

  std::ifstream is_value(CoachParam::instance()->getAbstractMCGraphValueFN().c_str());
  if (!is_value)
    errorlog << "Could not open value file '"
	     << CoachParam::instance()->getAbstractMCGraphValueFN() << "'"
	     << ende;
  is_value >> value;
  if (is_value.fail())
    errorlog << "Could not read value" << ende;
  is_value.close();

  std::ifstream is_tree_value(CoachParam::instance()->getAbstractMCGraphTreeValueFN().c_str());
  if (!is_tree_value)
    errorlog << "Could not open treevalue file '"
	     << CoachParam::instance()->getAbstractMCGraphTreeValueFN() << "'"
	     << ende;
  is_tree_value >> tree_value;
  if (is_tree_value.fail())
    errorlog << "Could not read tree_value" << ende;
  is_tree_value.close();
      
  ofstream of_graph(CoachParam::instance()->getAbstractMCGraphOutGraphFN().c_str());
  if (!of_graph)
    errorlog << "Could not open out graph file '"
	     << CoachParam::instance()->getAbstractMCGraphOutGraphFN() << "'" << ende;
  LibSeaGraphWriter writer(of_graph);
      
  value.createGraph(writer,
		    CoachParam::instance()->getAbstractMCGraphName().c_str(),
		    CoachParam::instance()->getAbstractMCGraphRootState(),
		    &tree_value);

}

void
ModAbstract::calcMarkovChainDist()
{
  cout << "Calculating distances in markov chain" << endl;
  MarkovChain chain;
  MarkovChainValue value(NULL, 0.0);
      
  std::ifstream is_chain(CoachParam::instance()->getAbstractMCCalcDistChainFN().c_str());
  if (!is_chain)
    errorlog << "Could not open chain file '"
	     << CoachParam::instance()->getAbstractMCCalcDistChainFN() << "'" << ende;

  is_chain >> chain;
  if (is_chain.fail())
    errorlog << "Failed reading the markov chain" << ende;
  is_chain.close();

  std::ofstream os_value(CoachParam::instance()->getAbstractMCCalcDistValueFN().c_str());
  if (!os_value)
    errorlog << "Could not open value file '"
	     << CoachParam::instance()->getAbstractMCCalcDistValueFN() << "'"
	     << ende;

  value.setChain(&chain);
      
  if (!value.setToDistance(CoachParam::instance()->getAbstractMCCalcDistTargetState()))
    errorlog << "Failed setting distances" << ende;

  os_value << value;
}

void
ModAbstract::drawAbstractStates(SingleCallRunner& runner)
{
  initializeStateDescription();
  //#define TEST_STATE_IDX_CODE
#ifdef TEST_STATE_IDX_CODE
  for (int sidx = 0; sidx < state_description.getNumStates(); sidx++)
    {
      AbstractState state(&state_description);
      state.setStateIdx(sidx);
      int resulting_sidx = state.getStateIdx();
      if (resulting_sidx != sidx)
	errorlog << "testing state idx: failed on " << sidx
		 << ", returned " << resulting_sidx
		 << ": " << state
		 << ende;
      else
	actionlog(50) << "AbstractState testing: success on " << sidx << ": "
		      << state << ende;
    }
#endif	
  if (CoachParam::instance()->getAbstractDrawStateNum() >= 0)
    {
      std::string fn = CoachParam::instance()->getAbstractDrawStateOutFPat();
      fn = findReplaceInStr(fn, "%S",
			    spades::toString(CoachParam::instance()->getAbstractDrawStateNum(), 6).c_str());
      actionlog(30) << "Drawing state: " << CoachParam::instance()->getAbstractDrawStateNum()
		    << " to " << fn
		    << ende;
      drawAbstractState(CoachParam::instance()->getAbstractDrawStateNum(), fn.c_str());
    }
  else
    {
      std::ifstream is(CoachParam::instance()->getAbstractDrawStateCycleFN().c_str());
      if (!is)
	{
	  errorlog << "Could not open cycle file for drawing '"
		   << CoachParam::instance()->getAbstractDrawStateCycleFN() << "'" << ende;
	  return;
	}

      for (;;)
	{
	  if (runner.isShutdownRequested())
	    break;
	  if (!skip_to_non_comment(is))
	    break;
	  int time, stateidx;
	  is >> time;
	  if (is.fail())
	    break;
	  if (!skip_to_character(is, ':'))
	    break;
	  is >> stateidx;
	  if (is.fail())
	    break;
	  std::string fn = CoachParam::instance()->getAbstractDrawStateOutFPat();
	  fn = findReplaceInStr(fn, "%T", spades::toString(time, 5).c_str());
	  fn = findReplaceInStr(fn, "%S", spades::toString(stateidx, 6).c_str());
	  drawAbstractState(stateidx, fn.c_str());
	}
    }
}

void
ModAbstract::drawAbstractState(int state_num, const char* fn)
{
  AbstractState state(&state_description);
  state.setStateIdx(state_num);

  FixedRWMI wmi;
  VecPosition ball_pos;
  if (state.extractBallPos(&ball_pos))
    {
      wmi.setBallPos(ball_pos);
    }
  else
    {
      actionlog(100) << "State did not set the ball pos" << ende;
    }
  FieldImage fi(CoachParam::instance()->getAbstractDrawLegendLines(),
		CoachParam::instance()->getAbstractDrawScale());
  fi.addFieldLines();
  state.draw(&fi, &wmi);
  
  if (!fi.writeTo(fn))
    errorlog << "ModAbstract::drawAbstractState: Could not write FieldImage to '"
	     << fn << "'" << ende;
}

void
ModAbstract::classifyMC()
{
  MarkovChain mc;
  ASDiffClassifierSet cset;

  cout << "Classifying Markov Chain transitions..." << endl;
  cout << "Reading input files..." << endl;
  
  // The the file of classifiers
  ASDCSetFileReader reader(&cset);
  reader.readFile(CoachParam::instance()->getAbstractClassifyMCInASDCSetFN().c_str());

  // Read the MarkovChain
  std::ifstream is_mc(CoachParam::instance()->getAbstractClassifyMCInMCFN().c_str());
  if (!is_mc)
    {
      errorlog << "Could not open markov chain file '"
	       << CoachParam::instance()->getAbstractClassifyMCInMCFN()
	       << "'" << ende;
      return;
    }
  is_mc >> mc;
  if (is_mc.fail())
    {
      errorlog << "Failed reading MarkovChain from '"
	       << CoachParam::instance()->getAbstractClassifyMCInMCFN()
	       << "'" << ende;
      return;
    }

  //Now, try to open the outfiles before doing all the work of classifying
  std::ofstream os_full(CoachParam::instance()->getAbstractClassifyMCOutFullFN().c_str());
  if (!os_full)
    {
      errorlog << "Could not open full out file '"
	       << CoachParam::instance()->getAbstractClassifyMCOutFullFN()
	       << "'" << ende;
      return;
    }
  std::ofstream os_summ(CoachParam::instance()->getAbstractClassifyMCOutSummFN().c_str());
  if (!os_summ)
    {
      errorlog << "Could not open summ out file '"
	       << CoachParam::instance()->getAbstractClassifyMCOutSummFN()
	       << "'" << ende;
      return;
    }

  ASDiffClassificationStore class_store(&cset);

  cout << "Classifying: ";
  initializeStateDescription();
  class_store.classifyMarkovChainTransitions(mc, &state_description, 1000);
  cout << endl;
  
  cout << "Writing output files..." << endl;
  class_store.writeSummary(os_summ);
  class_store.writeFull(os_full);
}

void
ModAbstract::convertMCToMDP()
{
  cout << "Creating MDP from MC..." << endl;

  cout << "Reading input file..." << endl;

  initializeStateDescription();

  MarkovChain mc;
  ASDiffClassifierSet cset;
  TransitionSorter *ptsort = new TransitionSorter(&cset, &state_description);

  // Read the MarkovChain
  std::ifstream is_mc(CoachParam::instance()->getAbstractConvertMDPInMCFN().c_str());
  if (!is_mc)
    {
      errorlog << "Could not open markov chain file '"
	       << CoachParam::instance()->getAbstractConvertMDPInMCFN()
	       << "'" << ende;
      return;
    }
  is_mc >> mc;
  if (is_mc.fail())
    {
      errorlog << "Failed reading MarkovChain from '"
	       << CoachParam::instance()->getAbstractConvertMDPInMCFN()
	       << "'" << ende;
      return;
    }
  is_mc.close();

  // Read the file of classifiers
  ASDCSetFileReader reader(&cset);
  reader.readFile(CoachParam::instance()->getAbstractConvertMDPInASDCSetFN().c_str());

  // Read the transition sorter
  std::ifstream is_tsort(CoachParam::instance()->getAbstractConvertMDPInTSortFN().c_str());
  if (!is_tsort)
    {
      errorlog << "Could not open transition sorter file '"
	       << CoachParam::instance()->getAbstractConvertMDPInTSortFN()
	       << "'" << ende;
      return;
    }
  is_tsort >> *ptsort;
  if (is_tsort.fail())
    {
      errorlog << "Failed reading TransitionSorter from '"
	       << CoachParam::instance()->getAbstractConvertMDPInTSortFN()
	       << "'" << ende;
      return;
    }
  is_tsort.close();

  // Open the out file
  std::ofstream os_mdp(CoachParam::instance()->getAbstractConvertMDPOutMDPFN().c_str());
  if (!os_mdp)
    {
      errorlog << "Could not open summ out file '"
	       << CoachParam::instance()->getAbstractConvertMDPOutMDPFN()
	       << "'" << ende;
      return;
    }

  cout << "Converting: " << flush;

  //converter takes over the ptsort mem
  MDPConverter converter(ptsort);
  if (!converter.convertMC(&mc, 1000))
    errorlog << "Conversion from MC failed!" << ende;

  cout << endl;

  cout << "Writing output files..." << endl;
  
  os_mdp << *converter.getMDP();
}

void
ModAbstract::solveMDP(SingleCallRunner& runner)
{
  cout << "Solving an MDP...." << endl;

  cout << "Reading Input Files...." << endl;
  std::ifstream in_mdp(CoachParam::instance()->getAbstractSolveMDPInMDPFN().c_str());
  if (!in_mdp)
    {
      errorlog << "Could not open MDP file '"
	       << CoachParam::instance()->getAbstractSolveMDPInMDPFN() << "'" << ende;
      return;
    }
  SoccerMDP mdp;
  if (!mdp.readTextOrBinary(in_mdp))
    {
      errorlog << "Failed reading MDP from file '"
	       << CoachParam::instance()->getAbstractSolveMDPInMDPFN() << "'" << ende;
      return;
    }
  in_mdp.close();
  
  std::ofstream out_qt(CoachParam::instance()->getAbstractSolveMDPOutQTFN().c_str());
  if (!out_qt)
    {
      errorlog << "Could not open out QTable file '"
	       << CoachParam::instance()->getAbstractSolveMDPOutQTFN() << "'" << ende;
      return;
    }

  QTableFlex qt(mdp, CoachParam::instance()->getAbstractSolveMDPGamma());
  
  cout << "Solving..." << endl;
  mdp.solveByQTable(qt, 10000, &runner.shutdown);

  cout << "Writing output files..." << endl;
  out_qt << qt;
}

void
ModAbstract::showMDPState()
{
  cout << "Showing MDP state for state "
       << CoachParam::instance()->getAbstractShowMDPStateNum()
       << endl;
  std::ifstream in_mdp(CoachParam::instance()->getAbstractShowMDPStateMDPFN().c_str());
  if (!in_mdp)
    {
      errorlog << "Could not open MDP file '"
	       << CoachParam::instance()->getAbstractShowMDPStateMDPFN() << "'" << ende;
      return;
    }
  SoccerMDP mdp;
  if (!mdp.readTextOrBinary(in_mdp))
    {
      errorlog << "Failed reading MDP from file '"
	       << CoachParam::instance()->getAbstractShowMDPStateMDPFN() << "'" << ende;
      return;
    }
  in_mdp.close();

  cout << "Num Actions: "
       << mdp.getNumActionsInState(CoachParam::instance()->getAbstractShowMDPStateNum())
       << endl;
  
  for (int aidx = 0;
       aidx < mdp.getNumActionsInState(CoachParam::instance()->getAbstractShowMDPStateNum());
       aidx++)
    {
      cout << *mdp.getAction(CoachParam::instance()->getAbstractShowMDPStateNum(), aidx)
	   << endl;
    }
}

void
ModAbstract::analyzeQTable()
{
  std::cout << "Analyzing a QTable" << std::endl;
  
  // Load the QTable
  std::cout << "Loading QTable" << std::endl;
  std::ifstream in_qtable(CoachParam::instance()->getAbstractAnalyzeQTableFN().c_str());
  if (!in_qtable)
    {
      errorlog << "Could not open QTable file '"
	       << CoachParam::instance()->getAbstractAnalyzeQTableFN() << "'" << ende;
      return;
    }
  QTableFlex qtable(0, 0);
  if (!qtable.readTextOrBinary(in_qtable))
    {
      errorlog << "Failed reading QTable from file '"
	       << CoachParam::instance()->getAbstractAnalyzeQTableFN() << "'" << ende;
      return;
    }
  in_qtable.close();

  // Create the StateValue for the max QTable
  std::cout << "Create Max StateValue" << std::endl;
  std::ofstream out_max(CoachParam::instance()->getAbstractAnalyzeQTableOutSvMaxFN().c_str());
  if (!out_max)
    {
      errorlog << "Could not open out SV max file '"
	       << CoachParam::instance()->getAbstractAnalyzeQTableOutSvMaxFN() << "'" << ende;
      return;
    }
  QTable::MaxQValuator max_valuator(&qtable, 1e9);
  StateValue<float>* psvmax = StateValue<float>::create(qtable.getNumStates(), &max_valuator);
  if (psvmax == NULL)
    {
      errorlog << "Failed to create max StateValue" << ende;
    }
  else
    {
      out_max << *psvmax;
      if (out_max.fail())
	errorlog << "Failed writing max StateValue" << ende;
    }
  out_max.close();
  
  // Create the StateValue for the rnage of the Qs
  std::cout << "Create Range StateValue" << std::endl;
  std::ofstream out_range(CoachParam::instance()->getAbstractAnalyzeQTableOutSvRangeFN().c_str());
  if (!out_range)
    {
      errorlog << "Could not open out SV range file '"
	       << CoachParam::instance()->getAbstractAnalyzeQTableOutSvRangeFN() << "'" << ende;
      return;
    }
  QTable::RangeQValuator range_valuator(&qtable, -1e9);
  StateValue<float>* psvrange = StateValue<float>::create(qtable.getNumStates(), &range_valuator);
  if (psvrange == NULL)
    {
      errorlog << "Failed to create range StateValue" << ende;
    }
  else
    {
      out_range << *psvrange;
      if (out_range.fail())
	errorlog << "Failed writing range StateValue" << ende;
    }
  out_range.close();
  
}


void
ModAbstract::compareStateTraceToModel()
{
  initializeStateDescription();
  initAdviceByMDPFilters(CoachParam::instance()->getAbstractABMDPPercOpt(),
                         CoachParam::instance()->getAbstractABMDPChooseWorstActions());
  
  SoccerModelTraceComparator comparator(&soccer_model,
					abmdp_lfilters_state,
					abmdp_lfilters_act);
  StateTraceAnalyzer analyzer(&comparator);

  int count =
    analyzer.processListFile(CoachParam::instance()->getAbstractCompareST2MInListFN().c_str());
  if (count < 0)
    errorlog << "ModAbstract::compareStateTraceToModel: analyzer failed " << count << ende;
  else
    std::cout << "Analyzed " << count << " cycle files" << std::endl;

  comparator.printCounts(std::cout, (double)count);
}

void
ModAbstract::computeAverageReward()
{
  initializeStateDescription();
  initAdviceByMDPFilters(CoachParam::instance()->getAbstractABMDPPercOpt(),
                         CoachParam::instance()->getAbstractABMDPChooseWorstActions());

  SingleDataSummary
    sds(soccer_model.estimateOptAvgStepReward(CoachParam::instance()->getAbstractComputeARStartState(),
					      CoachParam::instance()->getAbstractComputeARNumSteps(),
					      CoachParam::instance()->getAbstractComputeARNumReps(),
					      abmdp_lfilters_act,
					      true));

  std::cout << "Avg rew("
	    << "start=" << CoachParam::instance()->getAbstractComputeARStartState()
	    << ", steps=" << CoachParam::instance()->getAbstractComputeARNumSteps()
	    << ", reps=" << CoachParam::instance()->getAbstractComputeARNumReps()
	    << "): " << sds
	    << std::endl;
}

void
ModAbstract::analyzeMDP()
{
  cout << "Analyzing MDP from '"
       << CoachParam::instance()->getAbstractAnalyzeMDPInFN()
       << "'"
       << endl;
  cout << "Reading..." << std::endl;
  
  std::ifstream in_mdp(CoachParam::instance()->getAbstractAnalyzeMDPInFN().c_str());
  if (!in_mdp)
    {
      errorlog << "Could not open MDP file '"
	       << CoachParam::instance()->getAbstractAnalyzeMDPInFN() << "'" << ende;
      return;
    }
  SoccerMDP mdp;
  if (!mdp.readTextOrBinary(in_mdp))
    {
      errorlog << "Failed reading MDP from file '"
	       << CoachParam::instance()->getAbstractAnalyzeMDPInFN() << "'" << ende;
      return;
    }
  in_mdp.close();

  cout << "Analyzing..." << std::endl;
  // things to show: #states, #act, #tran, mean act/state, mean act/state(>1), mean tran/act, mean tran/act (>1)
  SingleDataSummary sdsAct;
  SingleDataSummary sdsActMin2;
  SingleDataSummary sdsTran;
  SingleDataSummary sdsTranMin2;

  mdp.addActCountData(sdsAct, 1);
  mdp.addActCountData(sdsActMin2, 2);
  mdp.addTranCountData(sdsTran, 1);
  mdp.addTranCountData(sdsTranMin2, 2);

  std::cout << "Num states: " << sdsAct.getNumPoints() << std::endl;
  std::cout << "Total acts: " << sdsAct.getSum() << std::endl;
  std::cout << "Total tran: " << sdsTran.getSum() << std::endl;
  std::cout << "Mean act/state: " << sdsAct.getMean() << std::endl;
  std::cout << "Mean act/state (>1): " << sdsActMin2.getMean() << std::endl;
  std::cout << "Mean tran/act: " << sdsTran.getMean() << std::endl;
  std::cout << "Mean tran/act (>1): " << sdsTranMin2.getMean() << std::endl;

  std::cout << "Now in a format for my gamelist.html:" << std::endl;
  std::cout << "  <td>" << sdsAct.getNumPoints() << "</td>" << std::endl;
  std::cout << "  <td>" << sdsAct.getSum() << "</td>" << std::endl;
  std::cout << "  <td>" << sdsTran.getSum() << "</td>" << std::endl;
  std::cout << "  <td>" << sdsAct.getMean() << "</td>" << std::endl;
  std::cout << "  <td>" << sdsActMin2.getMean() << "</td>" << std::endl;
  std::cout << "  <td>" << sdsTran.getMean() << "</td>" << std::endl;
  std::cout << "  <td>" << sdsTranMin2.getMean() << "</td>" << std::endl;

  
}

void
ModAbstract::generateMDPStateTraces()
{
  std::cout << "Generating MDP traces..." << std::endl;
  
  for (int trace = 0;
       trace < CoachParam::instance()->getAbstractGenerateMDPTracesCount();
       ++trace)
    {
      std::string thisfn(CoachParam::instance()->getAbstractGenerateMDPTracesOutFN());
      thisfn = findReplaceInStr(thisfn, "%I", spades::toString(trace).c_str());

      std::ofstream out_state_file(thisfn.c_str());
      
      if (!out_state_file)
	{
	  errorlog << "ModAbstract::generateMDPStateTraces: could not open out file '"
		   << thisfn << "'"
		   << ende;
	  return;
	}

      out_state_file << "# This file was generated by the generateMDPStateTraces mode of ModAbstract" << std::endl
		     << "# Format: <state_idx>" << std::endl;

      int state_idx = 1;
      for (int step = 0;
	   step < CoachParam::instance()->getAbstractGenerateMDPTracesSteps();
	   ++step)
	{
	  int act = soccer_model.getQTable().getActMaxRandom(state_idx);
	  if (act == -1)
	    {
	      state_idx = soccer_model.getMDP().getRandomValidState();
	    }
	  else
	    {
	      if (CoachParam::instance()->getAbstractGenerateMDPTracesUseMax())
		state_idx =
		  soccer_model.getMDP().takeMaxStep(state_idx, act);
	      else
		state_idx =
		  soccer_model.getMDP().takeStep(state_idx, act);
	    }
	  // we log the state_idx after the transition because we don't really want that
	  // initial state 1
	  out_state_file << state_idx << std::endl;
	}
    }
}

void
ModAbstract::convertToBinary()
{
  std::cout << "Convert MDP and QTable to binary" << std::endl;
  
  if (!CoachParam::instance()->getAbstractConvertMDPInFN().empty())
    {
      std::cout << "Converting MDP..." << std::endl;
      SoccerMDP mdp;
      std::ifstream in(CoachParam::instance()->getAbstractConvertMDPInFN().c_str());
      if (!in)
	errorlog << "ModAbstract::convertToBinary: Failed to open initial qtable file" << ende;

      if (!mdp.readTextOrBinary(in))
	{
	  errorlog << "ModAbstract::convertToBinary: Failed to load initial mdp" << ende;
	}
      else
	{
	  BinaryFileWriter writer(CoachParam::instance()->getAbstractConvertMDPOutFN().c_str());
	  if (!mdp.writeTo(writer))
	    errorlog << "ModAbstract::convertToBinary: Failed writing mdp to bin" << ende;
	}
    }

  if (!CoachParam::instance()->getAbstractConvertQTableInFN().empty())
    {
      std::cout << "Converting QTable..." << std::endl;
      QTableFlex qtable(0,0);
      std::ifstream in(CoachParam::instance()->getAbstractConvertQTableInFN().c_str());
      if (!in)
	errorlog << "ModAbstract::convertToBinary: Failed to open initial qtable file" << ende;

      if (!qtable.readTextOrBinary(in))
	{
	  errorlog << "ModAbstract::convertToBinary: Failed to load initial qtable" << ende;
	}
      else
	{
	  BinaryFileWriter writer(CoachParam::instance()->getAbstractConvertQTableOutFN().c_str());
	  if (!qtable.writeTo(writer))
	    errorlog << "ModAbstract::convertToBinary: Failed writing qtable to bin" << ende;
	}
    }
  std::cout << "Done" << std::endl;
}

void
ModAbstract::analyzePercentOpt()
{
  std::cout << "Analyzing the effect of percent optimal choice on action distribution..." << std::endl;

  initializeStateDescription();

  std::string summ_fn =
    CoachParam::instance()->getAbstractAnalyzePercentOptimalOutDir() + "/act_count_summ.data";
  std::ofstream os_summ(summ_fn.c_str());
  if (!os_summ)
    {
      errorlog << "ModAbstract::analyzePercentOpt: Could not open summary file '" << summ_fn << "'" << ende;
      return;
    }
  os_summ << "# This file was produced by ModAbstract::analyzePercentOpt" << std::endl
          << "# Format: <percent opt> <mean acts per state> <mean acts per state ignoring 0 acts>" << std::endl;
  
  for (std::vector<double>::const_iterator iter = CoachParam::instance()->getAbstractAnalyzePercentOptimalValues().begin();
       iter != CoachParam::instance()->getAbstractAnalyzePercentOptimalValues().end();
       ++iter)
    {
      initAdviceByMDPFilters(*iter, false);

      SoccerModelAdviserCountAct adviser;
      soccer_model.adviseFor(abmdp_lfilters_state, abmdp_lfilters_act, &adviser);

      os_summ << *iter << ' '
              << adviser.getBucket().getMean(0) << ' '
              << adviser.getBucket().getMean(1) << ' '
              << std::endl;
      
      std::string fn =
        CoachParam::instance()->getAbstractAnalyzePercentOptimalOutDir() + "/act_count_" + toString(*iter)+ ".data";
      std::ofstream os(fn.c_str());
      if (!os)
        errorlog << "ModAbstract::analyzePercentOpt: Could not open datafile '" << fn << "'" << ende;

      os << "# This file was produced by ModAbstract::analyzePercentOpt" << std::endl
         << "# Format: <num_acts> <num states with these advised acts" << std::endl
         << adviser.getBucket() << std::endl;
      
      clearABMDPFilters();
    }
  std::cout << "Done" << std::endl;
}

