
#include <ios>
#include <iomanip>
#include "LogfileRunner.h"
#include "CoachCommand.h"
#include "Module.h"
#include "ModuleRegistry.h"
#include "Logger.h"
#include "CoachParam.h"
#include "utility.h"

using namespace spades;
using namespace std;

/*************************************************************************************/
/* misc utility stuff */

class LogfileCaller : std::unary_function<Module*,void>
{
public:
  LogfileCaller(LogfileRunner* p) : p(p) {}
  void operator() (Module* m) { m->stateUpdateLogfileNotify(*p, p->getWorldHistory()); }
private:
  LogfileRunner* p;
};

class LogfileShutdownCaller : std::unary_function<Module*,void>
{
public:
  LogfileShutdownCaller(LogfileRunner* p) : p(p) {}
  void operator() (Module* m) { m->shutdownLogfileNotify(*p, p->getWorldHistory()); }
private:
  LogfileRunner* p;
};


/*************************************************************************************/

LogfileRunner::LogfileRunner(ModuleRegistry* pReg)
  : Runner(pReg),
    history(CoachParam::instance()->getCyclesToStore()),
    player_type_storage(),
    is_logfile(CoachParam::instance()->getLogfileFN().c_str()),
    data_handler(this),
    side_to_analyze(TS_None),
    left_team_name(),
    right_team_name()
{
  if (!is_logfile)
    {
      errorlog << "LogfileRunner: could not open logfile '"
	       << CoachParam::instance()->getLogfileFN() << "'"
	       << ende;
      initiateShutdown();
    }
  
  CoachTagFunction* pTag = dynamic_cast<CoachTagFunction*>(spades::Logger::instance()->getTagFunction());
  if (pTag == NULL)
    {
      errorlog << "LogfileRunner: how is the logger tag function not a CoachTagFunction?" << ende;
    }
  else
    {
      pTag->setTimeTagger(this);
      actionlog(100) << "LogfileRunner: set the logger tag function" << ende;
    }
}

LogfileRunner::~LogfileRunner()
{
  CoachTagFunction* pTag = dynamic_cast<CoachTagFunction*>(spades::Logger::instance()->getTagFunction());
  if (pTag == NULL)
    {
      errorlog << "LogfileRunner: how is the logger tag function not a CoachTagFunction?" << ende;
    }
  else
    {
      pTag->resetTimeTagger();
      actionlog(100) << "LogfileRunner: reset the logger time tag function" << ende;
    }
}

TeamSide
LogfileRunner::getSideOf(const char* team_name)
{
  if (left_team_name.empty())
    left_team_name = team_name;
  if (left_team_name == team_name)
    return TS_Left;
  if (right_team_name.empty())
    right_team_name = team_name;
  if (right_team_name == team_name)
    return TS_Right;
  return TS_None;
}


void
LogfileRunner::prot_initiateShutdown()
{
  //I shouldn't neet to do anything. The loop in run will check whether the shutdown
  // is requested.
}

// this method should call all of the the dependency check methods of the appropriate
// type
bool
LogfileRunner::dependencyCheck(ModuleRegistry* pReg)
{
  DependencyCaller<LogfileRunner> caller(pReg, *this);
  for_each(pReg->getStorage().begin(), pReg->getStorage().end(), caller);
  return getDependencySuccess();
}

// the method to enter the loop and do all the work
bool
LogfileRunner::run()
{
  Time last_progress_time = -1;
  
  actionlog(20) << "LogfileRunner: starting reading" << ende;
  cout << "Analyzing logfile: " << CoachParam::instance()->getLogfileFN() << endl;
  cout << "Processing: " << flush;
  
  rcss::RCGParser parser( data_handler );
  try
    {
      while(!isShutdownRequested() && parser.parse( is_logfile ) )
        {
	  if (data_handler.wasLastMsgShowInfo())
	    {
	      if (history.getNumAvail() > 0 &&
		  history.getWorldState().getTime() % 200 == 1 &&
		  history.getWorldState().getTime() != last_progress_time)
		{
		  cout << '.' << flush;
		  last_progress_time = history.getWorldState().getTime();
		}
	      
	      actionlog(150) << "LogfileRunner: calling modules" << ende;
	      history.finishPending();
	      for_each(pReg->getStorage().begin(), pReg->getStorage().end(), LogfileCaller(this));
	    }
        }
    }
  catch( const std::string& e )
    {
      errorlog << "Logfile parsing: " << e << ende;
    }
  catch( ... )
    {
      errorlog << "Logfile parsing: Unknown error occured" << ende;
    }

  cout << endl;
  
  for_each(pReg->getStorage().begin(), pReg->getStorage().end(), LogfileShutdownCaller(this));
  
  return true;
}

void
LogfileRunner::doSetTeamNames(const char* left, const char* right)
{
  left_team_name = left;
  right_team_name = right;

  side_to_analyze = determineSide(CoachParam::instance()->getLogfileSideToAnalyze().c_str(),
				  CoachParam::instance()->getLogfileTeamToAnalyze().c_str());

  if (side_to_analyze == TS_None)
    {
      errorlog << "I could not determine a side to analyze" << ende;
      initiateShutdown();
    }
  else
    {
      actionlog(100) << "LogfileRunner: Side='" << CoachParam::instance()->getLogfileSideToAnalyze()
		     << "' and Team='" << CoachParam::instance()->getLogfileTeamToAnalyze()
		     << "' says to analyze the " << side_to_analyze << " side: ";
      if (side_to_analyze == TS_Left)
	{
	  actionlog(100) << left;
	}
      else if (side_to_analyze == TS_Right)
	{
	  actionlog(100) << right;
	}
      else if (side_to_analyze == TS_Both)
	{
	  actionlog(100) << "Both";
	  warninglog(10) << "I suspect giving Both to logfile side to analyze will confuse some modules. Avoid it" << ende;
	}
      else
	errorlog << "What is side_to_analyze? " << side_to_analyze << ende;
      actionlog(100) << ende;
    }
}

