/* -*- Mode: C++ -*- */

#include <netinet/in.h>
#include <iomanip>
#include "LogfileDataHandler.h"
#include "LogfileRunner.h"
#include "utility.h"
#include "PlayerParam.h"
#include "ServerParam.h"
#include "Logger.h"

using namespace spades;

/************************************************************************************/
inline
double
convertLogfileLongToDouble(long l)
{
  return ((double)(signed long)ntohl(l))/SHOWINFO_SCALE2;
}

inline
int
convertLogfileShortToInt(short s)
{
  return (int)(signed short)ntohs(s);
}

/************************************************************************************/

LogfileDataHandler::LogfileDataHandler(LogfileRunner* runner)
  : runner(runner), ver(-1), was_last_msg_showinfo(false)
{
}

LogfileDataHandler::~LogfileDataHandler()
{
}


void
LogfileDataHandler::doHandleLogVersion( int ver )
{
  actionlog(150) << "LogfileDataHandler: Got a log version " << ver << ende;
  this->ver = ver;
  was_last_msg_showinfo = false;
}

void
LogfileDataHandler::doHandleDispInfo( std::streampos, 
				      const dispinfo_t& )
{
  errorlog << "How did the dispinfo handler get called?" << ende;
  //actionlog(150) << "LogfileDataHandler: Got a DispInfo" << ende;
  was_last_msg_showinfo = false;
}

void
LogfileDataHandler::doHandleShowInfo( std::streampos,
				      const showinfo_t& )
{
  //actionlog(150) << "LogfileDataHandler: Got a ShowInfo (old)" << ende;
  errorlog << "LogfileDataHandler: I don't support the old showinfo format" << ende;
  was_last_msg_showinfo = true;
}

void
LogfileDataHandler::doHandleShowInfo( std::streampos, 
				      const short_showinfo_t2& showinfo)
{
  actionlog(150) << "LogfileDataHandler: Got a ShowInfo (new)" << ende;

  runner->history.getPendingWorldState().setTime(ntohs(showinfo.time));
  runner->history.getPendingWorldState().setBall(convertBallInfo(showinfo.ball));
  for (int i = 0; i < MAX_PLAYER * 2; i++)
    {
      runner->history.getPendingWorldState().setPlayer(convertPlayerInfo(i, showinfo.pos[i]));
      //SMURF: I should do something with the player type!
    }
  was_last_msg_showinfo = true;
}

void
LogfileDataHandler::doHandleMsgInfo( std::streampos, 
				     short,
				     const std::string& )
{
  actionlog(150) << "LogfileDataHandler: Got a MsgInfo, doing nothing" << ende;
  was_last_msg_showinfo = false;
}

void
LogfileDataHandler::doHandlePlayMode( std::streampos, 
				      char c_mode)
{
  PlayMode mode = (PlayMode)c_mode;
  actionlog(150) << "LogfileDataHandler: Got a PlayMode: " << mode << ende;
  runner->history.getPendingWorldState().setPlayMode(mode);
  was_last_msg_showinfo = false;
}

void
LogfileDataHandler::doHandleTeamInfo( std::streampos,
				      const team_t& left,
				      const team_t& right )
{
  actionlog(150) << "LogfileDataHandler: Got a TeamInfo" << ende;
  runner->doSetTeamNames(left.name, right.name);

  runner->history.getPendingWorldState().setLeftScore(ntohs(left.score));
  runner->history.getPendingWorldState().setRightScore(ntohs(right.score));
  was_last_msg_showinfo = false;
}

void
LogfileDataHandler::doHandleServerParams( std::streampos,
					  const server_params_t& params)
{
  actionlog(150) << "LogfileDataHandler: Got a ServerParams" << ende;

  ServerParam::instance()->setParam("goal_width", convertLogfileLongToDouble(params.gwidth));
  ServerParam::instance()->setParam("inertia_moment", convertLogfileLongToDouble(params.inertia_moment));
  ServerParam::instance()->setParam("player_size", convertLogfileLongToDouble(params.psize));
  ServerParam::instance()->setParam("player_decay", convertLogfileLongToDouble(params.pdecay));
  ServerParam::instance()->setParam("player_rand", convertLogfileLongToDouble(params.prand));
  ServerParam::instance()->setParam("player_weight", convertLogfileLongToDouble(params.pweight));
  ServerParam::instance()->setParam("player_speed_max", convertLogfileLongToDouble(params.pspeed_max));
  ServerParam::instance()->setParam("player_accel_max", convertLogfileLongToDouble(params.paccel_max));
  ServerParam::instance()->setParam("stamina_max", convertLogfileLongToDouble(params.stamina_max));
  ServerParam::instance()->setParam("stamina_inc", convertLogfileLongToDouble(params.stamina_inc));
  ServerParam::instance()->setParam("recover_init", convertLogfileLongToDouble(params.recover_init));
  ServerParam::instance()->setParam("recover_dthr", convertLogfileLongToDouble(params.recover_dthr));
  ServerParam::instance()->setParam("recover_min", convertLogfileLongToDouble(params.recover_min));
  ServerParam::instance()->setParam("recover_dec", convertLogfileLongToDouble(params.recover_dec));
  ServerParam::instance()->setParam("effort_init", convertLogfileLongToDouble(params.effort_init));
  ServerParam::instance()->setParam("effort_dec_thr", convertLogfileLongToDouble(params.effort_dthr));
  ServerParam::instance()->setParam("effort_min", convertLogfileLongToDouble(params.effort_min));
  ServerParam::instance()->setParam("effort_dec", convertLogfileLongToDouble(params.effort_dec));
  ServerParam::instance()->setParam("effort_inc_thr", convertLogfileLongToDouble(params.effort_ithr));
  ServerParam::instance()->setParam("effort_inc", convertLogfileLongToDouble(params.effort_inc));
  ServerParam::instance()->setParam("kick_rand", convertLogfileLongToDouble(params.kick_rand));
  ServerParam::instance()->setParam("team_actuator_noise", convertLogfileShortToInt(params.team_actuator_noise));
    
  ServerParam::instance()->setParam("prand_factor_l", convertLogfileLongToDouble(params.prand_factor_l));
  ServerParam::instance()->setParam("prand_factor_r", convertLogfileLongToDouble(params.prand_factor_r));
  ServerParam::instance()->setParam("kick_rand_factor_l", convertLogfileLongToDouble(params.kick_rand_factor_l));
  ServerParam::instance()->setParam("kick_rand_factor_r", convertLogfileLongToDouble(params.kick_rand_factor_r));

  ServerParam::instance()->setParam("ball_size", convertLogfileLongToDouble(params.bsize));
  ServerParam::instance()->setParam("ball_decay", convertLogfileLongToDouble(params.bdecay));
  ServerParam::instance()->setParam("ball_rand", convertLogfileLongToDouble(params.brand));
  ServerParam::instance()->setParam("ball_weight", convertLogfileLongToDouble(params.bweight));
  ServerParam::instance()->setParam("ball_speed_max", convertLogfileLongToDouble(params.bspeed_max));
  ServerParam::instance()->setParam("ball_accel_max", convertLogfileLongToDouble(params.baccel_max));
  ServerParam::instance()->setParam("dash_power_rate", convertLogfileLongToDouble(params.dprate));
  ServerParam::instance()->setParam("kick_power_rate", convertLogfileLongToDouble(params.kprate));
  ServerParam::instance()->setParam("kickable_margin", convertLogfileLongToDouble(params.kmargin));
  //I don't seem to have these parameters! I don't really know what they are!
  //ServerParam::instance()->setParam("control_radius", convertLogfileLongToDouble(params.ctlradius));
  //ServerParam::instance()->setParam("ctlradius_width", convertLogfileLongToDouble(params.ctlradius_width));
  ServerParam::instance()->setParam("maxpower", convertLogfileLongToDouble(params.maxp));
  ServerParam::instance()->setParam("minpower", convertLogfileLongToDouble(params.minp));
  ServerParam::instance()->setParam("maxmoment", convertLogfileLongToDouble(params.maxm));
  ServerParam::instance()->setParam("minmoment", convertLogfileLongToDouble(params.minm));
  ServerParam::instance()->setParam("maxneckmoment", convertLogfileLongToDouble(params.maxnm));
  ServerParam::instance()->setParam("minneckmoment", convertLogfileLongToDouble(params.minnm));
  ServerParam::instance()->setParam("maxneckang", convertLogfileLongToDouble(params.maxn));
  ServerParam::instance()->setParam("minneckang", convertLogfileLongToDouble(params.minn));
  ServerParam::instance()->setParam("visual_angle", convertLogfileLongToDouble(params.visangle));
  ServerParam::instance()->setParam("visual_distance", convertLogfileLongToDouble(params.visdist));
  ServerParam::instance()->setParam("wind_dir", convertLogfileLongToDouble(params.windir));
  ServerParam::instance()->setParam("wind_force", convertLogfileLongToDouble(params.winforce));
  ServerParam::instance()->setParam("wind_ang", convertLogfileLongToDouble(params.winang));
  ServerParam::instance()->setParam("wind_rand", convertLogfileLongToDouble(params.winrand));
  ServerParam::instance()->setParam("kickable_area", convertLogfileLongToDouble(params.kickable_area));
  ServerParam::instance()->setParam("catch_area_l", convertLogfileLongToDouble(params.catch_area_l));
  ServerParam::instance()->setParam("catch_area_w", convertLogfileLongToDouble(params.catch_area_w));
  ServerParam::instance()->setParam("catch_prob", convertLogfileLongToDouble(params.catch_prob));
  ServerParam::instance()->setParam("goalie_max_moves", convertLogfileShortToInt(params.goalie_max_moves));
    
  ServerParam::instance()->setParam("ckick_margin", convertLogfileLongToDouble(params.ckmargin));
  ServerParam::instance()->setParam("offside_area", convertLogfileLongToDouble(params.offside_area));
  ServerParam::instance()->setParam("wind_none", convertLogfileShortToInt(params.win_no));
  ServerParam::instance()->setParam("wind_random", convertLogfileShortToInt(params.win_random));
  ServerParam::instance()->setParam("say_cnt_max", convertLogfileShortToInt(params.say_cnt_max));
  ServerParam::instance()->setParam("say_coach_msg_size", convertLogfileShortToInt(params.SayCoachMsgSize));
  ServerParam::instance()->setParam("clang_win_size", convertLogfileShortToInt(params.clang_win_size));
  ServerParam::instance()->setParam("clang_define_win", convertLogfileShortToInt(params.clang_define_win));
  ServerParam::instance()->setParam("clang_meta_win", convertLogfileShortToInt(params.clang_meta_win));
  ServerParam::instance()->setParam("clang_advice_win", convertLogfileShortToInt(params.clang_advice_win));
  ServerParam::instance()->setParam("clang_info_win", convertLogfileShortToInt(params.clang_info_win));
  ServerParam::instance()->setParam("clang_mess_delay", convertLogfileShortToInt(params.clang_mess_delay));
  ServerParam::instance()->setParam("clang_mess_per_cycle", convertLogfileShortToInt(params.clang_mess_per_cycle));
  ServerParam::instance()->setParam("half_time", convertLogfileShortToInt(params.half_time));
  ServerParam::instance()->setParam("simulator_step", convertLogfileShortToInt(params.sim_st));
  ServerParam::instance()->setParam("send_step", convertLogfileShortToInt(params.send_st));
  ServerParam::instance()->setParam("recv_step", convertLogfileShortToInt(params.recv_st));
  ServerParam::instance()->setParam("sense_body_step", convertLogfileShortToInt(params.sb_step));
  // I don't have this
  //ServerParam::instance()->setParam("lcm_st", convertLogfileShortToInt(params.lcm_st));
  ServerParam::instance()->setParam("say_msg_size", convertLogfileShortToInt(params.M_say_msg_size));
  ServerParam::instance()->setParam("hear_max", convertLogfileShortToInt(params.M_hear_max));
  ServerParam::instance()->setParam("hear_inc", convertLogfileShortToInt(params.M_hear_inc));
  ServerParam::instance()->setParam("hear_decay", convertLogfileShortToInt(params.M_hear_decay));
  ServerParam::instance()->setParam("catch_ban_cycle", convertLogfileShortToInt(params.cban_cycle));
  ServerParam::instance()->setParam("slow_down_factor", convertLogfileShortToInt(params.slow_down_factor));
  ServerParam::instance()->setParam("use_offside", convertLogfileShortToInt(params.useoffside));
  ServerParam::instance()->setParam("forbid_kick_off_offside", convertLogfileShortToInt(params.kickoffoffside));
    
  ServerParam::instance()->setParam("offside_kick_margin", convertLogfileLongToDouble(params.offside_kick_margin));
  ServerParam::instance()->setParam("audio_cut_dist", convertLogfileLongToDouble(params.audio_dist));
  ServerParam::instance()->setParam("quantize_step", convertLogfileLongToDouble(params.dist_qstep));
  ServerParam::instance()->setParam("quantize_step_l", convertLogfileLongToDouble(params.land_qstep));
  ServerParam::instance()->setParam("quantize_step_dir", convertLogfileLongToDouble(params.dir_qstep));
  ServerParam::instance()->setParam("quantize_step_dist_team_l", convertLogfileLongToDouble(params.dist_qstep_l));
  ServerParam::instance()->setParam("quantize_step_dist_team_r", convertLogfileLongToDouble(params.dist_qstep_r));
  ServerParam::instance()->setParam("quantize_step_dist_l_team_l", convertLogfileLongToDouble(params.land_qstep_l));
  ServerParam::instance()->setParam("quantize_step_dist_l_team_r", convertLogfileLongToDouble(params.land_qstep_r));
  ServerParam::instance()->setParam("quantize_step_dir_team_l", convertLogfileLongToDouble(params.dir_qstep_l));
  ServerParam::instance()->setParam("quantize_step_dir_team_r", convertLogfileLongToDouble(params.dir_qstep_r));
  ServerParam::instance()->setParam("coach", convertLogfileShortToInt(params.CoachMode));
  ServerParam::instance()->setParam("coach_w_referee", convertLogfileShortToInt(params.CwRMode));
  ServerParam::instance()->setParam("old_coach_hear", convertLogfileShortToInt(params.old_hear));
  ServerParam::instance()->setParam("send_vi_step", convertLogfileShortToInt(params.sv_st));

  ServerParam::instance()->setParam("slowness_on_top_for_left_team", convertLogfileLongToDouble(params.slowness_on_top_for_left_team));
  ServerParam::instance()->setParam("slowness_on_top_for_right_team", convertLogfileLongToDouble(params.slowness_on_top_for_right_team));
  ServerParam::instance()->setParam("keepaway_length", convertLogfileLongToDouble(params.ka_length));
  ServerParam::instance()->setParam("keepaway_width", convertLogfileLongToDouble(params.ka_width));

  ServerParam::instance()->setParam("start_goal_l", convertLogfileShortToInt(params.start_goal_l));
  ServerParam::instance()->setParam("start_goal_r", convertLogfileShortToInt(params.start_goal_r));
  ServerParam::instance()->setParam("fullstate_l", convertLogfileShortToInt(params.fullstate_l));
  ServerParam::instance()->setParam("fullstate_r", convertLogfileShortToInt(params.fullstate_r));
  ServerParam::instance()->setParam("drop_time", convertLogfileShortToInt(params.drop_time));
  ServerParam::instance()->setParam("synch_mode", convertLogfileShortToInt(params.synch_mode));
  ServerParam::instance()->setParam("synch_offset", convertLogfileShortToInt(params.synch_offset));
  ServerParam::instance()->setParam("synch_micro_sleep", convertLogfileShortToInt(params.synch_micro_sleep));
  ServerParam::instance()->setParam("point_to_ban", convertLogfileShortToInt(params.point_to_ban));
  ServerParam::instance()->setParam("point_to_duration", convertLogfileShortToInt(params.point_to_duration));

  was_last_msg_showinfo = false;
}

void
LogfileDataHandler::doHandlePlayerParams( std::streampos,
					  const player_params_t& params)
{
  actionlog(150) << "LogfileDataHandler: Got a PlayerParams" << ende;

  PlayerParam::instance()->setParam("player_types", convertLogfileShortToInt(params.player_types));
  PlayerParam::instance()->setParam("subs_max", convertLogfileShortToInt(params.subs_max));
  PlayerParam::instance()->setParam("pt_max", convertLogfileShortToInt(params.pt_max));
  PlayerParam::instance()->setParam("random_seed", convertLogfileShortToInt(params.random_seed));
    
  PlayerParam::instance()->setParam("player_speed_max_delta_min",
				    convertLogfileLongToDouble(params.player_speed_max_delta_min));
  PlayerParam::instance()->setParam("player_speed_max_delta_max",
				    convertLogfileLongToDouble(params.player_speed_max_delta_max));
  PlayerParam::instance()->setParam("stamina_inc_max_delta_factor",
				    convertLogfileLongToDouble(params.stamina_inc_max_delta_factor));
    
  PlayerParam::instance()->setParam("player_decay_delta_min",
				    convertLogfileLongToDouble(params.player_decay_delta_min));
  PlayerParam::instance()->setParam("player_decay_delta_max",
				    convertLogfileLongToDouble(params.player_decay_delta_max));
  PlayerParam::instance()->setParam("inertia_moment_delta_factor",
				    convertLogfileLongToDouble(params.inertia_moment_delta_factor));

  PlayerParam::instance()->setParam("dash_power_rate_delta_min",
				    convertLogfileLongToDouble(params.dash_power_rate_delta_min));
  PlayerParam::instance()->setParam("dash_power_rate_delta_max",
				    convertLogfileLongToDouble(params.dash_power_rate_delta_max));
  PlayerParam::instance()->setParam("player_size_delta_factor",
				    convertLogfileLongToDouble(params.player_size_delta_factor));

  PlayerParam::instance()->setParam("kickable_margin_delta_min",
				    convertLogfileLongToDouble(params.kickable_margin_delta_min));
  PlayerParam::instance()->setParam("kickable_margin_delta_max",
				    convertLogfileLongToDouble(params.kickable_margin_delta_max));
  PlayerParam::instance()->setParam("kick_rand_delta_factor",
				    convertLogfileLongToDouble(params.kick_rand_delta_factor));

  PlayerParam::instance()->setParam("extra_stamina_delta_min",
				    convertLogfileLongToDouble(params.extra_stamina_delta_min));
  PlayerParam::instance()->setParam("extra_stamina_delta_max",
				    convertLogfileLongToDouble(params.extra_stamina_delta_max));
  PlayerParam::instance()->setParam("effort_max_delta_factor",
				    convertLogfileLongToDouble(params.effort_max_delta_factor));
  PlayerParam::instance()->setParam("effort_min_delta_factor",
				    convertLogfileLongToDouble(params.effort_min_delta_factor));

  PlayerParam::instance()->setParam("new_dash_power_rate_delta_min",
				    convertLogfileLongToDouble(params.new_dash_power_rate_delta_min));
  PlayerParam::instance()->setParam("new_dash_power_rate_delta_max",
				    convertLogfileLongToDouble(params.new_dash_power_rate_delta_max));
  PlayerParam::instance()->setParam("new_stamina_inc_max_delta_factor",
				    convertLogfileLongToDouble(params.new_stamina_inc_max_delta_factor));

  //PlayerParam::instance()->printAll(std::cout);
  
  was_last_msg_showinfo = false;
}

void
LogfileDataHandler::doHandlePlayerType( std::streampos,
					const player_type_t& p)
{
  HeteroPlayerType htype( convertLogfileLongToDouble(p.player_speed_max),
			  convertLogfileLongToDouble(p.stamina_inc_max),
			  convertLogfileLongToDouble(p.player_decay),
			  convertLogfileLongToDouble(p.inertia_moment),
			  convertLogfileLongToDouble(p.dash_power_rate),
			  convertLogfileLongToDouble(p.player_size),
			  convertLogfileLongToDouble(p.kickable_margin),
			  convertLogfileLongToDouble(p.kick_rand),
			  convertLogfileLongToDouble(p.extra_stamina),
			  convertLogfileLongToDouble(p.effort_max),
			  convertLogfileLongToDouble(p.effort_min));
  actionlog(150) << "LogfileDataHandler: Got a PlayerType: " << htype << ende;
  short id = ntohs(p.id);
  runner->player_type_storage.addPlayerType(id, htype);
  was_last_msg_showinfo = false;
}

void
LogfileDataHandler::doHandleEOF()
{
  actionlog(150) << "LogfileDataHandler: Got an EOF" << ende;
  was_last_msg_showinfo = false;
}

/************************************************************************************/
/* static stuff */
const
double
LogfileDataHandler::LOGFILE_LONG_2_ERROR = 0.5 / (double)SHOWINFO_SCALE2;

BallInfo
LogfileDataHandler::convertBallInfo(const ball_t& b)
{
  VecPosition error(LOGFILE_LONG_2_ERROR, LOGFILE_LONG_2_ERROR);
  /*
    actionlog(250) << "Ball x (plain): " << std::setbase(16) << b.x
    << "\tBall y (plain): " << b.y
    << ende;
    actionlog(250) << "Ball x (just ntoh): " << ntohl(b.x)
    << "\tBall y (just noth): " << ntohl(b.y)
    << ende;
    actionlog(250) << "Ball x (no /): " << ((double)ntohl(b.x))
    << "\tBall y (no /): " << ((double)ntohl(b.y))
    << ende;
    actionlog(250) << "Ball x: " << ((double)ntohl(b.x))/SHOWINFO_SCALE2 
    << "\tBall y: " << ((double)ntohl(b.y))/SHOWINFO_SCALE2
    << ende;
  */
  return BallInfo(VecPosition(convertLogfileLongToDouble(b.x), convertLogfileLongToDouble(b.y)),
		  error,
		  VecPosition(convertLogfileLongToDouble(b.deltax), convertLogfileLongToDouble(b.deltay)),
		  error);
}

//idx is the index in the logfile struct -- gives the team and number
PlayerInfo
LogfileDataHandler::convertPlayerInfo(int idx, const player_t& p)
{
  VecPosition error(LOGFILE_LONG_2_ERROR, LOGFILE_LONG_2_ERROR);
  VecPosition pos(convertLogfileLongToDouble(p.x), convertLogfileLongToDouble(p.y));
  VecPosition vel(convertLogfileLongToDouble(p.deltax), convertLogfileLongToDouble(p.deltay));
  //these are in radians, we have to convert them to degrees
  ErrBoundValue body_angle(convertLogfileLongToDouble(p.body_angle), LOGFILE_LONG_2_ERROR);
  ErrBoundValue head_angle(convertLogfileLongToDouble(p.head_angle), LOGFILE_LONG_2_ERROR);
  body_angle *= RAD2DEG;
  head_angle *= RAD2DEG;
  PlayerInfo new_player( (idx < MAX_PLAYER) ? TS_Left : TS_Right,
			 (idx % MAX_PLAYER) + 1,
			 pos, error,
			 vel, error,
			 body_angle,
			 head_angle,
			 ntohs(p.mode) & GOALIE);
  return new_player;
}

