/* This class stores the log of a double instead of the double itself.
   This gives high precision when the number is near to 0 */

#ifndef _LOGDOUBLE_H_
#define _LOGDOUBLE_H_

#include <math.h>
#include <stream.h>

#define signpos(x) ((x)>=0.0?true:false)
#define signval(x) ((x)?1.0:-1.0)
#define XOR(a,b) (((a)&&(b))||((!a)&&(!b)))

class LogDouble 
{
 public:
  LogDouble(double x = 0.0) { fromDouble(x); }
  LogDouble(double log_x, bool p) { v = log_x; pos = p; }
  LogDouble(const LogDouble& ld) { v = ld.v; pos = ld.pos; }
  
  double toDouble() const { return signval(pos)*exp(v); }
  void fromDouble(double x)
    { if (x == 1.0) {v=0.0; pos=true;} else {v = log(fabs(x)); pos = signpos(x);} }
  void setToOne() { v = 0.0; pos = true; }
  void setToZero() { v = -HUGE_VAL; pos = true; }
  
  LogDouble operator -() const { return LogDouble(v, !pos); }
  LogDouble operator +(const LogDouble &a) const
    { if (isZero()) return a; if (a.isZero()) return *this;
    return LogDouble(signval(pos)*exp(v) + signval(a.pos)*exp(a.v)); } 
  LogDouble operator -(const LogDouble &a) const 
    { if (isZero()) return -a; if (a.isZero()) return *this;
    return LogDouble(signval(pos)*exp(v) - signval(a.pos)*exp(a.v)); } 
  LogDouble operator *(const LogDouble &a) const
    { return LogDouble(v+a.v, XOR(pos,a.pos)); }
  LogDouble operator /(const LogDouble &a) const
    { return LogDouble(v-a.v, XOR(pos,a.pos)); }
  LogDouble& operator +=(const LogDouble &a)
    { if (isZero()) *this = a; else if (a.isZero()) return *this;
    else fromDouble(signval(pos)*exp(v) + signval(a.pos)*exp(a.v)); return *this; }
  LogDouble& operator -=(const LogDouble &a)
    { if (isZero()) *this = -a; else if (a.isZero()) return *this;
    else fromDouble(signval(pos)*exp(v) - signval(a.pos)*exp(a.v)); return *this; }
  LogDouble& operator *=(const LogDouble &a)
    { v += a.v;  pos = XOR(pos,a.pos); return *this; }
  LogDouble& operator /=(const LogDouble &a)
    { v -= a.v;  pos = XOR(pos,a.pos); return *this; }

  bool operator !=(const LogDouble &a) const { return (v != a.v) || (pos != a.pos); }
  bool operator ==(const LogDouble &a) const { return (v == a.v) && (pos == a.pos); } 
  bool operator <(const LogDouble &a) const;
  bool operator <=(const LogDouble &a) const;
  bool operator >(const LogDouble &a) const;
  bool operator >=(const LogDouble &a) const;

  bool isZero() const { return (isinf(v) == -1); }
  //same return as the library function
  int isInf() const { return (pos?1:-1)*((int)(isinf(v) == 1)); }
  
  friend ostream& operator <<(ostream & os, const LogDouble &a)
    { return os << a.toDouble(); }
  friend istream& operator >>(istream & is, LogDouble &a)
    { double x; is >> x; a.fromDouble(x); return is; }

 private:
  double v;
  bool pos;

} ;


#endif
