/* -*- Mode: C++ -*- */

/* This class provides an interface to writing a LibSea graph file,
   http://www.caida.org/tools/visualization/libsea/
   which in partifcular, the Walrus graph visualization tool uses
   http://www.caida.org/tools/visualization/walrus

   You have to be careful to call the methods in the correct order
   The class doesn't do too much error checking
*/

#ifndef _LIB_SEA_GRAPH_WRITER_H_
#define _LIB_SEA_GRAPH_WRITER_H_

#include <iostream>
#include <vector>

class LibSeaGraphWriter
{
public:
  enum VerboseLevel {
    VL_None,
    VL_TopLevel,
    VL_2ndLevel,
    VL_All,
  };

public:
  static int encodeRGB(char r, char g, char b);
  
public:
  LibSeaGraphWriter(std::ostream& os, VerboseLevel v = VL_2ndLevel)
    : os(os), verbose(v), num_curr_elements() {}
  ~LibSeaGraphWriter() {}

  std::ostream& getStream() { return os; }

  VerboseLevel getVerbose() const { return verbose; }
  void setVerbose(VerboseLevel v) { verbose = v; }
  
  // Most things come in start in end/pairs. The order of methods here is
  // the order in which they can be called. Many of the ones inside start/end
  // pairs can be iterated

  void startGraph();

  void writeMetaData(const char* name, const char* description,
		     int num_nodes, int num_links,
		     int num_paths, int num_path_links);

  void startStructuralData();
  void startLinks();
  // returns the link ID
  int addLink(int source, int dest);
  void endLinks();
  void startPaths();
  //I have no real support for writing paths
  void endPaths();
  void endStructuralData();

  void startAttributeData();
  void writeEnumerations(); // I don't actually support these, so this is a placeholder
  void startAttributeDefs();

  // returns the ID of this attr
  int startAttribute(const char* name, const char* type, const char* def);
  void startNodeValues();
  void endNodeValues();
  void startLinkValues();
  void endLinkValues();
  void startPathValues();
  void endPathValues();

  // these methods can be used with NodeValues, LinkValues, PathValues
  // return the attr value id
  int addAttrValue(int id, const char* val);
  int addAttrValue(int id, double d);
  int addAttrValue(int id, float f);
  int addAttrValue(int id, int d);

  void endAttribute();

  void endAttributeDefs();

  // I don't support the full qualifier interface, but just spanning trees
  void startQualifiers();
  int writeSpanningTree(const char* name, const char* description,
			int root_attr_idx, int tree_link_attr_idx);
  void endQualifiers();
  
  void endAttributeData();

  // just writes the appropriate empty sections
  void writeVisualizationHints();
  // just writes the appropriate empty sections
  void writeInterfaceHints();

  void endGraph();
  
private:
  void startList(VerboseLevel vlevel, const char* label);
  int listElem();
  void endList();
  
  int& getTopCurrElements() { return num_curr_elements[num_curr_elements.size() - 1]; }

  std::ostream& os;

  VerboseLevel verbose;

  //used to count how many elements, The back is the most recent
  typedef std::vector<int> NumCurrElements;
  NumCurrElements num_curr_elements; 
};


#endif
