/* -*- Mode: C++ -*- */

/* This class provides an interface to writing a LibSea graph file,
   http://www.caida.org/tools/visualization/libsea/
   which in partifcular, the Walrus graph visualization tool uses
   http://www.caida.org/tools/visualization/walrus

   You have to be careful to call the methods in the correct order
   The class doesn't do too much error checking
*/

#include <sstream>
#include "LibSeaGraphWriter.h"

using namespace std;

int
LibSeaGraphWriter::encodeRGB(char r, char g, char b)
{
  unsigned int int_r = (unsigned char)r;
  unsigned int int_g = (unsigned char)g;
  unsigned int int_b = (unsigned char)b;
  
  return (int_r << 16) + (int_g << 16) + (int_b << 16);
}

void
LibSeaGraphWriter::startGraph()
{
  os << "Graph" << endl
     << "{" << endl;
}


void
LibSeaGraphWriter::writeMetaData(const char* name, const char* description,
				 int num_nodes, int num_links,
				 int num_paths, int num_path_links)
{
  os << "### metadata ###" << endl;
  os << ((verbose >= VL_TopLevel) ? "@name=" : "") << '"' << name << "\";" << endl;
  os << ((verbose >= VL_TopLevel) ? "@description=" : "") << '"' << description << "\";" << endl;
  os << ((verbose >= VL_TopLevel) ? "@numNodes=" : "") << num_nodes << ";" << endl;
  os << ((verbose >= VL_TopLevel) ? "@numLinks=" : "") << num_links << ";" << endl;
  os << ((verbose >= VL_TopLevel) ? "@numPaths=" : "") << num_paths << ";" << endl;
  os << ((verbose >= VL_TopLevel) ? "@numPathLinks=" : "") << num_path_links << ";" << endl;
}


void
LibSeaGraphWriter::startStructuralData()
{
  os << endl << "### structural data ###" << endl;
}


void
LibSeaGraphWriter::startLinks()
{
  startList(VL_TopLevel, "links");
}

// returns the link ID
int
LibSeaGraphWriter::addLink(int source, int dest)
{
  int idx = listElem();
  os << "\t{"
     << ((verbose >= VL_All) ? "@source=" : "") << source << ';'
     << ((verbose >= VL_All) ? "@dest=" : "") << dest << ';'
     << "}";
  return idx;
}

void
LibSeaGraphWriter::endLinks()
{
  endList();
}


void
LibSeaGraphWriter::startPaths()
{
  startList(VL_TopLevel, "paths");
}

void
LibSeaGraphWriter::endPaths()
{
  endList();
}

  
void
LibSeaGraphWriter::endStructuralData()
{
  //do nothing
}

void
LibSeaGraphWriter::startAttributeData()
{
  os << endl << "### attribute data ###" << endl;
}

void
LibSeaGraphWriter::writeEnumerations()
{
  os << "@enumerations=;" << endl;
}

  ; // I don't actually support these, so this is a placeholder
void
LibSeaGraphWriter::startAttributeDefs()
{
  startList(VL_TopLevel, "attributeDefinitions");
}


int
LibSeaGraphWriter::startAttribute(const char* name, const char* type, const char* def)
{
  int idx = listElem();
  os << "\t{" << endl
     << "\t\t" << ((verbose >= VL_2ndLevel) ? "@name=" : "") << name << ';' << endl
     << "\t\t" << ((verbose >= VL_2ndLevel) ? "@type=" : "") << type << ';' << endl
     << "\t\t" << ((verbose >= VL_2ndLevel) ? "@default=" : "") << def << ';' << endl;
  return idx;
}

void
LibSeaGraphWriter::startNodeValues()
{
  startList(VL_2ndLevel, "nodeValues");
}

void
LibSeaGraphWriter::endNodeValues()
{
  endList();
}

void
LibSeaGraphWriter::startLinkValues()
{
  startList(VL_2ndLevel, "linkValues");
}

void
LibSeaGraphWriter::endLinkValues()
{
  endList();
}

void
LibSeaGraphWriter::startPathValues()
{
  startList(VL_2ndLevel, "pathValues");
}

void
LibSeaGraphWriter::endPathValues()
{
  endList();
}

// this method can be used with NodeValues, LinkValues, PathValues
int
LibSeaGraphWriter::addAttrValue(int id, const char* val)
{
  int idx = listElem();
  os << "\t{"
     << ((verbose >= VL_All) ? "@id=" : "") << id << ';'
     << ((verbose >= VL_All) ? "@value=" : "") << val << ';'
     << "}";
  return idx;
}
int
LibSeaGraphWriter::addAttrValue(int id, double d)
{
  std::ostringstream str;
  str.setf(std::ios::showpoint);
  str << d;
  return addAttrValue(id, str.str().c_str());
}
int
LibSeaGraphWriter::addAttrValue(int id, float f)
{
  std::ostringstream str;
  str.setf(std::ios::showpoint);
  str << f << 'f';
  return addAttrValue(id, str.str().c_str());
}
int
LibSeaGraphWriter::addAttrValue(int id, int d)
{
  std::ostringstream str;
  str << d;
  return addAttrValue(id, str.str().c_str());
}


void
LibSeaGraphWriter::endAttribute()
{
  os << endl << "\t}";
}

void
LibSeaGraphWriter::endAttributeDefs()
{
  endList();
}

void
LibSeaGraphWriter::startQualifiers()
{
  startList(VL_TopLevel, "qualifiers");
}

int
LibSeaGraphWriter::writeSpanningTree(const char* name, const char* description,
				     int root_attr_idx, int tree_link_attr_idx)
{
  int idx = listElem();
  os << "\t{" << endl
     << "\t\t" << ((verbose >= VL_2ndLevel) ? "@type=" : "") << "$spanning_tree;" << endl
     << "\t\t" << ((verbose >= VL_2ndLevel) ? "@name=" : "") << "$" << name << ";" << endl
     << "\t\t" << ((verbose >= VL_2ndLevel) ? "@description=" : "") << '"' << description << "\";" << endl
     << "\t\t" << ((verbose >= VL_2ndLevel) ? "@attributes=" : "") << "[" << endl
     << "\t\t\t" << "{ "
     << ((verbose >= VL_All) ? "@attribute=" : "") << root_attr_idx << "; "
     << ((verbose >= VL_All) ? "@alias=" : "") << "$root; "
     << "}," << endl
     << "\t\t\t" << "{ "
     << ((verbose >= VL_All) ? "@attribute=" : "") << tree_link_attr_idx << "; "
     << ((verbose >= VL_All) ? "@alias=" : "") << "$tree_link; "
     << "}" << endl
     << "\t\t];" << endl
     << "\t}" << endl;
  return idx;
}

void
LibSeaGraphWriter::endQualifiers()
{
  endList();
}


void
LibSeaGraphWriter::endAttributeData()
{
  // do nothing
}

// just writes the appropriate empty sections
void
LibSeaGraphWriter::writeVisualizationHints()
{
  os << endl;
  if (verbose >= VL_TopLevel)
    {
      os << "   ### visualization hints ###" << endl;
      os << "   @filters=;" << endl;
      os << "   @selectors=;" << endl;
      os << "   @displays=;" << endl;
      os << "   @presentations=;" << endl;
    }
  else
    {
      os << "   ### visualization hints ###" << endl;
      os << "   ;" << endl;
      os << "   ;" << endl;
      os << "   ;" << endl;
      os << "   ;" << endl;
    }
}

// just writes the appropriate empty sections
void
LibSeaGraphWriter::writeInterfaceHints()
{
  os << endl;
  if (verbose >= VL_TopLevel)
    {
      os << "   ### interface hints ###" << endl;
      os << "   @presentationMenus=;" << endl;
      os << "   @displayMenus=;" << endl;
      os << "   @selectorMenus=;" << endl;
      os << "   @filterMenus=;" << endl;
      os << "   @attributeMenus=;" << endl;
    }
  else
    {
      os << "   ### interface hints ###" << endl;
      os << "   ;" << endl;
      os << "   ;" << endl;
      os << "   ;" << endl;
      os << "   ;" << endl;
      os << "   ;" << endl;
    }
}


  
void
LibSeaGraphWriter::endGraph()
{
  os << endl << endl << "}" << endl;
}


void
LibSeaGraphWriter::startList(VerboseLevel vlevel, const char* label)
{
  num_curr_elements.push_back(0);
  if (verbose >= vlevel)
    os << "@" << label << "=";
}

int
LibSeaGraphWriter::listElem()
{
  os << ((getTopCurrElements() == 0) ? '[' : ',');
  os << endl;
  getTopCurrElements()++;
  return getTopCurrElements() - 1;
}

void
LibSeaGraphWriter::endList()
{
  if (getTopCurrElements() > 0)
    os << endl << "]";
  num_curr_elements.pop_back();
  os << ';' << endl;
}
