/* -*- Mode: C++ -*- */

#ifndef _GAPREDICATEADAPTOR_H_
#define _GAPREDICATEADAPTOR_H_

#include "WorldHistory.h"
#include "ModGlobalAdapt.h"
#include "utility.h"

/* This class provides a simpler interface to what is provided by ModGlobalAdapt
   and associated storage to be used by objects that chose the next Strategy */

class GAPredicateAdaptor
{
public:
  // These are helper classes to pull about the "best" strategy that
  // saisfies some condition
  class StrategySpecFilter
  {
  public:
    StrategySpecFilter(GAPredicateAdaptor* ppred)
      : ppred(ppred), reverse_accept(false) {}
    virtual ~StrategySpecFilter() {}

    bool accept(const StrategySpec& spec);
    
    GAPredicateAdaptor* getPred() { return ppred; }

    void setReverseAccept(bool b = true) { reverse_accept = b; }
    
  protected:    
    virtual bool protAccept(const StrategySpec& spec) = 0;
    
  private:
    GAPredicateAdaptor* ppred;
    bool reverse_accept;
  };

  class StrategySpecOrder
  {
  public:
    StrategySpecOrder(GAPredicateAdaptor* ppred)
      : ppred(ppred), reverse_compare(false) {}
    virtual ~StrategySpecOrder() {}

    int compare(const StrategySpec& spec1, const StrategySpec& spec2);
    
    GAPredicateAdaptor* getPred() { return ppred; }

    void setReverseCompare(bool b = true) { reverse_compare = b; }
    
  protected:    
    // -1 if spec1 < spec 2, 0 if ==, 1 if > 
    virtual int protCompare(const StrategySpec& spec1, const StrategySpec& spec2) = 0;
    
  private:
    GAPredicateAdaptor* ppred;
    bool reverse_compare;
  };

public:
  GAPredicateAdaptor(const WorldHistory& history, ModGlobalAdapt* pMGA)
    : history(history), pMGA(pMGA) {}
  ~GAPredicateAdaptor() {}

  const StrategySpec& getCurrentStrategy() { return pMGA->current_strategy.getSpec(); }
  int timeForCurrentStrategy() { return pMGA->getTimeForCurrentStrategySpec(history.getWorldState()); }
  StrategySpec getDefaultStrategy() ;
  int timeForCurrentStrategyIsSmall() { return timeForCurrentStrategy() < 400; }
  
  const FlatStrategySpecSpace& getFlatStrategySpace() { return pMGA->spec_space; }
  int getFlatIdxOf(const StrategySpec& spec) { return pMGA->getFlatIdxOf(spec); }

  //stuff about past performance
  bool haveDataFor(const StrategySpec& spec);
  bool hasGoodDataAmt(const StrategySpec& spec);
  float meanScoreDiffPerCycle(const StrategySpec& spec);

  float meanBallX(const StrategySpec& spec);
  float possPercentage(const StrategySpec& spec, RelativeTeamSide rts);

  // it could be the case that neither of these is true
  bool looksBad(const StrategySpec& spec);
  bool looksGood(const StrategySpec& spec);
  
  float globalMeanScoreDiffPerCycle();
  float globalMeanMyGoalsPerCycle();
  float globalMeanTheirGoalsPerCycle();
  float globalMeanBallX();
  float globalPossPercentage(RelativeTeamSide rts);
  int globalHistoryCycles();

  bool haveDataForAll(const AdaptStyle& as);
  bool haveGoodDataAmtForAll(const AdaptStyle& as);
  
  StrategySpec getStrategyWBestScoreDiff();
  StrategySpec getStrategyWDataAndBestScoreDiff();
  StrategySpec getStrategyWBestMeanX();
  StrategySpec getStrategyWDataAndBestMeanX();
  StrategySpec getStrategyWBestPossPerc();
  StrategySpec getStrategyWDataAndBestPossPerc();
  StrategySpec getScorelessStrategyWData();
  StrategySpec getMoreDefThatNeedsData(const StrategySpec& spec)
  { return getCmpToThatNeedsData(CT_Less, spec, false); }
  StrategySpec getMoreOffThatNeedsData(const StrategySpec& spec)
  { return getCmpToThatNeedsData(CT_Greater, spec, true); }
  StrategySpec getMoreDef(const StrategySpec& spec)
  { return getCmpToElseEqual(CT_Less, spec, false); }
  StrategySpec getMoreOff(const StrategySpec& spec)
  { return getCmpToElseEqual(CT_Greater, spec, true); }
  
  TeamSide getMySide() { return pMGA->my_side; }

  //game state stuff
  int getScoreDiff();
  int getMyGoals();
  int getTheirGoals();
  bool winning() { return getScoreDiff() > 0; }
  bool losing() { return getScoreDiff() < 0; }
  bool tied() { return getScoreDiff() == 0; }

  int getTime() { return history.getWorldState().getTime(); }
  int getCyclesLeft();
  // returns 1,2,3,4
  int getQuarter();

  void printDebugTo(std::ostream& os);
  
protected:
  const ModGlobalAdapt::EpisodeSummaries& getEpisodeSummaries()
  { return pMGA->episode_summaries; }
  
  StrategySpec getCmpToThatNeedsData(const CmpType& ct, const StrategySpec& spec, bool reverse_ss_order);
  StrategySpec getCmpToElseEqual(const CmpType& ct, const StrategySpec& spec, bool reverse_ss_order);

  // return.getAdaptStyle() will be AS_Invalid if nothing matches
  StrategySpec getBestStrategy(std::vector<StrategySpecFilter*>& vfilter, StrategySpecOrder* order);
  StrategySpec getBestStrategy(StrategySpecOrder* order);
  StrategySpec getBestStrategy(StrategySpecFilter* filter1, StrategySpecOrder* order);
  StrategySpec getBestStrategy(StrategySpecFilter* filter1, StrategySpecFilter* filter2, StrategySpecOrder* order);

  const WorldHistory& history;
  ModGlobalAdapt* pMGA;

  friend class StrategySpecFilter;
  friend class StrategySpecOrder;
  friend class SSF_Const;
  friend class SSF_HaveData;
  friend class SSF_GoodAmtData;
  friend class SSF_CmpTo;
  friend class SSF_StyleIs;
  friend class SSF_Scoreless;
  friend class SSO_BestSD;
  friend class SSO_Const;
  friend class SSO_StrategySpecOrder;
};



#endif
