/* -*- Mode: C++ -*- */

#include "GAPredicateAdaptor.h"
#include "ServerParam.h"
#include "CoachParam.h"
#include "utility.h"
#include "Logger.h"

using namespace spades;

/***********************************************************************************/
/* Filter classes */
class SSF_Const
  : public GAPredicateAdaptor::StrategySpecFilter
{
public:
  SSF_Const(GAPredicateAdaptor* ppred, bool const_val)
    : StrategySpecFilter(ppred), const_val(const_val) {}

  bool protAccept(const StrategySpec& spec) { return const_val; }

private:
  bool const_val;
};

class SSF_HaveData
  : public GAPredicateAdaptor::StrategySpecFilter
{
public:
  SSF_HaveData(GAPredicateAdaptor* ppred) : StrategySpecFilter(ppred) {}

  bool protAccept(const StrategySpec& spec)
  { return getPred()->getEpisodeSummaries().haveDataFor(spec); }
};

class SSF_GoodAmtData
  : public GAPredicateAdaptor::StrategySpecFilter
{
public:
  SSF_GoodAmtData(GAPredicateAdaptor* ppred) : StrategySpecFilter(ppred) {}

  bool protAccept(const StrategySpec& spec)
  { return getPred()->hasGoodDataAmt(spec); }
};

class SSF_CmpTo
  : public GAPredicateAdaptor::StrategySpecFilter
{
public:
  SSF_CmpTo(GAPredicateAdaptor* ppred, const CmpType& ct, const StrategySpec& base_val)
    : StrategySpecFilter(ppred), ct(ct), base_val(base_val) {}

  bool protAccept(const StrategySpec& spec)
  { return cmpByCmpType(spec, base_val, ct); }

private:
  CmpType ct;
  StrategySpec base_val;
};

class SSF_StyleIs
  : public GAPredicateAdaptor::StrategySpecFilter
{
public:
  SSF_StyleIs(GAPredicateAdaptor* ppred, const AdaptStyle& as)
    : StrategySpecFilter(ppred), as(as) {}

  bool protAccept(const StrategySpec& spec)
  { return spec.getAdaptStyle() == as; }

private:
  AdaptStyle as;
};

class SSF_Scoreless
  : public GAPredicateAdaptor::StrategySpecFilter
{
public:
  SSF_Scoreless(GAPredicateAdaptor* ppred) : StrategySpecFilter(ppred) {}

  bool protAccept(const StrategySpec& spec)
  { return getPred()->haveDataFor(spec) &&
      !getPred()->getEpisodeSummaries().getDataFor(spec).hasGoals(); }
};
      

/***********************************************************************************/
/* Order classes */

class SSO_BestSD
  : public GAPredicateAdaptor::StrategySpecOrder
{
public:
  SSO_BestSD(GAPredicateAdaptor* ppred) : StrategySpecOrder(ppred) {}

  int protCompare(const StrategySpec& spec1, const StrategySpec& spec2)
  { return cmp3way(getPred()->meanScoreDiffPerCycle(spec1),
                   getPred()->meanScoreDiffPerCycle(spec2));
  }
};

class SSO_BestMeanX
  : public GAPredicateAdaptor::StrategySpecOrder
{
public:
  SSO_BestMeanX(GAPredicateAdaptor* ppred) : StrategySpecOrder(ppred) {}

  int protCompare(const StrategySpec& spec1, const StrategySpec& spec2)
  { return cmp3way(getPred()->meanBallX(spec1),
                   getPred()->meanBallX(spec2));
  }
};

class SSO_BestPossPerc
  : public GAPredicateAdaptor::StrategySpecOrder
{
public:
  SSO_BestPossPerc(GAPredicateAdaptor* ppred) : StrategySpecOrder(ppred) {}

  int protCompare(const StrategySpec& spec1, const StrategySpec& spec2)
  { return cmp3way(getPred()->possPercentage(spec1, RTS_Mine),
                   getPred()->possPercentage(spec2, RTS_Mine));
  }
};

class SSO_Const
  : public GAPredicateAdaptor::StrategySpecOrder
{
public:
  SSO_Const(GAPredicateAdaptor* ppred, int const_val)
    : StrategySpecOrder(ppred), const_val(const_val) {}

  int protCompare(const StrategySpec& spec1, const StrategySpec& spec2)
  { return const_val; }
    
private:
  int const_val;
};

class SSO_StrategySpecOrder
  : public GAPredicateAdaptor::StrategySpecOrder
{
public:
  SSO_StrategySpecOrder(GAPredicateAdaptor* ppred)
    : StrategySpecOrder(ppred) {}

  int protCompare(const StrategySpec& spec1, const StrategySpec& spec2)
  { return cmp3way(spec1, spec2); }
};

/***********************************************************************************/

bool
GAPredicateAdaptor::StrategySpecFilter::accept(const StrategySpec& spec)
{
  return reverse_accept ? !protAccept(spec) : protAccept(spec);
}

int
GAPredicateAdaptor::StrategySpecOrder::compare(const StrategySpec& spec1,
					       const StrategySpec& spec2)
{
  return reverse_compare ? (-1 * protCompare(spec1, spec2)) : protCompare(spec1, spec2);
}

/***********************************************************************************/

StrategySpec
GAPredicateAdaptor::getDefaultStrategy()
{
  return StrategySpec(getStyleFor(CoachParam::instance()->getGlobalAdaptInitialStrategyStyle().c_str()),
                      CoachParam::instance()->getGlobalAdaptInitialStrategyIndex());
}

bool
GAPredicateAdaptor::haveDataFor(const StrategySpec& spec)
{
  if (!pMGA->episode_summaries.haveDataFor(spec))
    return false;
  return pMGA->episode_summaries.getDataFor(spec).getCycles() > 1;
}

bool
GAPredicateAdaptor::hasGoodDataAmt(const StrategySpec& spec)
{
  if (!pMGA->episode_summaries.haveDataFor(spec))
    return false;
  return pMGA->episode_summaries.getDataFor(spec).hasGoodDataAmt();
}

float
GAPredicateAdaptor::meanScoreDiffPerCycle(const StrategySpec& spec)
{
  if (!pMGA->episode_summaries.haveDataFor(spec))
    return 0;
  return pMGA->episode_summaries.getDataFor(spec).meanScoreDiffPerCycle();
}

float
GAPredicateAdaptor::meanBallX(const StrategySpec& spec)
{
  if (!haveDataFor(spec))
    {
      errorlog << "GAPredicateAdaptor::meanBallX: don't have data for "
               << spec << ende;
      return 0;
    }
  return pMGA->episode_summaries.getDataFor(spec).getSDSBallX().getMean();
}

float
GAPredicateAdaptor::possPercentage(const StrategySpec& spec, RelativeTeamSide rts)
{
  if (!haveDataFor(spec))
    {
      errorlog << "GAPredicateAdaptor::possPercentage: don't have data for "
               << spec << " " << rts << ende;
      return 0;
    }
  return pMGA->episode_summaries.getDataFor(spec).getPossPercentage(rts);
}

bool
GAPredicateAdaptor::looksBad(const StrategySpec& spec)
{
  float ballX = meanBallX(spec);
  float possPerc = possPercentage(spec, RTS_Mine);
  return ballX < 15 || possPerc < 0.40;
}

bool
GAPredicateAdaptor::looksGood(const StrategySpec& spec)
{
  float ballX = meanBallX(spec);
  float possPerc = possPercentage(spec, RTS_Mine);
  return ballX > 15 || (ballX > 0 && possPerc > 0.60);
}

float
GAPredicateAdaptor::globalMeanScoreDiffPerCycle()
{
  return pMGA->episode_summaries.getGlobalSummaryEpisode().meanScoreDiffPerCycle();
}

float
GAPredicateAdaptor::globalMeanMyGoalsPerCycle()
{
  return pMGA->episode_summaries.getGlobalSummaryEpisode().meanMyGoalsPerCycle();
}

float
GAPredicateAdaptor::globalMeanTheirGoalsPerCycle()
{
  return pMGA->episode_summaries.getGlobalSummaryEpisode().meanTheirGoalsPerCycle();
}

float
GAPredicateAdaptor::globalMeanBallX()
{
  return pMGA->episode_summaries.getGlobalSummaryEpisode().meanTheirGoalsPerCycle();
}

float
GAPredicateAdaptor::globalPossPercentage(RelativeTeamSide rts)
{
  return pMGA->episode_summaries.getGlobalSummaryEpisode().getPossPercentage(rts);
}

int
GAPredicateAdaptor::globalHistoryCycles()
{
  return pMGA->episode_summaries.getGlobalSummaryEpisode().getCycles();
}  


bool
GAPredicateAdaptor::haveDataForAll(const AdaptStyle& as)
{
  for (FlatStrategySpecSpace::const_iterator iter = getFlatStrategySpace().begin();
       iter != getFlatStrategySpace().end();
       ++iter)
    {
      if (iter->getAdaptStyle() != as)
        continue;
      if (!haveDataFor(*iter))
        return false;
    }
  return true;
}

bool
GAPredicateAdaptor::haveGoodDataAmtForAll(const AdaptStyle& as)
{
  for (FlatStrategySpecSpace::const_iterator iter = getFlatStrategySpace().begin();
       iter != getFlatStrategySpace().end();
       ++iter)
    {
      if (iter->getAdaptStyle() != as)
        continue;
      if (!hasGoodDataAmt(*iter))
        return false;
    }
  return true;
}

StrategySpec
GAPredicateAdaptor::getStrategyWBestScoreDiff()
{
  SSF_HaveData filter(this);
  SSO_BestSD order(this);
  return getBestStrategy(&filter, &order);
}

StrategySpec
GAPredicateAdaptor::getStrategyWDataAndBestScoreDiff()
{
  SSF_GoodAmtData filter(this);
  SSO_BestSD order(this);
  return getBestStrategy(&filter, &order);
}

StrategySpec
GAPredicateAdaptor::getStrategyWBestMeanX()
{
  SSO_BestMeanX order(this);
  return getBestStrategy(&order);
}

StrategySpec
GAPredicateAdaptor::getStrategyWDataAndBestMeanX()
{
  SSF_GoodAmtData filter(this);
  SSO_BestMeanX order(this);
  return getBestStrategy(&filter, &order);
}

StrategySpec
GAPredicateAdaptor::getStrategyWBestPossPerc()
{
  SSO_BestPossPerc order(this);
  return getBestStrategy(&order);
}

StrategySpec
GAPredicateAdaptor::getStrategyWDataAndBestPossPerc()
{
  SSF_GoodAmtData filter(this);
  SSO_BestPossPerc order(this);
  return getBestStrategy(&filter, &order);
}


StrategySpec
GAPredicateAdaptor::getScorelessStrategyWData()
{
  SSF_GoodAmtData filter1(this);
  SSF_Scoreless filter2(this);
  SSO_Const order(this, 0);
  return getBestStrategy(&filter1, &filter2, &order);
}


int
GAPredicateAdaptor::getScoreDiff()
{
  return
    (history.getWorldState().getScore(relativeTeamSideToAbs(RTS_Mine, getMySide())) -
     history.getWorldState().getScore(relativeTeamSideToAbs(RTS_Theirs, getMySide())));
}

int
GAPredicateAdaptor::getMyGoals()
{
  return history.getWorldState().getScore(relativeTeamSideToAbs(RTS_Mine, getMySide()));
}

int
GAPredicateAdaptor::getTheirGoals()
{
  return history.getWorldState().getScore(relativeTeamSideToAbs(RTS_Theirs, getMySide()));
}


int
GAPredicateAdaptor::getCyclesLeft()
{
  return 2 * ServerParam::instance()->getSPHalfCycles() - getTime();
}


int
GAPredicateAdaptor::getQuarter()
{
  int quarter_game = (ServerParam::instance()->getSPHalfCycles()*2) / 4;
  for (int i = 1; ; i++)
    {
      if (getTime() <= i * quarter_game)
          return i;
    }

  errorlog << "GAPredicateAdaptor::getQuarter: how did I get here" << ende;
  return -1;
}

StrategySpec
GAPredicateAdaptor::getCmpToThatNeedsData(const CmpType& ct,
					  const StrategySpec& spec,
					  bool reverse_ss_order)
{
  SSF_GoodAmtData filter1(this);
  filter1.setReverseAccept();
  SSF_CmpTo filter2(this, ct, spec);
  SSO_StrategySpecOrder order(this);
  if (reverse_ss_order)
    order.setReverseCompare();
  return getBestStrategy(&filter1, &filter2, &order);
}

StrategySpec
GAPredicateAdaptor::getCmpToElseEqual(const CmpType& ct,
				      const StrategySpec& spec,
				      bool reverse_ss_order)
{
  SSF_CmpTo filter(this, ct, spec);
  SSO_StrategySpecOrder order(this);
  if (reverse_ss_order)
    order.setReverseCompare();
  StrategySpec newspec = getBestStrategy(&filter, &order);
  return (newspec.getAdaptStyle() == AS_Invalid) ? spec : newspec;
}


StrategySpec
GAPredicateAdaptor::getBestStrategy(std::vector<StrategySpecFilter*>& vfilter, StrategySpecOrder* order)
{
  StrategySpec retval(AS_Invalid, -1);
  for (FlatStrategySpecSpace::const_iterator iter = getFlatStrategySpace().begin();
       iter != getFlatStrategySpace().end();
       ++iter)
    {
      bool match = true;
      for (std::vector<StrategySpecFilter*>::iterator filter_iter = vfilter.begin();
	   filter_iter != vfilter.end();
	   ++filter_iter)
	{
	  if (!(*filter_iter)->accept(*iter))
	    {
	      match = false;
	      break;
	    }
	}

      if (!match)
	continue;
      
      if (retval.getAdaptStyle() == AS_Invalid || order->compare(retval, *iter) < 0)
        retval = *iter;
    }
  return retval;
}

StrategySpec
GAPredicateAdaptor::getBestStrategy(StrategySpecOrder* order)
{
  std::vector<StrategySpecFilter*> v;
  return getBestStrategy(v, order);  
}

StrategySpec
GAPredicateAdaptor::getBestStrategy(StrategySpecFilter* filter1, StrategySpecOrder* order)
{
  std::vector<StrategySpecFilter*> v;
  v.push_back(filter1);
  return getBestStrategy(v, order);
}

StrategySpec
GAPredicateAdaptor::getBestStrategy(StrategySpecFilter* filter1, StrategySpecFilter* filter2,
				    StrategySpecOrder* order)
{
  std::vector<StrategySpecFilter*> v;
  v.push_back(filter1);
  v.push_back(filter2);
  return getBestStrategy(v, order);
}


void
GAPredicateAdaptor::printDebugTo(std::ostream& os)
{
  os << "GAPredicateAdaptor: DEBUG output" << std::endl;
  os << "getCurrentStrategy(): " << getCurrentStrategy() << std::endl;
  os << "timeForCurrentStrategy(): " << timeForCurrentStrategy() << std::endl;
  os << "getDefaultStrategy(): " << getDefaultStrategy() << std::endl;
  os << "globalMeanScoreDiffPerCycle(): " << globalMeanScoreDiffPerCycle() << std::endl;
  os << "globalMeanMyGoalsPerCycle(): " << globalMeanMyGoalsPerCycle() << std::endl;
  os << "globalMeanTheirGoalsPerCycle(): " << globalMeanTheirGoalsPerCycle() << std::endl;
  os << "getStrategyWBestScoreDiff(): " << getStrategyWBestScoreDiff() << std::endl;
  os << "getStrategyWDataAndBestScoreDiff(): " << getStrategyWDataAndBestScoreDiff() << std::endl;
  os << "getScorelessStrategyWData(): " << getScorelessStrategyWData() << std::endl;
  os << "getMySide(): " << getMySide() << std::endl;
  os << "getScoreDiff(): " << getScoreDiff() << std::endl;
  os << "getMyGoals(): " << getMyGoals() << std::endl;
  os << "getTheirGoals(): " << getTheirGoals() << std::endl;
  os << "winning(): " << winning() << std::endl;
  os << "losing(): " << losing() << std::endl;
  os << "tied(): " << tied() << std::endl;
  os << "getTime(): " << getTime() << std::endl;
  os << "getCyclesLeft(): " << getCyclesLeft() << std::endl;
  os << "getQuarter(): " << getQuarter() << std::endl;

  for (FlatStrategySpecSpace::const_iterator iter = getFlatStrategySpace().begin();
       iter != getFlatStrategySpace().end();
       ++iter)
    {
      os << "haveDataFor(" << *iter << "): " << haveDataFor(*iter) << std::endl;
      os << "hasGoodDataAmt(" << *iter << "): " << hasGoodDataAmt(*iter) << std::endl;
      os << "meanScoreDiffPerCycle(" << *iter << "): " << meanScoreDiffPerCycle(*iter) << std::endl;
      os << "getMoreDefThatNeedsData(" << *iter << "): " << getMoreDefThatNeedsData(*iter) << std::endl;
      os << "getMoreOffThatNeedsData(" << *iter << "): " << getMoreOffThatNeedsData(*iter) << std::endl;
      os << "getMoreDef(" << *iter << "): " << getMoreDef(*iter) << std::endl;
      os << "getMoreOff(" << *iter << "): " << getMoreOff(*iter) << std::endl;
  }
}


/***********************************************************************************/

