/* -*- Mode: C++ -*- */

#ifndef _FORMATION_H_
#define _FORMATION_H_

#include <iostream>
#include <vector>
#include "Geometry.h"
#include "region.h"
#include "soccer_utils.h"
#include "CoachMessageQueue.h"
#include "WorldState.h"
#include "StoreRegionMsgBuilder.h"

class FormationFormatter;

/* Formation coordinates are all left hand side */
class Formation 
{
public:
  Formation(const char* n = NULL);
  Formation(const Formation& f);
  ~Formation();

  const Formation& operator=(const Formation& f);

  void generateRandom(float xmean, float xstdev, float ymean, float ystdev);
  
  rcss::clang::Region* getHomeRegion(Unum num) const;
  VecPosition getHomeRegionCentralPoint(Unum num) const;  
  //takes over the memory
  void setHomeRegion(Unum num, rcss::clang::Region* r);

  PlayerRole getRole(Unum num);
  
  friend std::ostream& operator <<(std::ostream & os, const Formation &f);
  friend std::istream& operator >>(std::istream & is, Formation &f);  

  const std::string& getName() const { return name; }
  void setName(const char* n) { name = n; }
  void setNameFromFilename(const char* filename);  
  
  std::string getRuleName() const { return RULE_NAME_PREFIX + getName(); }
  
  /* In our internal representation, the goalie is always number 1. This sets our
     permutation to put the goalie in the right place */
  void permuteForGoalie(const WorldState& ws, TeamSide side);
  void permuteForGoalie(int goalie_num);

  void createRule(CoachMessageQueue* pq, const std::vector<FormationFormatter*>& vformatter) const;
  void setEnableState(bool state, CoachMessageQueue* pq) const;
  
  /* Remember that stored numbers are 0-based!!!! */
  int getStoredNumberOfUnum(Unum num) const;
  Unum getUnumOfStoredNumber(int num) const;

  void draw(FieldImage* pfi);
  
  static const int GOALIE_STORED_NUMBER;
  static const std::string RULE_NAME_PREFIX;
  static const float INSIDE_ALPHA_VALUE;
  
private:
  //perm is 1-based!
  typedef std::vector<int> PermType;
  PermType perm;
  typedef std::vector<rcss::clang::Region*> RegionStorage;
  RegionStorage vReg;
  std::string name;

  typedef std::vector<PlayerRole> PlayerRoleStorage;
  PlayerRoleStorage vRole;
  bool need_to_update_roles;

};

/********************************************************************/

/* this class should take a home region and generate a rule for a player */
class FormationFormatter
{
public:
  FormationFormatter() {}
  virtual ~FormationFormatter() {}

  //num is 1-based
  virtual rcss::clang::Rule* createRuleFromHomeRegion(rcss::clang::Region* reg, int num) = 0;

  static FormationFormatter* createFrom(const char* str);
};

class FormationFormatter_Null
  : public FormationFormatter
{
public:
  FormationFormatter_Null() : FormationFormatter() {}
  ~FormationFormatter_Null() {};
  
  rcss::clang::Rule* createRuleFromHomeRegion(rcss::clang::Region* reg, int num) { return NULL; }
};

class FormationFormatter_Home
  : public FormationFormatter
{
public:
  FormationFormatter_Home() : FormationFormatter() {}
  ~FormationFormatter_Home() {};
  
  rcss::clang::Rule* createRuleFromHomeRegion(rcss::clang::Region* reg, int num);
};

class FormationFormatter_CenterPlusAttract
  : public FormationFormatter
{
public:
  FormationFormatter_CenterPlusAttract(const VecPosition& attraction)
    : FormationFormatter(), attraction(attraction) {}
  // string should be of the form cplus(NUM,NUM)
  FormationFormatter_CenterPlusAttract(const char* str);
  ~FormationFormatter_CenterPlusAttract() {};
  
  rcss::clang::Rule* createRuleFromHomeRegion(rcss::clang::Region* reg, int num);

private:
  VecPosition attraction;
};

class FormationFormatter_BKOAdjust
  : public FormationFormatter
{
public:
  FormationFormatter_BKOAdjust(FormationFormatter* pNormalFF)
    : FormationFormatter(), pNormalFF(pNormalFF) {}
  ~FormationFormatter_BKOAdjust() {};
  
  rcss::clang::Rule* createRuleFromHomeRegion(rcss::clang::Region* reg, int num);

private:
  FormationFormatter* pNormalFF;
  
};

#endif
