#include <iostream>
#include <sstream>
#include "Formation.h"
#include "ServerParam.h"
#include "Logger.h"
#include "misc.h"
#include "constants.h"
#include "clangparser.h"
#include "FixedRWMI.h"

using namespace spades;

/**************************************************************************/
/************************** Formation **********************************/  
/**************************************************************************/
const int Formation::GOALIE_STORED_NUMBER = 0;
const std::string Formation::RULE_NAME_PREFIX("FORM_");
const float Formation::INSIDE_ALPHA_VALUE = 0.5;

Formation::Formation(const char* name)
  : perm(),
    vReg(ServerParam::instance()->getSPTeamSize()),
    name(name==NULL ? "" : name),
    vRole(ServerParam::instance()->getSPTeamSize(), PR_None),
    need_to_update_roles(true)
{
  std::back_insert_iterator<PermType> iter(perm);
  for (int i=0; i<=ServerParam::instance()->getSPTeamSize(); i++)
    *iter++ = i-1;
}

Formation::Formation(const Formation& f)
  : perm(f.perm),
    vReg(), //we'll do a deep copy below
    name(f.name),
    vRole(f.vRole),
    need_to_update_roles(f.need_to_update_roles)
{
  std::transform(f.vReg.begin(), f.vReg.end(),
		 std::back_insert_iterator<RegionStorage>(vReg),
		 rcss::clang::RegionDeepCopy());
}


Formation::~Formation()
{
  std::for_each(vReg.begin(), vReg.end(), deleteptr<rcss::clang::Region>());
}

const Formation&
Formation::operator=(const Formation& f)
{
  //get rid of our old memory
  std::for_each(vReg.begin(), vReg.end(), deleteptr<rcss::clang::Region>());
  //now copy all the values
  perm = f.perm;
  std::transform(f.vReg.begin(), f.vReg.end(),
		 std::back_insert_iterator<RegionStorage>(vReg),
		 rcss::clang::RegionDeepCopy());
  name = f.name;
  vRole = f.vRole;
  need_to_update_roles = f.need_to_update_roles;
  return *this;
}


rcss::clang::Region*
Formation::getHomeRegion(Unum num) const
{
  if (num < 1 || num > (signed)perm.size())
    {
      errorlog << "Formation::getHomeRegion: num out of range " << num << "(" << vReg.size() << ")" << ende;
      return NULL;
    }
  return vReg[perm[num]];
}

VecPosition
Formation::getHomeRegionCentralPoint(Unum num) const
{
  rcss::clang::Region* r = getHomeRegion(num);
  if (r == NULL)
    return VecPosition(0,0);

  //SMURF: This is a big hack. I'll assume that the home regions are made
  //       of rectangles with fixed points
  rcss::clang::RegRec* rec = dynamic_cast<rcss::clang::RegRec*>(r);

  if (rec == NULL)
    {
      errorlog << "Formation region is not a rec! " << ende;
      return VecPosition(0,0);
    }
  
  VecPosition p0 = ((rcss::clang::PointSimple*)(rec->getPt(0)))->getVec();
  VecPosition p1 = ((rcss::clang::PointSimple*)(rec->getPt(1)))->getVec();

  return (p0 + p1) / 2;
}


void
Formation::setHomeRegion(Unum num, rcss::clang::Region* r)
{
  if (num < 1 || num > (signed)perm.size())
    {
      errorlog << "Formation::setHomeRegion: num out of range " << num << "(" << vReg.size() << ")" << ende;
      return;
    }
  rcss::clang::Region *oldr = vReg[perm[num]];
  if (oldr != NULL)
    delete oldr;
  vReg[perm[num]] = r;
}

PlayerRole
Formation::getRole(Unum num)
{
  if (num < 1 || num > (signed)perm.size())
    {
      errorlog << "Formation::getRole num out of range " << num << "(" << vReg.size() << ")" << ende;
      return PR_None;
    }
  if (need_to_update_roles)
    {
      if (!inferRolesFromRegion(vReg, vRole, 1))
	errorlog << "Formation: could not infer roles" << ende;
    }
  return vRole[perm[num]];
}


/* FILE FORMAT:
   Name: <name>
   Region 1
   Region 2
   ...
   Region 11
*/ 
std::ostream&
operator <<(std::ostream & os, const Formation &f)
{
  os << "Name: " << f.name << std::endl;
  os << "# Permutation: ";
  copy(f.perm.begin(), f.perm.end(), std::ostream_iterator<int>(os, " "));
  os << std::endl;
  for_each(f.vReg.begin(), f.vReg.end(), DerefPrinter<rcss::clang::Region>(os,"\n"));
  return os;
}

std::istream&
operator >>(std::istream & is, Formation &f)
{
  skip_to_non_comment(is);
  //skip to a colon
  is.ignore(MAX_SERVER_MSG_LEN, ':');
  if (is.eof() || !skip_white_space(is))
    {
      is.setstate(std::ios::failbit);
      return is;
    }
  is >> f.name;
  
  StoreRegionMsgBuilder msg_builder;
  rcss::clang::Parser clang_parser(msg_builder);

  for (int idx=0; idx<ServerParam::instance()->getSPTeamSize(); idx++)
    {
      if (!skip_to_non_comment(is))
	{
	  //eof
	  break;
	}
      
      char line[MAX_SERVER_MSG_LEN];
      is.getline(line, MAX_SERVER_MSG_LEN);
      try {
	if (clang_parser.parse(line) != 0)
	  errorlog << "Could not parse line for Formation: " << line << ende;
	f.vReg[idx] = msg_builder.getLastRegion();
      }
      catch( rcss::clang::BuilderErr e)
	{ errorlog << "Error parsing Formation: " << e.what() << ende; }
      catch( rcss::util::NullErr e)
	{ errorlog << "Error parsing Formation: " << e.what() << ende; }
    }

  return is;
}


/* In our internal representation, the goalie is always region 0. This sets our
   permutation to put the goalie in the right place */
void
Formation::permuteForGoalie(const WorldState& ws, TeamSide side)
{
  int gnum = ws.getGoalieNum(side);
  if (gnum <= 0)
    {
      actionlog(100) << "Formation: Asked to permute for goalie, but found no goalie "
		     << side << " " << gnum << ende;
      return;
    }
  permuteForGoalie(gnum);
}

void
Formation::permuteForGoalie(int gnum)
{
  for (int i=1; i< (signed)perm.size(); i++)
    {
      if (perm[i] == 0)
	{
	  //This is the old goalie!
	  perm[i] = perm[gnum];
	  break;
	}
    }
  perm[gnum] = 0;
}


void
Formation::createRule(CoachMessageQueue* pq,
		      const std::vector<FormationFormatter*>& vformatter) const
{
  rcss::clang::NestedRule* overallrule =
    new rcss::clang::NestedRule( std::auto_ptr <rcss::clang::Cond> (new rcss::clang::CondBool(true)) );
    // This is probably a temp hack to avoid our team always going home
    //new rcss::clang::SimpleRule( std::auto_ptr <rcss::clang::Cond> (new rcss::clang::CondPlayMode(rcss::clang::BeforeKickOff)) );
  for (int num=1; num <= ServerParam::instance()->getSPTeamSize(); num++)
    {
      if (!vReg[perm[num]]) continue;
      rcss::clang::Rule* r = vformatter[perm[num]]->createRuleFromHomeRegion(vReg[perm[num]], num);
      if (r == NULL)
	continue;
      overallrule->push_back( std::auto_ptr <rcss::clang::Rule> (r) );
    }
  rcss::clang::DefRule* def_rule =
    new rcss::clang::DefRule(getRuleName(),
			     std::auto_ptr<rcss::clang::Rule>(overallrule),
			     false);
  pq->getDefineContainer().push(def_rule);
  setEnableState(false, pq);
}

void
Formation::setEnableState(bool state, CoachMessageQueue* pq) const
{
  rcss::clang::RuleIDList l;
  l.push_back(getRuleName());
  pq->getRuleContainer().push( new rcss::clang::ActivateRules(state, l) );
}


#define LOTS_OF_COLOR
#ifdef LOTS_OF_COLOR
const FieldImage::Color formation_draw_col[] = {
  FieldImage::Color("#15ff00"),
  FieldImage::Color("#f4f00e"),
  FieldImage::Color("#f30ef7"),
  FieldImage::Color("#f90d09"),
  FieldImage::Color("#0973f4"),
  FieldImage::Color("#ffa100"),
  FieldImage::Color("#09f4d9"),
  FieldImage::Color("#ea5da1"),
  FieldImage::Color("#020f68"),
  FieldImage::Color("#0c6802"),
  FieldImage::Color("#870291")
};
#else
const FieldImage::Color formation_draw_col[] = {
  FieldImage::Color("#15ff00"),
  FieldImage::Color("#f90d09"),
  FieldImage::Color("#020f68"),
  FieldImage::Color("#f90d09"),
  FieldImage::Color("#020f68"),
  FieldImage::Color("#f90d09"),
  FieldImage::Color("#020f68"),
  FieldImage::Color("#f90d09"),
  FieldImage::Color("#020f68"),
  FieldImage::Color("#f90d09"),
  FieldImage::Color("#020f68")
};
#endif

void
Formation::draw(FieldImage* pfi)
{
  if (pfi->getLegendLinesLeft())
    pfi->addLegendLine(name.c_str());

  FixedRWMI wmi;
  VarBindings bindings;
  
  for (int i=1; i<= ServerParam::instance()->getSPTeamSize(); i++)
    {
      if (!vReg[perm[i]])
	continue;
      FieldImage::Color c_border = formation_draw_col[perm[i]];
      FieldImage::Color c_inside(c_border);
      c_inside.setAlpha(INSIDE_ALPHA_VALUE);
      vReg[perm[i]]->draw(pfi, true, c_border, c_inside, &wmi, bindings);
      VecPosition pt = getHomeRegionCentralPoint(i);
      pfi->addPoint(pt, formation_draw_col[perm[i]]);
      char num[5];
      sprintf(num, "%d", i);
      pfi->addText(num, pt);
  }
}


void
Formation::generateRandom(float xmean, float xstdev, float ymean, float ystdev)
{
  for (int i=0; i<(signed)perm.size(); i++)
    perm[i] = i-1;
  for (int num=0; num<(signed)vReg.size(); num++)
    {
      float wid = gaussian_sample(xmean, xstdev);
      float hgt = gaussian_sample(ymean, ystdev);
      VecPosition center(range_random(-ServerParam::instance()->getSPPitchLength()/2,
				      ServerParam::instance()->getSPPitchLength()/2),
			 range_random(-ServerParam::instance()->getSPPitchWidth()/2,
				      ServerParam::instance()->getSPPitchWidth()/2));

      vReg[num] =
	new rcss::clang::RegRec ( std::auto_ptr<rcss::clang::Point>(new rcss::clang::PointSimple(center.getX()-wid/2, center.getY()-hgt/2)),
				  std::auto_ptr<rcss::clang::Point>(new rcss::clang::PointSimple(center.getX()+wid/2, center.getY()+hgt/2)) );
    }

  need_to_update_roles = true;
}

int
Formation::getStoredNumberOfUnum(Unum num) const
{
  if (num < 1 || num > (signed)perm.size())
    {
      errorlog << "Formation::getStoredNumberOfUnum: num out of range " << num << ende;
      return -1;
    }
  return perm[num];
}

Unum
Formation::getUnumOfStoredNumber(int num) const
{
  if (num < 0 || num >= (signed)perm.size())
    {
      errorlog << "Formation::getUnumOfStoredNumbe:: num out of range " << num << ende;
      return -1;
    }
  for (int i=1; i<(signed)perm.size(); i++)
    if (perm[i] == num)
      return i;
  errorlog << "How could I not find the Unum? " << num << ende;
  return -1;
}

void
Formation::setNameFromFilename(const char* filename)
{
  const char* pstart = strrchr(filename, '/');
  if (pstart == NULL)
    pstart = filename;
  else
    pstart++;

  std::string s;
  char* pext = strrchr(filename, '.');
  if (pext)
    s.assign(pstart, pext-pstart);
  else
    s.assign(pstart);

  setName(s.c_str());
}

/********************************************************************/
//static
FormationFormatter*
FormationFormatter::createFrom(const char* str)
{
  if (prefixMatch("null", str))
    {
      return new FormationFormatter_Null;
    }
  else if (prefixMatch("home", str))
    {
      return new FormationFormatter_Home;
    }
  else if (prefixMatch("cplusa", str))
    {
      return new FormationFormatter_CenterPlusAttract(str);
    }
  else if (prefixMatch("bkoadj_home", str))
    {
      return new FormationFormatter_BKOAdjust(new FormationFormatter_Home);
    }
  else if (prefixMatch("bkoadj_cplusa", str))
    {
      return new FormationFormatter_BKOAdjust(new FormationFormatter_CenterPlusAttract(str));
    }
  else
    {
      errorlog << "FormationFormatter::createFrom: didn't understand '" << str << "'" << ende;
      return NULL;
    }
}


rcss::clang::Rule*
FormationFormatter_Home::createRuleFromHomeRegion(rcss::clang::Region* reg, int num)
{
  actionlog(170) << "FormationFormatter_Home: running for " << num << " and " << *reg << ende;

  rcss::clang::SimpleRule* r =
    new rcss::clang::SimpleRule( std::auto_ptr <rcss::clang::Cond> (new rcss::clang::CondBool(true)) );

  rcss::clang::UNumSet uset;
  uset.add((rcss::clang::UNum::unum_t)num);
  rcss::clang::DirComm* d = new rcss::clang::DirComm();
  d->setSense(true);
  d->setOurSide(true);
  d->set(uset);
  rcss::clang::ActHome* a = new rcss::clang::ActHome();
  a->set( std::auto_ptr<rcss::clang::Region>( reg->deepCopy() ));
  d->push_back( std::auto_ptr<rcss::clang::Action>(a) );
  r->push_back( std::auto_ptr<rcss::clang::Dir>(d) );

  return r;
}

FormationFormatter_CenterPlusAttract::FormationFormatter_CenterPlusAttract(const char* str)
{
  //skip the cplusa
  char* paren = strchr(str, '(');
  if (paren == NULL)
    {
      errorlog << "FormationFormatter_CenterPlusAttract: could not find paren in '"
	       << str << "'" << ende;
      return;
    }

  std::istringstream is(paren);

  is >> attraction;
  if (is.fail())
    errorlog << "FormationFormatter_CenterPlusAttract: could not read arg from '"
             << str << "'" << ende;
}

rcss::clang::Rule*
FormationFormatter_CenterPlusAttract::createRuleFromHomeRegion(rcss::clang::Region* reg, int num)
{
  actionlog(170) << "FormationFormatter_CenterPlusAttrach: running for " << num << " and " << *reg << ende;

  rcss::clang::SimpleRule* r =
    new rcss::clang::SimpleRule( std::auto_ptr <rcss::clang::Cond> (new rcss::clang::CondBool(true)) );

  rcss::clang::UNumSet uset;
  uset.add((rcss::clang::UNum::unum_t)num);
  rcss::clang::DirComm* d = new rcss::clang::DirComm();
  d->setSense(true);
  d->setOurSide(true);
  d->set(uset);
  rcss::clang::ActHome* a = new rcss::clang::ActHome();

  VarBindings bindings;
  FixedRWMI rwmi;
  
  rcss::clang::PointArith* ptAttract =
    new rcss::clang::PointArith( std::auto_ptr<rcss::clang::Point>( new rcss::clang::PointBall()),
				 std::auto_ptr<rcss::clang::Point>( new rcss::clang::PointSimple(attraction.getX(), attraction.getY())),
				 rcss::util::ArithOp::mult() );

  // HACK: randomPtIn returns the center of a rec region
  VecPosition home_center = reg->getRandomPtIn(&rwmi, bindings);
  
  rcss::clang::PointArith* ptTotal =
    new rcss::clang::PointArith( std::auto_ptr<rcss::clang::Point>( new rcss::clang::PointSimple(home_center.getX(), home_center.getY())),
				 std::auto_ptr<rcss::clang::Point>( ptAttract ),
				 rcss::util::ArithOp::plus() );

  a->set( std::auto_ptr<rcss::clang::Region>( std::auto_ptr<rcss::clang::Region>(new rcss::clang::RegPoint( std::auto_ptr<rcss::clang::Point>(ptTotal)))));
  
  d->push_back( std::auto_ptr<rcss::clang::Action>(a) );
  r->push_back( std::auto_ptr<rcss::clang::Dir>(d) );

  return r;
}

rcss::clang::Rule*
FormationFormatter_BKOAdjust::createRuleFromHomeRegion(rcss::clang::Region* reg, int num)
{
  actionlog(170) << "FormationFormatter_HomeAndBKOAdjust: running for " << num << " and " << *reg << ende;
  
  rcss::clang::CondAnd* pBKOCond = new rcss::clang::CondAnd;
  pBKOCond->push_back(std::auto_ptr<rcss::clang::Cond>(new rcss::clang::CondPlayMode(rcss::clang::BeforeKickOff)));
  pBKOCond->push_back(std::auto_ptr<rcss::clang::Cond>(new rcss::clang::CondPlayMode(rcss::clang::AfterGoal_Our)));
  pBKOCond->push_back(std::auto_ptr<rcss::clang::Cond>(new rcss::clang::CondPlayMode(rcss::clang::AfterGoal_Opp)));
  
  rcss::clang::NestedRule* rNotBKO =
    new rcss::clang::NestedRule( std::auto_ptr <rcss::clang::Cond> ( std::auto_ptr <rcss::clang::Cond> (new rcss::clang::CondNot(pBKOCond->deepCopy() ))));
  rNotBKO->push_back( std::auto_ptr<rcss::clang::Rule>(pNormalFF->createRuleFromHomeRegion(reg, num)));

  rcss::clang::SimpleRule* rBKO =
    new rcss::clang::SimpleRule( std::auto_ptr <rcss::clang::Cond> ( pBKOCond ));
  
  {
    VarBindings bindings;
    FixedRWMI rwmi;
    rcss::clang::UNumSet uset;
    uset.add((rcss::clang::UNum::unum_t)num);
    rcss::clang::DirComm* d = new rcss::clang::DirComm();
    d->setSense(true);
    d->setOurSide(true);
    d->set(uset);
    rcss::clang::ActHome* a = new rcss::clang::ActHome();
    VecPosition bko_pos = reg->getRandomPtIn(&rwmi, bindings);
    bko_pos.setX( min (bko_pos.getX(), -1));
    a->set( std::auto_ptr<rcss::clang::Region>( new rcss::clang::RegPoint( std::auto_ptr<rcss::clang::Point>(new rcss::clang::PointSimple(bko_pos.convertToVector2D())))));
    d->push_back( std::auto_ptr<rcss::clang::Action>(a) );
    rBKO->push_back( std::auto_ptr<rcss::clang::Dir>(d) );
  }

  rcss::clang::NestedRule* r =
    new rcss::clang::NestedRule(std::auto_ptr <rcss::clang::Cond> (new rcss::clang::CondBool(true)));
  r->push_back( std::auto_ptr<rcss::clang::Rule>(rNotBKO) );
  r->push_back( std::auto_ptr<rcss::clang::Rule>(rBKO) );
  
  return r;
}

