/* -*- Mode: C++ -*- */

#include <algorithm>
#include "FixedAdviceAU.h"
#include "ModCMRep.h"
#include "FileReader.h"
#include "ModuleRegistry.h"
#include "version.h"
#include "CLangFileReader.h"
#include "Logger.h"

using namespace spades;

/************************************************************************************/

class FixedAdviceMapFileReader
  : public FileReader
{
public:
  FixedAdviceMapFileReader(FixedAdviceAdaptationUnit* p) : pFAAU(p) {}
  ~FixedAdviceMapFileReader() {}

  bool
  processLine(std::istrstream& line, const char* fileid, const char* path, float version)
  {
    std::string style_str;
    std::string rule_name;
    std::string file_name;
    AdaptStyle style;

    line >> style_str >> rule_name >> file_name;
    if (line.fail())
      {
        errorlog << "FixedAdviceMapFileReader: failed read at " << fileid << ende;
        return false;
      }

    style = getStyleFor(style_str.c_str());
    if (style == AS_Invalid)
      {
        errorlog << "FixedAdviceMapFileReader: at " << fileid
                 << ": didn't understand style '" << style_str << "'"
                 << ende;
        return false;
      }
    
    return pFAAU->addAdvice(style, rule_name.c_str(), file_name.c_str());
  }
  
private:
  FixedAdviceAdaptationUnit* pFAAU;
};



/************************************************************************************/

FixedAdviceAdaptationUnit::FixedAdviceAdaptationUnit(ModuleRegistry* pReg, const char* infn)
  : mqueue("FixedAdvice"), pCMRep(NULL), rule_names()
{
  pCMRep = (ModCMRep*)pReg->lookup("CMRep");
  if (pCMRep == NULL)
    {
      errorlog << "FixedAdviceAdaptationUnit needs ModCMRep" << ende;
    }
  else
    {
      pCMRep->addCMQ(&mqueue);
    }

  FixedAdviceMapFileReader reader(this);
  reader.readFile(infn, g_Version);
}

FixedAdviceAdaptationUnit::~FixedAdviceAdaptationUnit()
{
  if (pCMRep)
    pCMRep->removeCMQ(&mqueue);
}

 
int
FixedAdviceAdaptationUnit::getNumStrategiesForStyle(AdaptStyle style)
{
  return std::count_if(rule_names.begin(), rule_names.end(), StrategySpecStyleEq(style));
}


void
FixedAdviceAdaptationUnit::changeStrategy(const StrategySpec& oldspec, const StrategySpec& newspec)
{
  if (oldspec.getAdaptStyle() != AS_Invalid && newspec.getAdaptStyle() != AS_Invalid &&
      rule_names[oldspec] == rule_names[newspec])
    return;
  
  if (oldspec.getAdaptStyle() != AS_Invalid)
    changeAdviceState(oldspec, false); 
  if (newspec.getAdaptStyle() != AS_Invalid)
    changeAdviceState(newspec, true); 
}


bool
FixedAdviceAdaptationUnit::addAdvice(AdaptStyle style, const char* rule_name, const char* fn)
{
  if (files_read.count(fn) == 0)
    {
      CLangFileReader reader(mqueue);
      reader.readFile(fn, g_Version);
      files_read.insert(fn);
    }

  StrategySpec newspec(style, getNumStrategiesForStyle(style));
  rule_names[newspec] = std::string(rule_name);

  changeAdviceState(newspec, false);

  return true;
}

void
FixedAdviceAdaptationUnit::changeAdviceState(const StrategySpec& spec, bool on)
{
  RuleNamesStorage::iterator iter = rule_names.find(spec);
  if (iter == rule_names.end())
    {
      errorlog << "FixedAdviceAdaptationUnit::changeAdviceState: out of range: "
               << spec
               << ende;
      return;
    }

  rcss::clang::RuleIDList l;

  //#define USE_CASPIAN_HACK	
#ifdef USE_CASPIAN_HACK	
  // BIG HIGE MONDO HACK FOR THE BROKEN Caspian players
  switch (spec.getAdaptStyle())
    {
    case AS_Defensive:
      if (spec.getStyleIdx() == 0)
	{
	  actionlog(150) << "Caspian hack: handling D0"  << ende;
	  l.push_back("XVDCBU");	  
	  l.push_back("XVDRBU");	  
	  l.push_back("XVDLBU");
	  l.push_back("XVDClear");
	}
      else if (spec.getStyleIdx() == 1)
	{
	  actionlog(150) << "Caspian hack: handling D1 "  << ende;
	  l.push_back(iter->second);
	}
      else
	{
	  errorlog << "FixedAdviceAU: what is def idx?" << spec << ende;
	  l.push_back(iter->second);	  
	}
      break;
    case AS_Normal:
      actionlog(150) << "Caspian hack: handling N"  << ende;
      l.push_back(iter->second);
      break;
    case AS_Offensive:
      if (spec.getStyleIdx() == 0)
	{
	  actionlog(150) << "Caspian hack: handling O0"  << ende;
	  l.push_back(iter->second);
	}
      else if (spec.getStyleIdx() == 1)
	{
	  actionlog(150) << "Caspian hack: handling O1 "  << ende;
	  l.push_back("XVOAdv5");      
	  l.push_back("XVOAdv6");      
	  l.push_back("XVOAdv7");      
	  l.push_back("XVOAdv8");      
	  l.push_back("XVOAdv9");
	}
      else
	{
	  errorlog << "FixedAdviceAU: what is off idx?" << spec << ende;
	  l.push_back(iter->second);	  
	}
      break;
    default:
      errorlog << "FixedAdviceAU: what is spec?" << spec << ende;
      return;
    }
#else
  l.push_back(iter->second);
#endif	

#ifdef USE_CASPIAN_HACK	
  rcss::clang::ActivateRules actv_rule(on, l);
  splitIndividualNamesTo( actv_rule, mqueue.getRuleContainer());
#else
  mqueue.getRuleContainer().push( new rcss::clang::ActivateRules(on, l) );
#endif	

}


