/* -*- Mode: c++ -*- */

/*
 *Copyright:

    Copyright (C) 2002 Patrick Riley
    Copyright (C) 2001 Patrick Riley and Emil Talpes

    This file is part of the SPADES simulation system.

    The SPADES simulation system is free software; you can
    redistribute it and/or modify it under the terms of the GNU Lesser
    General Public License as published by the Free Software
    Foundation; either version 2 of the License, or (at your option)
    any later version.

    The SPADES simulation system is distributed in the hope that it
    will be useful, but WITHOUT ANY WARRANTY; without even the implied
    warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
    See the GNU Lesser General Public License for more details.

    You should have received a copy of the GNU Lesser General Public
    License along with the SPADES simulation system; if not, write to
    the Free Software Foundation, Inc., 59 Temple Place, Suite 330,
    Boston, MA 02111-1307 USA

 *EndCopyright:
*/

/* an abstract base class to provide reuse of functionality for reading from a file
   The file reader assumes a line based reading of the scheme
   # is a comment character
*/

#include <iostream>
#include <fstream>
#include <string>
#include <errno.h>
#include "FileReader.h"
#include "utility.h"
#include "Logger.h"

using namespace std;
using namespace spades;

const int spades::FileReader::MaxRecFileDepth = 16;

void
spades::FileReader::readFile (const char *path_name, float max_version, int depth)
{
  if (depth >= MaxRecFileDepth)
    {
      errorlog << "FileReader: Max recursive file depth (" << MaxRecFileDepth << ") exceeded for file '"
	       << path_name << "'" << ende;
      return;
    }
  
  string path = tildeExpand (path_name);
  ifstream ifile;
  float version = -1.0;
  bool got_version_line = false;
  
  ifile.open (path.c_str ());
  if (!ifile.is_open ())
    {
      errorlog << "FileReader: Can't open file '" << path << "'" << ende;
      return;
    }

  int line_no = 1;
  while (ifile.good ())
    {                           //!ifile.eof () )
      //Some versions of libio set the failbit on get(tmp, '\n') if \n is the next character
      // This is the wrong nehavior as far as I am concerned, but here is a workaround
      if (ifile.peek() == '\n')
	{
          ifile.ignore ();
          line_no++;
	  continue;
	}
      
      /* ignore remark line */
      if (ifile.peek () == '#')
        {
          // ignore this line
          strstreambuf tmp;
          ifile.get (tmp, '\n');
          // get the '\n' that the get above, didn't get
          ifile.ignore ();
          line_no++;
          continue;
        }

      strstreambuf linebuf;
      ifile.get (linebuf, '\n');
      ifile.ignore(); //get the '\n' that the get didn't absorb

      if (linebuf.pcount() == 0)
	{
	  line_no++;
	  continue;
	}

      istrstream tryfileline(linebuf.str(), linebuf.pcount());

      //cout << path << ":" << line_no << ": line is: " << linebuf.pcount() << ":" << linebuf.str() << endl;
	   
      string param;
      tryfileline >> param;
      if (!tryfileline || param.empty())
	{
	  line_no++;
	  continue;
	}

      /* this will also trim off the colon if we have 'file:' or 'version:' */
      bool find_colon = true;
      if (param[param.length () - 1] == ':')
        {
          param.resize (param.length () - 1);
          find_colon = false;
        }

      /* now check to see if we need to go recursive */
      if (param == "file")
        {
	  if (!got_version_line)
	    errorlog << "FileReader: " << path << ": " << line_no
		     << ": version line should be the first non-comment line!" << ende;

	  bool format_error = false;
	  
	  if (find_colon)
	    {
	      if (!skip_white_space_on_line(tryfileline))
		{
		  errorlog << "FileReader: " << path << ": " << line_no << ": " 
			   << "Have a file line with no file" << ende;
		  format_error = true;
		}
	      else
		{
		  //we already skipped the white space
		  char c;
		  tryfileline.get(c);
		  if (c != ':')
		    {
		      errorlog << "FileReader: " << path << ": " << line_no
			       << ": Expecting colon after parameter name, not `" << c
			       << ende;

		      format_error = true;
		    }
		}
	    }

	  if (!format_error)
	    {
              string nextfn;
              tryfileline >> nextfn;
              if (!tryfileline.fail())
                {
		  string fn = resolvePathGivenStartingFile(path_name, nextfn.c_str());

		  if (fn == "")
		    {
		      errorlog << "FileReader: Could not resolve path name for nextfn '" << nextfn << "': "
			       << errno << ' ' << strerror(errno)
			       << ende;
		    }
		  else
		    {
		      //cout << "Reading file recursive " << fn << endl;
		      readFile(fn.c_str(), depth+1);
		    }
                }
	      else
		{
		  errorlog << "FileReader: " << path << ": " << line_no
			   << ": Failed to read file name after 'file'"
			   << ende;
		}
	      
            }
        }
      else if (param == "version")
	{
	  bool format_error = false;
	  
	  if (find_colon)
	    {
	      if (!skip_white_space_on_line(tryfileline))
		{
		  errorlog << "FileReader: " << path << ": " << line_no << ": " 
			   << "Have a version line with no version" << ende;
		  format_error = true;
		}
	      else
		{
		  //we already skipped the white space
		  char c;
		  tryfileline.get(c);
		  if (c != ':')
		    {
		      errorlog << "FileReader: " << path << ": " << line_no
			       << ": Expecting colon after parameter name, not `" << c << "'"
			       << ende;

		      format_error = true;
		    }
		}
	    }
	
	  if (!format_error)
	    {	
              tryfileline >> version;
              if (tryfileline.fail())
                {
		  errorlog << "FileReader: " << path << ": " << line_no
		       << ": Failed reading version number" << ende;
                }
	      else
		{
		  got_version_line = true;
		}
            }
	}
      else
	{
	  if (!got_version_line)
	    errorlog << "FileReader: " << path << ": " << line_no
		     << ": version line should be the first non-comment line!"
		     << ende;

	  istrstream line(linebuf.str(), linebuf.pcount());
	  ostrstream fileid;
	  fileid << path << ": " << line_no << ends;
	  if (!processLine(line, fileid.str(), path.c_str(), version))
	    errorlog << "FileReader: Error processing line " << line_no
		     << " of file '" << path << "'" << ende;
	  fileid.freeze(false);
	}

      linebuf.freeze(false);

      line_no++;
      
    }

  ifile.close ();

}

