/* -*- Mode: C++ -*- */

#ifndef _ERR_BOUND_VALUE_H_
#define _ERR_BOUND_VALUE_H_

#include <iostream>

class ErrBoundValue
{
public:
  ErrBoundValue(double d = 0.0) : d(d), lower(0), upper(0), print_verbose(false) {}
  //this is the one-sided error 
  ErrBoundValue(double d, double err) : d(d), lower(err), upper(err), print_verbose(false) {}
  //this is the err, not the absolute value
  ErrBoundValue(double d, double lower, double upper, bool print_verbose=false)
    : d(d), lower(lower), upper(upper), print_verbose(print_verbose) {}

  double getVal() const { return d; }
  double getLowerOffset() const { return lower; }
  double getUpperOffset() const { return upper; }
  double getLowerBound() const { return d-lower; }
  double getUpperBound() const { return d+upper; }

  bool isPossibleValue(double val) const { return val >= getLowerBound() && val <= getUpperBound(); }
  
  void clear() { d = upper = lower = 0.0; }

  void setVal(double val)         { d = val; }
  void setLowerOffset(double val) { lower = val; }
  void setUpperOffset(double val) { upper = val; }
  
  friend std::ostream & operator << (std::ostream & o, const ErrBoundValue& v)
  {
    if (v.print_verbose)
      o << "[" << v.lower << "," << v.d << "," << v.upper << "]";
    else
      o << v.d;
    return o;
  }

  bool getPrintVerbose() const { return print_verbose; }
  void setPrintVerbose(bool state) { print_verbose = state; }

  // returns the multiplicative inverse
  ErrBoundValue multInverse() const;
  
  // overloaded arithmetic operators
  ErrBoundValue operator - ( ) const { return ErrBoundValue(-d, upper, lower, print_verbose); }
  ErrBoundValue operator + ( const ErrBoundValue &p ) const
  { return ErrBoundValue(d+p.d, lower+p.lower, upper+p.upper, print_verbose); }
  ErrBoundValue operator - ( const ErrBoundValue &p            ) const
  { return ErrBoundValue(d-p.d, lower+p.upper, upper+p.lower, print_verbose); }
  ErrBoundValue operator * ( const ErrBoundValue &p ) const;
  ErrBoundValue operator / ( const ErrBoundValue &p ) const;
  ErrBoundValue operator += ( const ErrBoundValue &p )
  { *this = *this + p; return *this; }
  ErrBoundValue operator -= ( const ErrBoundValue &p )
  { *this = *this - p; return *this; }
  ErrBoundValue operator *= ( const ErrBoundValue &p )
  { *this = *this * p; return *this; }
  ErrBoundValue operator /= ( const ErrBoundValue &p )
  { *this = *this / p; return *this; }

  //relational tests: these ignore the error bounds
  bool operator !=( const ErrBoundValue &p) const { return (d != p.d); }
  bool operator ==( const ErrBoundValue &p) const { return (d == p.d); }
  bool operator < ( const ErrBoundValue &p) const { return (d < p.d); }
  bool operator <=( const ErrBoundValue &p) const { return (d <= p.d); }
  bool operator > ( const ErrBoundValue &p) const { return (d > p.d); }
  bool operator >=( const ErrBoundValue &p) const { return (d >= p.d); }

  
private:

  // sets the bounds based on four possible extrema
  void setBoundFromFourPoints(double p1, double p2, double p3, double p4);
  
  double d;
  //this is the upper and lower relative errors: they are always positive
  double lower;
  double upper;

  bool print_verbose;
  
};



#endif
