/* -*- Mode: C++ -*- */

#ifndef _COACH_MESSAGE_QUEUE_H_
#define _COACH_MESSAGE_QUEUE_H_

#include <algorithm>
#include <functional>
#include <list>
#include "clangmsg.h"
#include "clangdefmsg.h"
#include "clangrulemsg.h"
#include "clangdelmsg.h"
#include "clangfreeformmsg.h"
#include "coach_lang_comp.h"
#include "utility.h"

/* An object of this type should be in each module that wants to send coach messages
   The object should then be registered with ModCMRep */
class CoachMessageQueue
{
public:
  template <class T> class ComponentContainer 
  {
  public:
    typedef std::list<T*> Storage;

  public:
    ComponentContainer() : storage() {}
    ~ComponentContainer() { std::for_each(storage.begin(), storage.end(), deleteptr<T>()); }

    void push(T* x) { storage.push_back(x); }
    void push(std::vector<T*>& v) {
      for (typename std::vector<T*>::iterator iter = v.begin(); iter != v.end(); ++iter)
	push(*iter);
    }

    T* pop() { T* x; x = storage.empty() ? NULL : *storage.begin(); storage.pop_front(); return x; }
    T* peek() { return storage.empty() ? NULL : *storage.begin(); }
    typename Storage::const_iterator getBegin() { return storage.begin();}
    typename Storage::const_iterator getEnd() { return storage.end(); }

    friend std::ostream& operator<<(std::ostream& os, const ComponentContainer& c)
    {
      for (typename Storage::const_iterator iter = c.storage.begin();
	   iter != c.storage.end();
	   iter++)
	os << '\t' << **iter << std::endl;
      return os;
    }
    
  private:
    Storage storage;
  };  
  
public:
  CoachMessageQueue(const char* name) : name(name), unique_id(0) {}
  typedef ComponentContainer<rcss::clang::Def>           DefineContainer;
  typedef ComponentContainer<rcss::clang::ActivateRules> RuleContainer;
  typedef ComponentContainer<rcss::clang::RuleIDList>    DelContainer;
  typedef ComponentContainer<rcss::clang::FreeformMsg>   FreeformContainer;

  DefineContainer&   getDefineContainer() { return define; }
  RuleContainer&     getRuleContainer() { return rule; }
  DelContainer&      getDelContainer() { return del; }
  FreeformContainer& getFreeformContainer() { return freeform; }

  std::string getName() { return name; }

  // You can use this to get a unique numerical value for using your advice
  int createUniqueID() { return unique_id++; }
  
  friend std::ostream& operator<<(std::ostream& os, const CoachMessageQueue& q)
  {
    os << "CoachMessageQueue(" << q.name << ")" << std::endl;
    os << "Defines: " << std::endl << q.define;
    os << "Rule: " << std::endl << q.rule;
    os << "Del: " << std::endl << q.del;
    os << "Freeform: " << std::endl << q.freeform;
    return os;
  }
  
  
private:
  // not defined, should never be called
  CoachMessageQueue(const CoachMessageQueue&); 
  // not defined, should never be called
  const CoachMessageQueue& operator=(const CoachMessageQueue&); 

  DefineContainer   define;
  RuleContainer     rule;
  DelContainer      del;
  FreeformContainer freeform;

  std::string name;

  int unique_id;
};

struct CMQDefineExtractor
  : public std::unary_function<CoachMessageQueue*, CoachMessageQueue::ComponentContainer<rcss::clang::Def>&>
{
  CoachMessageQueue::ComponentContainer<rcss::clang::Def>&
  operator() (CoachMessageQueue* p) { return p->getDefineContainer(); }
};
struct CMQRuleExtractor
  : public std::unary_function<CoachMessageQueue*, CoachMessageQueue::ComponentContainer<rcss::clang::ActivateRules>&>
{
  CoachMessageQueue::ComponentContainer<rcss::clang::ActivateRules>&
  operator() (CoachMessageQueue* p) { return p->getRuleContainer(); }
};
struct CMQDelExtractor
  : public std::unary_function<CoachMessageQueue*, CoachMessageQueue::ComponentContainer<rcss::clang::RuleIDList>&>
{
  CoachMessageQueue::ComponentContainer<rcss::clang::RuleIDList>&
  operator() (CoachMessageQueue* p) { return p->getDelContainer(); }
};
struct CMQFreeformExtractor
  : public std::unary_function<CoachMessageQueue*, CoachMessageQueue::ComponentContainer<rcss::clang::FreeformMsg>&>
{
  CoachMessageQueue::ComponentContainer<rcss::clang::FreeformMsg>&
  operator() (CoachMessageQueue* p) { return p->getFreeformContainer(); }
};

void splitIndividualNamesTo(rcss::clang::ActivateRules& actv_rule, CoachMessageQueue::RuleContainer& ruleq);

#endif
