/* -*- Mode: C++ -*- */

/*
    libscoach: Soccer Coach library for use with the SoccerServer system 

    Copyright (C) 2001  Patrick Riley, Paul Carpenter, Gal Kaminka, Manuela Veloso
    Copyright (C) 2000  Patrick Riley, Peter Stone, Manuela Veloso

    This library is free software; you can redistribute it and/or
    modify it under the terms of the GNU Lesser General Public
    License as published by the Free Software Foundation; either
    version 2.1 of the License, or (at your option) any later version.

    This library is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
    Lesser General Public License for more details.

    You should have received a copy of the GNU Lesser General Public
    License along with this library; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

*/

/* This class takes allows you to feed in bits and get out a character string
   from a set of characters which you specify */
 
#ifndef _BINARY_STRING_H_
#define _BINARY_STRING_H_

#include <stream.h>

#define BS_MAX_VALID_CHARS  128
#define BS_MAX_STRING_LEN  1024

/* Here is how to use this thing:
   1. Use SetValidChars to set up what the valid chars for the string are
      The number of characters used will be rounded down to a power of 2
   2. To read a string, 	
    i) use LoadString or LoadPartial (as described below). Watch the return value! 
       if the checksum fails of if there are invalid chars in the string, 
       you will get false 
    ii) use any combination of GetBit, GetInt, GetFloat (see below for more info)
    iii) If you get an error return, either the string did not load correctly, or 
     you ran out of string while processing
   3. To create a string
    i) NewString()
    ii) Use any combination of PutBit, PutInt, PutFloat (see below for more info)
     watch for error returns! Should only happen if the string gets too long or
     you already called CompleteString()
    iii) call CompleteString to get a pointer to the final character string.
     This adds the checksum and that sort of thing.
     Feel free to call this more than once. The class is smart enough to only add 
     the checksum once */

/* to use LoadPartial, do 
   NewString()
   LoadPartial(a);
   LoadPartial(b);
   ...
   LoadPartialComplete();
*/

/* int info 
   You can only put and get positive ints! In other words, you can only put in ints
   between 0 and 2^<num bits> -1
 */

/* float info 
   This is more complicated.
   First a sign bit is put out. 
   Then the integer part of the number is written out (with intbits possible values,
    in other words, any number between 0 and 2^<num bits> -1
   Then the fraction part. What does fracbits mean? basically if fracbits is 3,
    you can represent all the /((2^3)-1) fractions 
      (0/7, 1/7, 2/7, 3/7, 4/7, 5/7, 6/7, 7/7)
 */

/* I might set this up to have a magic number or something at the beginning of the 
   string as additional protection from parsing wrong information. It might also be 
   useful to put some magic numbers in the middle of the string */

/* in each char, high order bits went in first */

/* WARNING: do not mix GetBit and PutBit on the same string. Weird things will
   happen */

class BinaryString 
{
public:
  BinaryString();
  BinaryString(const char* vc);
  //~BinaryString();

  bool SetValidChars(const char* vc);
  
  //returns whether this is a valid string
  bool LoadString(char* new_str);  

  bool LoadPartial(char* part_str);
  bool LoadPartialComplete();  

  void NewString();

  char GetBit();
  bool PutBit(char bit);

  int  GetInt(int num_bits);  
  bool PutInt(int val, int num_bits);  

  bool GetFloat(float* pval, int intbits, int fracbits);  
  bool  PutFloat(float val, int intbits, int fracbits);  

  char* CompleteString();
  
  void testIndexing(ostream& out);
  
private:
  void Reset();
  
  //this array will be sorted
  char valid_chars[BS_MAX_VALID_CHARS+1];

  char char_to_idx[255];
  int  bits_per_char; //the effective number of bits per char

  char current_str[BS_MAX_STRING_LEN];  
  int current_str_len;  //this will be -1 if we are loading a partial
  int next_char;  

  bool string_complete;

  char partial_bits;
  int num_partial_bits;
  
  int checksum;  

  int num_trailing_bits; //the number of left over bits  

  char idx_to_valid_char(int idx);
  int  valid_char_to_idx(char ch);
  bool load_helper(int len);  
  
};



#endif
