/* -*- Mode: C++ -*- */

/*
    libscoach: Soccer Coach library for use with the SoccerServer system 

    Copyright (C) 2001 Patrick Riley, Paul Carpenter, Gal Kaminka, Manuela Veloso
    Copyright (C) 2000 Patrick Riley, Peter Stone, Manuela Veloso

    This library is free software; you can redistribute it and/or
    modify it under the terms of the GNU Lesser General Public
    License as published by the Free Software Foundation; either
    version 2.1 of the License, or (at your option) any later version.

    This library is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
    Lesser General Public License for more details.

    You should have received a copy of the GNU Lesser General Public
    License along with this library; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

*/

/* This class takes allows you to feed in bits and get out a character string
   from a set of characters which you specify */

#include <string.h>
#include <iostream.h>
#include <math.h>
#include "misc.h"
#include "BinaryString.h"

#define BITS_PER_CHAR 8

//#define ERROR_MESSAGE(x) x
#define ERROR_MESSAGE(x)

BinaryString::BinaryString()
{
  Reset();  
}

BinaryString::BinaryString(const char* vc)
{
  Reset();
  SetValidChars(vc);  
}


void BinaryString::Reset() 
{
  valid_chars[0]= 0;  
  for (int i=0; i<255; i++)
    char_to_idx[i] = -1;
  
  current_str[0] = 0;  
  next_char = 0;  

  string_complete = false;  

  partial_bits = 0;
  num_partial_bits = 0;  

  checksum = 0;  

  num_trailing_bits = 0;  
}


bool BinaryString::SetValidChars(const char* vc)
{
  Reset();
  

  int idx;  
  int len = strlen(vc);
  if (len > BS_MAX_VALID_CHARS) {
    ERROR_MESSAGE(cerr << "BS: too many valid chars\n");
    return false;
  }
  
  //sort the vc string into valid_chars
  //this is a really stupid slow way to do this
  idx = 0;  
  for (int i=1; i<255; i++) {
    if (strchr(vc, i)) {
      valid_chars[idx] = i;
      char_to_idx[i] = idx;
      idx++;
    } else {
      char_to_idx[i] = -1;
    }
  }
  valid_chars[idx] = 0;

  //set bits per char
  bits_per_char = (int)(floor(log(strlen(valid_chars))/log(2))+.1);  

  //some debug output
  /*
  cout << "valid: " << valid_chars << endl;
  cout << "min: " << min_valid << "\tmax: " << max_valid << endl;
  cout << "excl: " << exclusion_list << endl;  
  */

  return true;  
}

  
bool BinaryString::LoadString(char* new_str)
{
  int len = strlen(new_str);
  if (len > BS_MAX_STRING_LEN) {
    ERROR_MESSAGE(cerr << "BS:LoadString: too long\n");
    return false;
  }
  
  //copy the string
  memcpy(current_str, new_str, len+1); //get the null

  return load_helper(len);  
}

bool BinaryString::LoadPartial(char* part_str)
{
  current_str_len = -1; //indicate that we're loading partial   
  int curr_len = strlen(current_str);  
  int part_len = strlen(part_str);  
  if (curr_len+part_len > BS_MAX_STRING_LEN) {
    ERROR_MESSAGE(cerr << "BS:LoadPartial: too long\n");    
    return false;  
  }
  
  //copy the string
  memcpy(current_str+curr_len, part_str, part_len+1); //get the null
  
  return true;  
}

bool BinaryString::LoadPartialComplete()
{
  int len = strlen(current_str);
  return load_helper(len);  
}



//the only thing that should be done before calling this is copying in
//the string; don't set current_str_len
bool BinaryString::load_helper(int len)
{
  partial_bits = 0;
  num_partial_bits = 0;
  string_complete = false;
  next_char = 0;  
  num_trailing_bits = 0;  

  //check that we only actually use the characters that we do
  //we round down to the nearest power of 2 and only use those characters
  int max_idx = (2 << bits_per_char) - 1;
  for (int i=0; i < len; i++) {
    int idx = valid_char_to_idx(current_str[i]);    
    if (idx > max_idx || idx == -1) {
      ERROR_MESSAGE(cerr << "BS:load_helper: out of range character" << 
		    idx << " " << i << " " << current_str[i] << endl);
      return false;    
    }    
  }
  
  //do checksum
  checksum = valid_char_to_idx(current_str[len-1]);
  int test_checksum=0;
  for (int i=0; i<len-2; i++) {
    test_checksum ^= valid_char_to_idx(current_str[i]);    
  }
  if (checksum != test_checksum) {
    ERROR_MESSAGE(cerr << "BS:load_helper: checksum failed '" << current_str << "'" << endl);
    current_str[0] = 0;
    current_str_len = 0;    
    return false;    
  }
  current_str[len--] = 0; //cut off the checksum character
  

  //get the number of trailing bits, this is now the last char in the string
  num_trailing_bits = valid_char_to_idx(current_str[len-1]);
  current_str[len--] = 0; //cut off the trailing bits character
  
  current_str_len = len;  

  return true;  
}

void BinaryString::NewString()
{
  current_str[0] = 0;
  current_str_len = 0;
  partial_bits = 0;
  num_partial_bits = 0;
  string_complete = false;
  next_char = 0;  
  num_trailing_bits = 0;
  checksum = 0;  
}



//remember that we have to pull the high order bits off first
char BinaryString::GetBit()
{
  if (next_char >= current_str_len && num_partial_bits == 0) {    
    ERROR_MESSAGE(cerr << "BS:GetBit: out of bits\n");
    return -1; //out of bits
  }
  
  char bit;  

  if (num_partial_bits == 0) {
    partial_bits = valid_char_to_idx(current_str[next_char++]);    
    if (next_char == current_str_len && num_trailing_bits > 0) {
      //this was the last char
      num_partial_bits = num_trailing_bits;    
      partial_bits <<= (BITS_PER_CHAR - num_trailing_bits);    
    } else {
      num_partial_bits = bits_per_char;    
      partial_bits <<= (BITS_PER_CHAR - bits_per_char);    
    }
  }
  
  bit = !!(partial_bits & (1<<(BITS_PER_CHAR-1)));
  partial_bits <<= 1;
  num_partial_bits--;
   
  return bit;  
}



bool BinaryString::PutBit(char bit)
{
  if (string_complete || current_str_len < 0) {
    ERROR_MESSAGE(cerr << "BS:PutBit: complete or loading partial\n");
    return false;
  }
  
  if (next_char >= BS_MAX_STRING_LEN) {
    ERROR_MESSAGE(cerr << "BS:PutBit: too long\n");
    return false;  
  }
  

  partial_bits = (partial_bits << 1) | (!!bit); //logical nots inser 1 or 0
  num_partial_bits++;
  
  if (num_partial_bits == bits_per_char) {
    current_str[next_char++] = idx_to_valid_char(partial_bits);
    checksum ^= partial_bits;    
    num_partial_bits = 0;
    partial_bits = 0;
  }

  return true;  
}


int  BinaryString::GetInt(int num_bits)
{
  int val = 0;  
  for (int i=0; i<num_bits; i++) {
    char bit = GetBit();
    if (bit == -1)
      return -1;
    val = (val << 1) | bit;
  }

  return val;
}

bool BinaryString::PutInt(int val, int num_bits)
{
  for (int i=num_bits-1; i>=0; i--) {
    if (!PutBit(!!(val & (1<<i))))
      return false;
  }
  return true;  
}

bool BinaryString::GetFloat(float* pval, int intbits, int fracbits)
{
  int intval;
  float sign;

  //sign bit
  if ( (intval = GetBit()) == -1 )
    return false;
  sign = (intval) ? 1.0 : -1.0;
  
  //integer part
  if ( (intval = GetInt(intbits)) == -1 )
    return false;
  *pval = intval;

  //fractional part
  if ( (intval = GetInt(fracbits)) == -1 )
    return false;
  *pval += ((float)intval) / ~((~0) << fracbits);
  
  *pval *= sign;
  
  return true;
}

bool BinaryString::PutFloat(float val, int intbits, int fracbits)
{
  //sign bit
  if (!PutBit( (val >= 0) ? 1 : 0))
    return false;
  
  //integer part
  val = fabs(val);
  if (!PutInt( (int)floor(val), intbits ))
    return false;

  //fractional part
  if (!PutInt( roundToInt((val - floor(val)) * ~((~0) << fracbits)), fracbits))
    return false;
  
  return true;  
}




char* BinaryString::CompleteString()
{
  if (string_complete == true)
    return current_str;
  
  string_complete = true;
  num_trailing_bits = num_partial_bits;
  
  if (num_trailing_bits > 0) {
    checksum ^= partial_bits;    
    current_str[next_char++] = idx_to_valid_char(partial_bits);
  }
  
  current_str[next_char++] = idx_to_valid_char(num_trailing_bits);

  //add checksum
  current_str[next_char++] = idx_to_valid_char(checksum);

  current_str[next_char] = 0; //null term the string
  
  return current_str;  
}



char BinaryString::idx_to_valid_char(int idx)
{
  return valid_chars[idx];
}

int  BinaryString::valid_char_to_idx(char ch)
{
  return (int)char_to_idx[ch];
}

  
void BinaryString::testIndexing(ostream& out)
{
  int num_valid = strlen(valid_chars);
  
  out << "idx\tto valid\tto idx" << endl;
  for (int i=0; i<num_valid; i++) {
    out << i << "\t"
        << idx_to_valid_char(i) << "\t"
        << valid_char_to_idx(idx_to_valid_char(i)) << endl;
  }
  
}

