/* These classes help in manipulating binary files */

#include <netinet/in.h>
#include "BinaryFile.h"
#include "Logger.h"

using namespace spades;


/**************************************************************************/

BinaryFileWriter::BinaryFileWriter(const char* fn)
  : poutfile(new std::ofstream(fn, std::ios::out | std::ios::binary)), out(*poutfile)
{
}

BinaryFileWriter::BinaryFileWriter(std::ostream& out)
  : poutfile(NULL), out(out)
{
}

BinaryFileWriter::~BinaryFileWriter()
{
  out.flush();
  delete poutfile;
}

// assume a null term string
bool
BinaryFileWriter::writeMagicHeader(const char* header)
{
  return writeString(header, strlen(header)); 
}


bool
BinaryFileWriter::writeChar(unsigned char c)
{
  out.write(reinterpret_cast<char*>(&c), sizeof(char));
  return out.good();
}

bool
BinaryFileWriter::writeShort(unsigned short s)
{
  s = htons(s);
  out.write(reinterpret_cast<char*>(&s), sizeof(short));
  return out.good();
}

bool
BinaryFileWriter::writeInt(unsigned int i)
{
  i = htonl(i);
  out.write(reinterpret_cast<char*>(&i), sizeof(int));
  return out.good();
}

bool
BinaryFileWriter::writeLong(unsigned long l)
{
  l = htonl(l);
  out.write(reinterpret_cast<char*>(&l), sizeof(long));
  return out.good();
}

bool
BinaryFileWriter::writeFloat(float f)
{
  out.write(reinterpret_cast<char*>(&f), sizeof(float));
  return out.good();
}

bool
BinaryFileWriter::writeDouble(double d)
{
  out.write(reinterpret_cast<char*>(&d), sizeof(double));
  return out.good();
}

bool
BinaryFileWriter::writeIntAsShort(int i)
{
  if (i > std::numeric_limits<short>::max() || i < std::numeric_limits<short>::min())
    errorlog << "BinaryFileWriter::writeIntAsShort: truncation of " << i << ende;
  return writeShort(static_cast<short>(i));
}

bool
BinaryFileWriter::writeIntAsChar(int i)
{
  if (i > std::numeric_limits<char>::max() || i < std::numeric_limits<char>::min())
    errorlog << "BinaryFileWriter::writeIntAsChar: truncation of " << i << ende;
  return writeChar(static_cast<char>(i));
}


bool
BinaryFileWriter::writeString(const char* s, unsigned len)
{
  out.write(s, len);
  return out.good();
}


/**************************************************************************/

BinaryFileReader::BinaryFileReader(const char* fn)
  : pinfile(new std::ifstream(fn, std::ios::in | std::ios::binary)), in(*pinfile)
{
}

BinaryFileReader::BinaryFileReader(std::istream& in)
  : pinfile(NULL), in(in)
{
}
BinaryFileReader::~BinaryFileReader()
{
  delete pinfile;
}

bool
BinaryFileReader::checkMagicHeader(const char* header)
{
  for (;
       *header != 0;
       header++)
    {
      char c;
      in.get(c);
      if (in.fail())
	return false;
      if (c != *header)
	return false;
    }
  return true;
}


bool
BinaryFileReader::readChar(char* pc)
{
  in.get(*pc);
  return in.good();
}

bool
BinaryFileReader::readShort(unsigned short* ps)
{
  in.read(reinterpret_cast<char*>(ps), sizeof(short));
  *ps = ntohs(*ps);
  return in.good();
}

bool
BinaryFileReader::readInt(unsigned int *pi)
{
  in.read(reinterpret_cast<char*>(pi), sizeof(int));
  *pi = ntohl(*pi);
  return in.good();
}

bool
BinaryFileReader::readLong(unsigned long *pl)
{
  in.read(reinterpret_cast<char*>(pl), sizeof(long));
  *pl = ntohl(*pl);
  return in.good();
}

bool
BinaryFileReader::readFloat(float *pf)
{
  in.read(reinterpret_cast<char*>(pf), sizeof(float));
  return in.good();
}

bool
BinaryFileReader::readDouble(double *pd)
{
  in.read(reinterpret_cast<char*>(pd), sizeof(double));
  return in.good();
}


int
BinaryFileReader::readString(char* s, unsigned buffer_size)
{
  for(unsigned i=0; i<buffer_size; i++)
    {
      char c;
      in.get(c);
      if (in.fail())
	return i;
      s[i] = c;
      if (c==0)
	return i;
    }
  return buffer_size;
}

bool
BinaryFileReader::readString(std::string* ps)
{
  ps->clear();
  for (;;)
    {
      char c;
      in.get(c);
      if (in.fail())
	return false;
      if (c==0)
	return true;
      ps->push_back(c);
    }
  errorlog << "BinaryFileReader::readString: How did I get here?" << ende;
  return false;
}

