/* -*- Mode: C++ -*- */

#ifndef _ADVICE_TREE_H_
#define _ADVICE_TREE_H_

#include <vector>
#include <map>
#include "clangaction.h"
#include "cond.h"
#include "coach_lang_comp.h"

class AbstractState;
class AbstractStateFactorLeaf;
class CoachMessageQueue;

/********************************************************************************/

class AdviceTree;
class AdviceTreeAction;
class AdviceTreeNode;
class AdviceTreeNodeInternal;
class AdviceTreeNodeLeaf;

/********************************************************************************/

class AdviceTree 
{
public:
  class LevelInfo
  {
  public:
    // Does not take over factor memory
    LevelInfo(AbstractStateFactorLeaf* pfac, int factor_idx)
      : pfac(pfac), factor_idx(factor_idx) {}
    LevelInfo()
      : pfac(NULL), factor_idx(-1) {}
    ~LevelInfo() {}

    AbstractStateFactorLeaf* pfac;
    int factor_idx;
  };

  typedef std::vector<LevelInfo> LevelStorage;
  // maps abstract state indices into the relevant rules
  // It is completely valid of the states to have no associated rules
  typedef std::map<int, std::string> AssocMap;

public:
  AdviceTree();
  ~AdviceTree();

  void clear();

  void addLevel(const LevelInfo& l) { levels.push_back(l); }

  void createTreeFromLevels();
  
  // Does take over the action memeory, but not the state
  void addAction(AbstractState* pstate, AdviceTreeAction* pact);

  //returns the name of the top level rule
  // empty on error
  std::string createAdvice(CoachMessageQueue* pqueue, const std::string& prefix,
			   std::ostream* pos_assoc = NULL);

  const std::string& getRuleForState(int state_idx) const;

  //randomly shuffles all actions in the tree
  void shuffleActions();
  
  friend std::ostream& operator<<(std::ostream& os, const AdviceTree& tree);
  
private:
  AdviceTreeNode* createTree(LevelStorage::iterator ilevel);
  
  AdviceTreeNode* root;
  LevelStorage levels;

  AssocMap assoc_map;
  
};

/********************************************************************************/

class AdviceTreeAction
{
public:
  typedef std::list<rcss::clang::Cond*> CondStorage;
  typedef std::list<rcss::clang::Dir*>  DirStorage;
  
public:
  AdviceTreeAction() : conds(), dirs() {}
  ~AdviceTreeAction() { clear(); }

  void clear();

  /* You need to be careful adding conditions here! You could add conditions
     which cause the actions to no longer have an effect.
     I added this first just to get variable bindings done correctly */
  CondStorage& getConds() { return conds; }
  const CondStorage& getConds() const { return conds; }
  void addCondition(rcss::clang::Cond* p);
  
  DirStorage& getDirs() { return dirs; }
  const DirStorage& getDirs() const { return dirs; }
  void addDirective(rcss::clang::Dir* p);

  void addCondsTo(rcss::clang::CondAnd* pAnd);
  void addDirsTo(rcss::clang::SimpleRule::Storage& store);

  friend std::ostream& operator<<(std::ostream& os, const AdviceTreeAction& a);

private:
  CondStorage conds;
  DirStorage  dirs;
};

/********************************************************************************/

class AdviceTreeNode
{
public:
  AdviceTreeNode() : marked(false) {}
  virtual ~AdviceTreeNode() {}

  void setMark(bool b = true) { marked = b; }
  bool isMarked() const { return marked; }

  void addAction(AbstractState* pstate, AdviceTreeAction* pact,
		 AdviceTree::LevelStorage& levels, int depth);

  //returns the name of the top level rule, "" if no rules
  virtual std::string createAdvice(CoachMessageQueue* pqueue, const std::string& prefix,
				   AdviceTree::LevelStorage& levels, int depth,
				   std::ostream* pos_assoc,
				   AdviceTree::AssocMap* p_assoc_map) = 0;

  //recurses down the tree, adding all leaves to vleaves
  virtual void findLeaves(std::vector<AdviceTreeNodeLeaf*>& vleaves,
			  bool require_marked) = 0;
  
  virtual void print(std::ostream& os, const std::string& prefix,
		     const AdviceTree::LevelStorage& levels, int depth) const = 0;
  
protected:
  virtual void protAddAction(AbstractState* pstate, AdviceTreeAction* pact,
			     AdviceTree::LevelStorage& levels, int depth) = 0;
  
private:
  bool marked;
  
};

/********************************************************************************/

class AdviceTreeNodeInternal
  : public AdviceTreeNode
{
public:
  AdviceTreeNodeInternal() : AdviceTreeNode() {}
  ~AdviceTreeNodeInternal();

  // takes over the memory
  void addChild(AdviceTreeNode* p) { children.push_back(p); }
  
  //returns the name of the top level rule, "" if no rules
  std::string createAdvice(CoachMessageQueue* pqueue, const std::string& prefix,
			   AdviceTree::LevelStorage& levels, int depth,
			   std::ostream* pos_assoc,
			   AdviceTree::AssocMap* p_assoc_map);

  void findLeaves(std::vector<AdviceTreeNodeLeaf*>& vleaves, bool require_marked);

  void print(std::ostream& os, const std::string& prefix,
	     const AdviceTree::LevelStorage& levels, int depth) const;

protected:
  void protAddAction(AbstractState* pstate, AdviceTreeAction* pact,
		     AdviceTree::LevelStorage& levels, int depth);
  
private:
  typedef std::vector<AdviceTreeNode*> ChildrenStorage;

  ChildrenStorage children;
};

/********************************************************************************/

class AdviceTreeNodeLeaf
  : public AdviceTreeNode
{
public:
  AdviceTreeNodeLeaf() : AdviceTreeNode(), actions(), relevant_states() {}
  ~AdviceTreeNodeLeaf();

  //does take over the memory
  void addAdvice(AdviceTreeAction* p) { actions.push_back(p); }
  
  //returns the name of the top level rule, "" if no rules
  std::string createAdvice(CoachMessageQueue* pqueue, const std::string& prefix,
			   AdviceTree::LevelStorage& levels, int depth,
			   std::ostream* pos_assoc,
			   AdviceTree::AssocMap* p_assoc_map);

  void findLeaves(std::vector<AdviceTreeNodeLeaf*>& vleaves, bool require_marked);

  void print(std::ostream& os, const std::string& prefix,
	     const AdviceTree::LevelStorage& levels, int depth) const;

  void swapActions(AdviceTreeNodeLeaf& other_leaf);
  
protected:
  void protAddAction(AbstractState* pstate, AdviceTreeAction* pact,
		     AdviceTree::LevelStorage& levels, int depth);
  
private:
  typedef std::vector<AdviceTreeAction*> ActionStorage;
  typedef std::set<int> StateIdxStorage;
  
  ActionStorage actions;
  //stores all the states that this node has advice from
  StateIdxStorage relevant_states; 
};

/********************************************************************************/



#endif
