/* -*- Mode: C++ -*- */

/* These are classes to represent pieces of the abstract state space */

#ifndef _ABSTRACT_STATE_ELEMENTS_H_
#define _ABSTRACT_STATE_ELEMENTS_H_

#include <iostream>
#include <fstream>
#include <vector>
#include "AbstractState.h"
#include "WorldState.h"
#include "PlayerOccupancy.h"

class ModFeatures;

/***********************************************************************************/

class PlayModeFilter
  : public AbstractStateFilter
{
public:
  PlayModeFilter(PlayMode valid_pm) : AbstractStateFilter(), valid_pm(valid_pm) {}
  ~PlayModeFilter() {}

  bool isValid(const WorldState& ws, TeamSide my_side)
  { return ws.getPlayMode() == valid_pm; }

  void print(std::ostream& os) const
  { os << "PlayModeFilter(" << valid_pm << ")"; }
  
  rcss::clang::Cond* createCondition();
  
private:
  PlayMode valid_pm;
};


/***********************************************************************************/

class BallKickableFilter
  : public AbstractStateFilter
{
public:
  BallKickableFilter(ModFeatures* pmFeatures);
  ~BallKickableFilter() {}

  bool isValid(const WorldState& ws, TeamSide my_side);

  void print(std::ostream& os) const
  { os << "BallKickableFilter()"; }

  void createInitialAdvice(CoachMessageQueue& mqueue);
  
  rcss::clang::Cond* createCondition();

private:
  ModFeatures* pmFeatures;

private:
  static const char* COND_NAME;
  static const char* REG_NAME;
};

/***********************************************************************************/

class POElementFactor
  : public AbstractStateFactorLeaf
{
public:
  POElementFactor(const PlayerOccupancyElement& elem)
    : AbstractStateFactorLeaf(ASF_POElement), elem(elem), my_unique_id(-1) {}
  ~POElementFactor() {}

  // Returns the number of states in this factor
  int getNumStates() { return elem.getNumIdx(); }

  // Sets elements in the AbstractState for this factor
  // returns the next factor idx to use
  int setStateFor(AbstractState* pstate, int first_idx,
		  const WorldState& ws, TeamSide my_side);
  
  void print(std::ostream& os) const
  { os << "POElementFactor:" << elem; }

  int draw(FieldImage* pimg, RegionWorldModelInterface* wmi,
	   const AbstractState* pstate, int first_idx);

  void createInitialAdvice(CoachMessageQueue& mqueue);

  rcss::clang::Cond* createCondition(int my_val);

private:
  PlayerOccupancyElement elem;
  // an ID to avoid name collisions since we will have many POElements
  // established at the createInitialAdvice
  int my_unique_id; 
  
};

/***********************************************************************************/

class POSetFactor
  : public AbstractStateFactorLeaf
{
public:
  POSetFactor(const PlayerOccupancySet& poset)
    : AbstractStateFactorLeaf(ASF_POSet), poset(poset), state(&(this->poset)) {}
  ~POSetFactor() {}

  // Returns the number of states in this factor
  int getNumStates() { return poset.getNumIdx(); }

  // Sets elements in the AbstractState for this factor
  // returns the next factor idx to use
  int setStateFor(AbstractState* pstate, int first_idx,
		  const WorldState& ws, TeamSide my_side);
  
  void print(std::ostream& os) const
  { os << "POSetFactor:" << poset; }

  int draw(FieldImage* pimg, RegionWorldModelInterface* wmi,
	   const AbstractState* pstate, int first_idx);

  void createInitialAdvice(CoachMessageQueue& mqueue);

  rcss::clang::Cond* createCondition(int my_val);

private:
  PlayerOccupancySet poset;
  // it's mutable so we can futz with it in a const method
  // since it doesn't really change the meaning
  mutable PlayerOccupancyState state; // so that we don't have to keep realloc
};

/***********************************************************************************/

//The grid will be centered at (-0.5, -0.5) and 1 m larger than the field
class BallGridFactor
  : public AbstractStateFactorLeaf
{
public:
  BallGridFactor(int x_bins, int y_bins);
  ~BallGridFactor() {}

  // Returns the number of states in this factor
  int getNumStates() { return num_states; }
  // Sets elements in the AbstractState for this factor
  // returns the next factor idx to use
  int setStateFor(AbstractState* pstate, int first_idx,
		  const WorldState& ws, TeamSide my_side);

  void print(std::ostream& os) const
  { os << "BallGridFactor(x=" << x_bins << ", y=" << y_bins << ")"; }
  
  int draw(FieldImage* pimg, RegionWorldModelInterface* wmi,
	   const AbstractState* pstate, int first_idx);

  // draw a numbered grid on the field
  void drawEntireGrid(FieldImage* pimg,
		      FieldImage::Color cborder = FieldImage::COLOR_GREY,
		      FieldImage::Color cinside = FieldImage::COLOR_CLEAR);
  
  bool extractBallPos(VecPosition* pv, const AbstractState* pstate, int *pfirst_idx);

  void createInitialAdvice(CoachMessageQueue& mqueue);

  rcss::clang::Cond* createCondition(int my_val);

  Rectangle getGridRectangle(int grid_idx) const;
  // returns false if grid_idx invalid

  bool decodeGridPos(int grid_idx, int* px, int* py) const;

  bool isLocBetween(int gidx, int gidx_pt1, int gidx_pt2);
  
  std::string getGridName(int gidx);
  
private:
  void updateNumStates() { num_states = x_bins * y_bins; }

  int x_bins;
  int y_bins;
  double x_size;
  double y_size;
  int num_states;

private:
  static const char* REG_PREFIX;
  class SentRegElem
  {
  public:
    SentRegElem(int x_bins = -1, int y_bins = -1) : x_bins(x_bins), y_bins(y_bins) {}

    bool operator< (const SentRegElem& e) const
    {
      if (x_bins < e.x_bins) return true;
      if (x_bins > e.x_bins) return false;
      return (y_bins < e.y_bins);
    }
  private:
    int x_bins;
    int y_bins;
  };
  
  typedef std::set<SentRegElem> SentRegStorage;
  
  static SentRegStorage s_sent_reg_storage;
};

/***********************************************************************************/

class BallOwnerFactor
  : public AbstractStateFactorLeaf
{
public:
  enum State {
    ST_Invalid = -1,
    ST_Other,
    ST_Mine,
    ST_Theirs,
    ST_Max
  };

public:
  BallOwnerFactor(ModFeatures* pmFeatures);
  ~BallOwnerFactor() {}

  // Returns the number of states in this factor: other, left, right
  int getNumStates() { return (int)ST_Max; }
  // Sets elements in the AbstractState for this factor
  // returns the next factor idx to use
  // 0: Other, 1:mine, 2:theirs
  int setStateFor(AbstractState* pstate, int first_idx,
		  const WorldState& ws, TeamSide my_side);
  
  void print(std::ostream& os) const
  { os << "BallOwnerFactor()"; }

  int draw(FieldImage* pimg, RegionWorldModelInterface* wmi,
	   const AbstractState* pstate, int first_idx);

  rcss::clang::Cond* createCondition(int my_val);

  // we use this to remove the BallOwner factor from the advice tree
  void addAdviceTreeLevels(AdviceTree* ptree, int* pfirst_idx)
  { (*pfirst_idx)++; }

private:
  ModFeatures* pmFeatures;
};

/***********************************************************************************/

/* 2 states: I just scored, they just scored
   -1 otherwise */
class GoalFactor
  : public AbstractStateFactorLeaf
{
public:
  enum State {
    ST_Invalid = -1,
    ST_Mine,
    ST_Theirs,
    ST_Max
  };

public:
  GoalFactor() : AbstractStateFactorLeaf(ASF_Goal) {}
  ~GoalFactor() {}

  // Returns the number of states in this factor: other, left, right
  int getNumStates() { return (int)ST_Max; }
  // Sets elements in the AbstractState for this factor
  // returns the next factor idx to use
  // 0:mine, 1:theirs
  int setStateFor(AbstractState* pstate, int first_idx,
		  const WorldState& ws, TeamSide my_side);
  
  void print(std::ostream& os) const
  { os << "GoalFactor()"; }

  int draw(FieldImage* pimg, RegionWorldModelInterface* wmi,
	   const AbstractState* pstate, int first_idx);

  rcss::clang::Cond* createCondition(int my_val);

};

/***********************************************************************************/

/* 2 states: My dead ball kick, their dead ball kick
   -1 otherwise */
class DeadBallFactor
  : public AbstractStateFactorLeaf
{
public:
  enum State {
    ST_Invalid = -1,
    ST_Mine,
    ST_Theirs,
    ST_Max
  };

public:
  DeadBallFactor() : AbstractStateFactorLeaf(ASF_DeadBall) {}
  ~DeadBallFactor() {}

  // Returns the number of states in this factor: other, left, right
  int getNumStates() { return (int)ST_Max; }
  // Sets elements in the AbstractState for this factor
  // returns the next factor idx to use
  // 0:mine, 1:theirs
  int setStateFor(AbstractState* pstate, int first_idx,
		  const WorldState& ws, TeamSide my_side);
  
  void print(std::ostream& os) const
  { os << "DeadBallFactor()"; }

  int draw(FieldImage* pimg, RegionWorldModelInterface* wmi,
	   const AbstractState* pstate, int first_idx);

  void createInitialAdvice(CoachMessageQueue& mqueue);

  rcss::clang::Cond* createCondition(int my_val);

private:
  static const char* MINE_COND_NAME;
  static const char* THEIRS_COND_NAME;
};

/***********************************************************************************/

class ConstantFactor
  : public AbstractStateFactorLeaf
{
public:
  ConstantFactor(int num_states = 1, int val = 0);
  ~ConstantFactor() {}

  int getNumStates() { return num_states; }
  void setNumStates(int n) { num_states = n; }

  int getConstantReturn() { return val; }
  void setConstantReturn(int v) { val = v; }
  
  // Sets elements in the AbstractState for this factor
  // returns the next factor idx to use
  int setStateFor(AbstractState* pstate, int first_idx,
		  const WorldState& ws, TeamSide my_side);
  
  void print(std::ostream& os) const
  { os << "ConstantFactor(" << num_states << ", " << val << ")"; }

  int draw(FieldImage* pimg, RegionWorldModelInterface* wmi,
	   const AbstractState* pstate, int first_idx);

  rcss::clang::Cond* createCondition(int my_val);

private:
  int num_states;
  int val;
};
#endif
