/* -*- Mode: C++ -*- */

#include <sstream>
#include "AbstractStateElements.h"
#include "ServerParam.h"
#include "CoachParam.h"
#include "ModFeatures.h"
#include "Logger.h"

using namespace spades;

/*********************************************************************/
rcss::clang::Cond*
PlayModeFilter::createCondition()
{
  // Hello; BIG HACK SMURF
  // I actually only us PlayOn mode, so we'll just hack this to be play on
  // Really, the PlayMode here should be a clang play mode
  if (valid_pm != PM_PlayOn)
    errorlog << "PlayModeFilter: I only support PM_PlayOn; I'm a BIG HACK" << ende;
  return new rcss::clang::CondPlayMode(rcss::clang::PlayOn);
}

/*********************************************************************/

const char* BallKickableFilter::COND_NAME = "ASF_BallKickable";
const char* BallKickableFilter::REG_NAME = "ASF_BallKickableReg";

BallKickableFilter::BallKickableFilter(ModFeatures* pmFeatures)
  : AbstractStateFilter(),
    pmFeatures(pmFeatures)
{
}

bool
BallKickableFilter::isValid(const WorldState& ws, TeamSide my_side)
{
  if (!pmFeatures)
    {
      errorlog << "BallKickableFilter: need real pmFeatures" << ende;
      return false;
    }  
  if (pmFeatures->getLastStateUpdateTime() != ws.getTime())
    errorlog << "BallKickableFilter: I don't think ModFeatures is in sync with me! "
	     << pmFeatures->getLastStateUpdateTime() << ' ' << ws.getTime() << ende;
  return !pmFeatures->getBallPossessors().empty();
}

void
BallKickableFilter::createInitialAdvice(CoachMessageQueue& mqueue)
{
  rcss::clang::RegArc* pBallReg =
    new rcss::clang::RegArc( std::auto_ptr<rcss::clang::Point>(new rcss::clang::PointBall),
			     0,
			     ServerParam::instance()->getSPKickableArea() +
			     CoachParam::instance()->getAbstractBallKickableFilterBuffer(),
			     0, 360);
  mqueue.getDefineContainer().push(new rcss::clang::DefReg(REG_NAME,
							   std::auto_ptr<rcss::clang::Region>(pBallReg)));
  
  rcss::clang::CondOr* pOr = new rcss::clang::CondOr();

  rcss::clang::CondPlayerPos* pCond;
  rcss::clang::UNumSet uset;
  uset.add(rcss::clang::UNum::uAll);
  
  // check for ourside and then their side
  pCond = new rcss::clang::CondPlayerPos(true, uset, 1, 11,
					 std::auto_ptr<rcss::clang::Region>(new rcss::clang::RegNamed(REG_NAME)));
  pOr->push_back(std::auto_ptr<rcss::clang::Cond>(pCond));

  pCond = new rcss::clang::CondPlayerPos(false, uset, 1, 11,
					 std::auto_ptr<rcss::clang::Region>(new rcss::clang::RegNamed(REG_NAME)));
  pOr->push_back(std::auto_ptr<rcss::clang::Cond>(pCond));
  
  mqueue.getDefineContainer().push(new rcss::clang::DefCond(COND_NAME,
							    std::auto_ptr<rcss::clang::Cond>(pOr)));
}

rcss::clang::Cond*
BallKickableFilter::createCondition()
{
  if (CoachParam::instance()->getAbstractSendBallKickableFilter())
    return new rcss::clang::CondNamed(COND_NAME);
  return NULL;
}


/*********************************************************************/
int
POElementFactor::setStateFor(AbstractState* pstate, int first_idx,
			     const WorldState& ws, TeamSide my_side)
{
  pstate->setFactorIdx(first_idx, elem.getIdxForWorldState(ws, my_side));
  pstate->setValid(true);
  return first_idx + 1;
}

int
POElementFactor::draw(FieldImage* pimg, RegionWorldModelInterface* wmi,
		      const AbstractState* pstate, int first_idx)
{
  int myval = pstate->getFactorIdx(first_idx);
  actionlog(150) << "POElemFactor: drawing with my val of " << myval
		 << " at idx " << first_idx << " of state: " << *pstate << ende;
  if (myval == -1)
    return first_idx + 1;
  VarBindings bindings; // just a placeholder
  FieldImage::Color c_border;
  FieldImage::Color c_inside;
  PlayerOccupancyState::getColorsFor(myval, elem.isAMySide(),
				     &c_border, &c_inside);
  elem.draw(pimg, false, c_border, c_inside, wmi, bindings);
  return first_idx + 1;
}

void
POElementFactor::createInitialAdvice(CoachMessageQueue& mqueue)
{
  if (my_unique_id != -1)
    errorlog << "POElementFactor::createInitialAdvice: createInitialAdvice seems to have been called twice, I already had a unique_id: " << my_unique_id << ende;

  my_unique_id = mqueue.createUniqueID();
  elem.createInitialAdvice(mqueue, my_unique_id);
}


rcss::clang::Cond*
POElementFactor::createCondition(int myval)
{
  if (myval == -1)
    return NULL;
  return elem.createCondition(my_unique_id, myval);
}


/*********************************************************************/

// Sets elements in the AbstractState for this factor
// returns the next factor idx to use
int
POSetFactor::setStateFor(AbstractState* pabs_state, int first_idx,
			 const WorldState& ws, TeamSide my_side)
{
  if (!poset.getPOStateForWorldState(ws, my_side, &state))
    {
      errorlog << "POSetFactor: getPOState failed" << ende;
      pabs_state->setFactorIdx(first_idx, -1);
      pabs_state->setValid(false);
    }
  else
    {
      pabs_state->setFactorIdx(first_idx, state.getOverallIdx());
      pabs_state->setValid(true);
    }
  return first_idx + 1;
}

int
POSetFactor::draw(FieldImage* pimg, RegionWorldModelInterface* wmi,
		  const AbstractState* pstate, int first_idx)
{
  int myval = pstate->getFactorIdx(first_idx);
  actionlog(150) << "POSetFactor: drawing with my val of " << myval
		 << " at idx " << first_idx << " of state: " << *pstate << ende;
  if (myval == -1)
    return first_idx + 1;
  state.setOverallIdx(myval);
  VarBindings bindings; // we just use empty bindings
  state.draw(pimg, false, wmi, bindings);
  return first_idx + 1;
}

void
POSetFactor::createInitialAdvice(CoachMessageQueue& mqueue)
{
  poset.createInitialAdvice(mqueue);
}

rcss::clang::Cond*
POSetFactor::createCondition(int myval)
{
  if (myval == -1)
    return NULL;
  state.setOverallIdx(myval);
  return state.createCondition();
}


/*********************************************************************/

const char* BallGridFactor::REG_PREFIX = "BG";
BallGridFactor::SentRegStorage BallGridFactor::s_sent_reg_storage = SentRegStorage();

BallGridFactor::BallGridFactor(int x_bins, int y_bins)
  : AbstractStateFactorLeaf(ASF_BallGrid),
    x_bins(x_bins),
    y_bins(y_bins),
    x_size((ServerParam::instance()->getSPPitchLength()+1.0)/x_bins),
    y_size((ServerParam::instance()->getSPPitchWidth()+1.0)/y_bins),
    num_states(-1)
{
  updateNumStates();
}

// Sets elements in the AbstractState for this factor
// returns the next factor idx to use
int
BallGridFactor::setStateFor(AbstractState* pstate, int first_idx,
			    const WorldState& ws, TeamSide my_side)
{
  VecPosition p = ws.getBall().getPos();
  if (my_side ==TS_Right)
    p = p.flipCoords();
  p += VecPosition(ServerParam::instance()->getSPPitchLength()/2,
		   ServerParam::instance()->getSPPitchWidth()/2);
  // switch off center a bit
  p += VecPosition(0.5, 0.5);
  int x = MinMax(0, (int)floor(p.getX() / x_size), x_bins - 1);
  int y = MinMax(0, (int)floor(p.getY() / y_size), y_bins - 1);

  pstate->setFactorIdx(first_idx, y*x_bins + x);
  pstate->setValid(true);
  return first_idx + 1;
}

int
BallGridFactor::draw(FieldImage* pimg, RegionWorldModelInterface* wmi,
		     const AbstractState* pstate, int first_idx)
{
  int myval = pstate->getFactorIdx(first_idx);
  actionlog(150) << *this << ": drawing with my val of " << myval
		 << " at idx " << first_idx << " of state: " << *pstate << ende;
  if (myval == -1)
    return first_idx + 1;
  Rectangle r = getGridRectangle(myval);
  FieldImage::Color cborder = FieldImage::COLOR_BLUE;
  //FieldImage::Color cfill = FieldImage::COLOR_BLUE;
  FieldImage::Color cfill = FieldImage::COLOR_CLEAR;
  cfill.setAlpha(0.85);
  pimg->addRectangle(r, cborder, cfill);
  return first_idx + 1;
}


void
BallGridFactor::drawEntireGrid(FieldImage* pimg, FieldImage::Color cborder, FieldImage::Color cinside)
{
  for (int i=0; i < num_states; i++)
    {
      Rectangle r = getGridRectangle(i);
      pimg->addRectangle(r, cborder, cinside);
      pimg->addText(toString(i).c_str(), r.getCenter());
    }
}

bool
BallGridFactor::extractBallPos(VecPosition* pv,
			       const AbstractState* pstate,
			       int *pfirst_idx)
{
  int myval = pstate->getFactorIdx(*pfirst_idx);
  (*pfirst_idx)++;
  if (myval == -1)
    return false;
  actionlog(150) << *this << ": extracting ball pos from my val " << myval
		 << " of state: " << *pstate << ende;
  Rectangle r = getGridRectangle(myval);
  *pv = r.getCenter();
  return true;
}

void
BallGridFactor::createInitialAdvice(CoachMessageQueue& mqueue)
{
  // First check if we have already sent regions for this
  SentRegElem sent_elem(x_bins, y_bins);

  if (s_sent_reg_storage.count(sent_elem))
    return;

  s_sent_reg_storage.insert(sent_elem);
  
  for (int gidx = 0; gidx < x_bins * y_bins; gidx++)
    {
      rcss::clang::RegRec* pReg = new rcss::clang::RegRec(getGridRectangle(gidx));
      mqueue.getDefineContainer().push(new rcss::clang::DefReg(getGridName(gidx),
							       std::auto_ptr<rcss::clang::Region>(pReg)));
    }
}

rcss::clang::Cond*
BallGridFactor::createCondition(int myval)
{
  if (myval == -1)
    return NULL;
  return new rcss::clang::CondBallPos(std::auto_ptr<rcss::clang::Region>(new rcss::clang::RegNamed(getGridName(myval))));
}


Rectangle
BallGridFactor::getGridRectangle(int grid_idx) const
{
  int x, y;
  if (!decodeGridPos(grid_idx, &x, &y))
    {
      errorlog << "BallGridFactor::getGridRectangle: couldn't decode pos for grid_idx " << grid_idx << ende;
      return Rectangle(VecPosition(0,0), VecPosition(0,0));
    }

  VecPosition upper_left = VecPosition(x_size * x, y_size * y) -
    VecPosition(ServerParam::instance()->getSPPitchLength() / 2,
		ServerParam::instance()->getSPPitchWidth() / 2) -
    VecPosition(0.5, 0.5);
  VecPosition lower_right = upper_left + VecPosition(x_size, y_size);
  return Rectangle(upper_left, lower_right);
}

bool
BallGridFactor::decodeGridPos(int grid_idx, int* px, int* py) const
{
  if (grid_idx < 0 || grid_idx >= x_bins * y_bins)
    return false;
  *px = grid_idx % x_bins;
  *py = grid_idx / x_bins;
  return true;
}

bool
BallGridFactor::isLocBetween(int gidx, int gidx_pt1, int gidx_pt2)
{
  // We shrink the rectangle in so that merely adjacent does not mean on the way
  Rectangle r = getGridRectangle(gidx).shrink(.1);
  Rectangle r1 = getGridRectangle(gidx_pt1).shrink(.1);
  Rectangle r2 = getGridRectangle(gidx_pt2).shrink(.1);

  return r.isInBetween(r1, r2);
}

std::string
BallGridFactor::getGridName(int grid_idx)
{
  if (grid_idx < 0 || grid_idx >= x_bins * y_bins)
    {
      errorlog << "BallGridFactor::getGridName: Asked for a bad grid idx "
	       << grid_idx << ende;
      return "";
    }
  if (x_bins * y_bins >= 100)
    errorlog << "BallGridFactor: My naming scheme only works up to 100 bins!" << ende;
  //SMURF: This is a little bit of a hack -- I assume two chars always
  return std::string(REG_PREFIX) +
    toStringGivenMax(x_bins, 99) + toStringGivenMax(y_bins, 99)  +
    toStringGivenMax(grid_idx, 99);
}



/*********************************************************************/

BallOwnerFactor::BallOwnerFactor(ModFeatures* pmFeatures)
  : AbstractStateFactorLeaf(ASF_BallOwner), pmFeatures(pmFeatures)
{
}

// 0: Other, 1:mine, 2:theirs
// Sets elements in the AbstractState for this factor
// returns the next factor idx to use
int
BallOwnerFactor::setStateFor(AbstractState* pstate, int first_idx,
			     const WorldState& ws, TeamSide my_side)
{
  if (!pmFeatures)
    {
      errorlog << "BallOwnerFactor: need real pmFeatures" << ende;
      return -1;
    }
  if (my_side != TS_Left && my_side != TS_Right)
    errorlog << "BallOwnerFactor: need real my_side, not: " << my_side << ende;
  State state_val = ST_Invalid;
  if (pmFeatures->getLastStateUpdateTime() != ws.getTime())
    errorlog << "BallOwnerFactor: I don't think ModFeatures is in sync with me! "
	     << pmFeatures->getLastStateUpdateTime() << ' ' << ws.getTime() << ende;
  switch(pmFeatures->getBallOwner().side)
    {
    case TS_Left:
      state_val = (my_side == TS_Left) ? ST_Mine : ST_Theirs;
      break;
    case TS_Right:
      state_val = (my_side == TS_Right) ? ST_Mine : ST_Theirs;
      break;
    case TS_None:
    case TS_Both:
      state_val = ST_Other;
      break;
    default:
      errorlog << "BallOwnerFactor: what is the ball owner side? "
	       << pmFeatures->getBallOwner().side << ende;
      break;
    }
  
  pstate->setFactorIdx(first_idx, (int)state_val);
  pstate->setValid(state_val != ST_Invalid);
  return first_idx + 1;
}

int
BallOwnerFactor::draw(FieldImage* pimg, RegionWorldModelInterface* wmi,
		      const AbstractState* pstate, int first_idx)
{
  State myval = (State)pstate->getFactorIdx(first_idx);
  actionlog(150) << *this << ": drawing with my val of " << myval
		 << " at idx " << first_idx << " of state: " << *pstate << ende;
  if (myval == ST_Invalid)
    return first_idx + 1;
  FieldImage::Color color = FieldImage::COLOR_BLUE;
  const float ARROW_LEN = 10;
  const float ARROW_SCALE = 2;
  if (myval == ST_Mine || myval == ST_Other)
    {
      VecPosition from(-ARROW_LEN / 2, ServerParam::instance()->getSPPitchWidth() / 2);
      VecPosition to  (ARROW_LEN / 2, ServerParam::instance()->getSPPitchWidth() / 2);
      pimg->addArrow(from, to, color, ARROW_SCALE);
    }
  if (myval == ST_Theirs || myval == ST_Other)
    {
      VecPosition from(ARROW_LEN / 2, ServerParam::instance()->getSPPitchWidth() / 2);
      VecPosition to  (-ARROW_LEN / 2, ServerParam::instance()->getSPPitchWidth() / 2);
      pimg->addArrow(from, to, color, ARROW_SCALE);
    }
  return first_idx + 1;
}

rcss::clang::Cond*
BallOwnerFactor::createCondition(int my_val_idx)
{
  State myval = (State)my_val_idx;

  rcss::clang::UNumSet uset;
  uset.add(rcss::clang::UNum::uAll);
  
  switch (myval)
    {
    case ST_Invalid:
      return NULL;
    case ST_Other:
      // We can actually express this FightBall correctly.
      // I am going to just make it that we are the ball owner.
      // At least some of the team interpret fight balls this way
      return new rcss::clang::CondBallOwner(true, uset);
    case ST_Mine:
      return new rcss::clang::CondBallOwner(true, uset);
    case ST_Theirs:
      return new rcss::clang::CondBallOwner(false, uset);
    default:
      errorlog << "BallOwnerFactor::createCondition: What is myval? " << myval << ende;
      return NULL;
    }
}


/*********************************************************************/

// Sets elements in the AbstractState for this factor
// returns the next factor idx to use
// 0:mine, 1:theirs
int
GoalFactor::setStateFor(AbstractState* pstate, int first_idx,
			const WorldState& ws, TeamSide my_side)
{
  State state_val = ST_Invalid;
  switch (ws.getPlayMode())
    {
    case PM_AfterGoal_Left:
      state_val = (my_side == TS_Left) ? ST_Mine : ST_Theirs;
      break;
    case PM_AfterGoal_Right:
      state_val = (my_side == TS_Right) ? ST_Mine : ST_Theirs;
    default:
      state_val = ST_Invalid;
      break;
    }
  pstate->setFactorIdx(first_idx, (int)state_val);
  pstate->setValid(state_val != ST_Invalid);
  return first_idx + 1;
}

int
GoalFactor::draw(FieldImage* pimg, RegionWorldModelInterface* wmi,
		 const AbstractState* pstate, int first_idx)
{
  State myval = (State)pstate->getFactorIdx(first_idx);
  actionlog(150) << *this << ": drawing with my val of " << myval
		 << " at idx " << first_idx << " of state: " << *pstate << ende;
  if (myval == ST_Invalid)
    return first_idx + 1;
  FieldImage::Color color = FieldImage::COLOR_BLUE;
  switch (myval)
    {
    case ST_Invalid:
      break;
    case ST_Mine:
      pimg->addX(VecPosition(ServerParam::instance()->getSPGoalPos(false)),
		 10, color, 3);
      break;
    case ST_Theirs:
      pimg->addX(VecPosition(ServerParam::instance()->getSPGoalPos(true)),
		 10, color, 3);
      break;
    default:
      errorlog << "GoalFactor::draw: what is myval? " << myval << ende;
      break;
    }
  return first_idx + 1;
}

rcss::clang::Cond*
GoalFactor::createCondition(int my_val_idx)
{
  State myval = (State)my_val_idx;

  switch (myval)
    {
    case ST_Invalid:
      return NULL;
    case ST_Mine:
      return new rcss::clang::CondPlayMode(rcss::clang::AfterGoal_Our);
    case ST_Theirs:
      return new rcss::clang::CondPlayMode(rcss::clang::AfterGoal_Opp);
    default:
      errorlog << "GoalFactor::createCondition: What is myval? " << myval << ende;
      return NULL;
    }
}


/*********************************************************************/
const char* DeadBallFactor::MINE_COND_NAME = "AS_DB_Mine";
const char* DeadBallFactor::THEIRS_COND_NAME = "AS_DB_Theirs";

// Sets elements in the AbstractState for this factor
// returns the next factor idx to use
// 0:mine, 1:theirs
int
DeadBallFactor::setStateFor(AbstractState* pstate, int first_idx,
			    const WorldState& ws, TeamSide my_side)
{
  State state_val = ST_Invalid;
  switch (ws.getPlayMode())
    {
    case PM_BeforeKickOff:
    case PM_TimeOver:
    case PM_PlayOn:
    case PM_AfterGoal_Left:
    case PM_AfterGoal_Right:
    case PM_Drop_Ball:
      // We'll let these pass by because they will decay to free kick soon
    case PM_OffSide_Left:
    case PM_OffSide_Right:
    case PM_Back_Pass_Left:
    case PM_Back_Pass_Right:
    case PM_Free_Kick_Fault_Left:
    case PM_Free_Kick_Fault_Right:
    case PM_CatchFault_Left:
    case PM_CatchFault_Right:
      // we'll let penalty kicks be ignored here
    case PM_PK_Left:
    case PM_PK_Right:
    case PM_PenaltySetup_Left:
    case PM_PenaltySetup_Right:
    case PM_PenaltyReady_Left:
    case PM_PenaltyReady_Right:
    case PM_PenaltyTaken_Left:
    case PM_PenaltyTaken_Right:
    case PM_PenaltyMiss_Left:
    case PM_PenaltyMiss_Right:
    case PM_PenaltyScore_Left:
    case PM_PenaltyScore_Right:
      actionlog(150) << "DeadBallFactor: Letting play mode pass " << ws.getPlayMode() << ende;
      break;
      
    case PM_KickOff_Left:
    case PM_KickIn_Left:
    case PM_FreeKick_Left:
    case PM_CornerKick_Left:
    case PM_GoalKick_Left:
    case PM_IndFreeKick_Left:
      state_val = (my_side == TS_Left) ? ST_Mine : ST_Theirs;
      break;
      
    case PM_KickOff_Right:
    case PM_KickIn_Right:
    case PM_FreeKick_Right:
    case PM_CornerKick_Right:
    case PM_GoalKick_Right:
    case PM_IndFreeKick_Right:
      state_val = (my_side == TS_Right) ? ST_Mine : ST_Theirs;
      break;

    case PM_FirstHalfOver:
    case PM_Pause:
    case PM_Human:
    case PM_Foul_Charge_Left:
    case PM_Foul_Charge_Right:
    case PM_Foul_Push_Left:
    case PM_Foul_Push_Right:
    case PM_Foul_MultipleAttacker_Left:
    case PM_Foul_MultipleAttacker_Right:
    case PM_Foul_BallOut_Left:
    case PM_Foul_BallOut_Right:
    case PM_Null:
    case PM_MAX:
      errorlog << "DeadBallFactor: invalid mode " << ws.getPlayMode() << ende;
      break;
    }

  pstate->setFactorIdx(first_idx, (int)state_val);
  pstate->setValid(state_val != ST_Invalid);
  return first_idx + 1;
}

int
DeadBallFactor::draw(FieldImage* pimg, RegionWorldModelInterface* wmi,
		     const AbstractState* pstate, int first_idx)
{
  State myval = (State)pstate->getFactorIdx(first_idx);
  actionlog(150) << *this << ": drawing with my val of " << myval
		 << " at idx " << first_idx << " of state: " << *pstate << ende;
  if (myval == ST_Invalid)
    return first_idx + 1;
  FieldImage::Color color = FieldImage::COLOR_GREEN;
  switch (myval)
    {
    case ST_Invalid:
      break;
    case ST_Mine:
      pimg->addArrow(VecPosition(wmi->getBallPos()),
		     VecPosition(wmi->getBallPos()) + VecPosition(5, 0),
		     color);
      break;
    case ST_Theirs:
      pimg->addArrow(VecPosition(wmi->getBallPos()),
		     VecPosition(wmi->getBallPos()) - VecPosition(5, 0),
		     color);
      break;
    default:
      errorlog << "DeadBallFactor::draw: what is myval? " << myval << ende;
      break;
    }
  return first_idx + 1;
}

void
DeadBallFactor::createInitialAdvice(CoachMessageQueue& mqueue)
{
  rcss::clang::CondOr* pOr;

  pOr = new rcss::clang::CondOr;
  pOr->push_back(std::auto_ptr<rcss::clang::Cond>(new rcss::clang::CondPlayMode(rcss::clang::KickOff_Our)));
  pOr->push_back(std::auto_ptr<rcss::clang::Cond>(new rcss::clang::CondPlayMode(rcss::clang::KickIn_Our)));
  pOr->push_back(std::auto_ptr<rcss::clang::Cond>(new rcss::clang::CondPlayMode(rcss::clang::FreeKick_Our)));
  pOr->push_back(std::auto_ptr<rcss::clang::Cond>(new rcss::clang::CondPlayMode(rcss::clang::CornerKick_Our)));
  pOr->push_back(std::auto_ptr<rcss::clang::Cond>(new rcss::clang::CondPlayMode(rcss::clang::GoalKick_Our)));
  pOr->push_back(std::auto_ptr<rcss::clang::Cond>(new rcss::clang::CondPlayMode(rcss::clang::GoalieCatch_Our)));
  pOr->push_back(std::auto_ptr<rcss::clang::Cond>(new rcss::clang::CondPlayMode(rcss::clang::AfterGoal_Our)));
  mqueue.getDefineContainer().push(new rcss::clang::DefCond(MINE_COND_NAME,
							   std::auto_ptr<rcss::clang::Cond>(pOr)));
  
  pOr = new rcss::clang::CondOr;
  pOr->push_back(std::auto_ptr<rcss::clang::Cond>(new rcss::clang::CondPlayMode(rcss::clang::KickOff_Opp)));
  pOr->push_back(std::auto_ptr<rcss::clang::Cond>(new rcss::clang::CondPlayMode(rcss::clang::KickIn_Opp)));
  pOr->push_back(std::auto_ptr<rcss::clang::Cond>(new rcss::clang::CondPlayMode(rcss::clang::FreeKick_Opp)));
  pOr->push_back(std::auto_ptr<rcss::clang::Cond>(new rcss::clang::CondPlayMode(rcss::clang::CornerKick_Opp)));
  pOr->push_back(std::auto_ptr<rcss::clang::Cond>(new rcss::clang::CondPlayMode(rcss::clang::GoalKick_Opp)));
  pOr->push_back(std::auto_ptr<rcss::clang::Cond>(new rcss::clang::CondPlayMode(rcss::clang::GoalieCatch_Opp)));
  pOr->push_back(std::auto_ptr<rcss::clang::Cond>(new rcss::clang::CondPlayMode(rcss::clang::AfterGoal_Opp)));
  mqueue.getDefineContainer().push(new rcss::clang::DefCond(THEIRS_COND_NAME,
							    std::auto_ptr<rcss::clang::Cond>(pOr)));
}

rcss::clang::Cond*
DeadBallFactor::createCondition(int my_val_idx)
{
  State myval = (State)my_val_idx;

  switch (myval)
    {
    case ST_Invalid:
      return NULL;
    case ST_Mine:
      return new rcss::clang::CondNamed(MINE_COND_NAME);
    case ST_Theirs:
      return new rcss::clang::CondNamed(THEIRS_COND_NAME);
    default:
      errorlog << "DeadBallFactor::createCondition: What is myval? " << myval << ende;
      return NULL;
    }
}


/*********************************************************************/

ConstantFactor::ConstantFactor(int num_states, int val)
  : AbstractStateFactorLeaf(ASF_Constant),
    num_states(num_states),
    val(val)
{
}
  
// Sets elements in the AbstractState for this factor
// returns the next factor idx to use
int
ConstantFactor::setStateFor(AbstractState* pstate, int first_idx,
			    const WorldState& ws, TeamSide my_side)
{
  pstate->setFactorIdx(first_idx, val);
  pstate->setValid(true);
  return first_idx + 1;
}

int
ConstantFactor::draw(FieldImage* pimg, RegionWorldModelInterface* wmi,
		     const AbstractState* pstate, int first_idx)
{
  int myval = pstate->getFactorIdx(first_idx);
  actionlog(150) << *this << " drawing with my val of " << myval
		 << " at idx " << first_idx << " of state: " << *pstate << ende;
  if (myval == -1)
    return first_idx + 1;
  std::ostringstream label;
  label << *this << ": " << myval;
  pimg->addLegendLine(label.str().c_str());
  return first_idx + 1;
}

rcss::clang::Cond*
ConstantFactor::createCondition(int myval)
{
  errorlog << "ConstantFactor: I really don't think you should be asking me to create a condition" << ende;
  return NULL;
}

